/*
 * File: CosObjectStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.lang.ref.SoftReference;
import java.util.Iterator;
import java.util.Map;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.IO;
import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * @author sweet
 * @author gish
 * @author peters
 *
 * Object streams are streams that contain other objects. They were
 * added to the language in v1.5, and are described 3.4.6 of the reference
 * manual.
 */
public class CosObjectStream extends CosStream
{
	private CosList mNewObjList;			// List of objects added to stream
	private int[] mNewObjNumbers;			// The object number of the nth new object in this stream data
	private int[] mNewObjOffsets;			// The offset of the nth new object in this stream data
	private int[] mOldObjNumbers;			// The object number of the nth old object in this stream data
	private int[] mOldObjOffsets;			// The offset of the nth old object in this stream data
	private Object mOldDataStream;			// Cached version of old decoded object stream data

	/**
	 * Constructor for CosObjectStream that creates an empty CosObjectStream
	 * @throws IOException
	 */
	CosObjectStream(CosDocument doc)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		super(doc, doc.newObjectInfo()); //Initialize the CosStream values
		put(ASName.k_Type, ASName.k_ObjStm);
		getInfo().setIsObjStm(true);
	}

	/**
	 * Constructor for CosObjectStream that is used while tokenizing
	 * @throws PDFIOException
	 */
	CosObjectStream(CosDocument doc, Map map, CosObjectInfo info, long pos)
		throws PDFCosParseException, IOException,  PDFSecurityException, PDFIOException
	{
		super(doc, map, info, pos); // initialize all the CosStream stuff
		int numObjects = (get(ASName.k_N)).intValue();
		int objBase = (get(ASName.k_First)).intValue();

		// Now we parse the directory information at the start of the data portion of the stream.
		InputByteStream stmData = getDataStream();
		stmData.seek(0); // read from the beginning of the data
		CosParseBuf pBuf = new CosParseBuf(stmData, objBase);
		mOldObjNumbers = new int[numObjects];
		mOldObjOffsets = new int[numObjects];
		for (int objNdx = 0; objNdx < numObjects; objNdx++) {
			mOldObjNumbers[objNdx] = (CosToken.readNumber(pBuf, CosToken.skipWhitespace(pBuf))).intValue();
			int offset = (CosToken.readNumber(pBuf, CosToken.skipWhitespace(pBuf))).intValue();
			mOldObjOffsets[objNdx] = objBase + offset;
		}
		info.setIsObjStm(true);
	}

	/**
	 * Used for tokenizing objects that reside in an existing object stream
	 * @throws PDFIOException
	 */
	InputByteStream getBytesForObject(int objNum)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		for (int i = 0; i < mOldObjNumbers.length; i++) {
			if (mOldObjNumbers[i] == objNum) {
				InputByteStream stm = getDataStream();
				stm.seek(mOldObjOffsets[i]);
				int objLen = 0;
				if (i == mOldObjNumbers.length - 1)
					objLen = (int)stm.length();
				else
					objLen = mOldObjOffsets[i + 1];
				objLen -= mOldObjOffsets[i];
				
				return new CosParseBuf(stm, objLen + 1, true);
			}
		}
		return null;
	}

	/**
	 * Get the underlying data stream and maintain the SoftReference
	 * @throws PDFIOException
	 */
	InputByteStream getDataStream()
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		if (mOldDataStream instanceof InputByteStream)
			return (InputByteStream)mOldDataStream;
		InputByteStream dataStream = null;
		if (mOldDataStream instanceof SoftReference)
			dataStream = (InputByteStream)((SoftReference)mOldDataStream).get();
		if (dataStream == null) {
			dataStream = getStream(false, false, false);
			mOldDataStream = new SoftReference(dataStream);
		}
		return dataStream;
	}

	/**
	 * Add a new object to the object stream's output list
	 */
	void addObjectToStream(CosObject obj)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		getInfo().markDirty();
		if (mNewObjList == null)
			mNewObjList = new CosList();
		mNewObjList.add(obj.getInfo());
	}

	/**
	 * Set a completed list of objects to the object stream's output list
	 */
	void setObjectStreamList(CosList objectList)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		getInfo().markDirty();
		mNewObjList = objectList;
	}

	/**
	 * Update the object stream internal data after save
	 */
	void update()
	{
		if (mNewObjList != null) {
			mOldObjNumbers = mNewObjNumbers;
			mOldObjOffsets = mNewObjOffsets;
			reset();
		}
	}

	/**
	 * Reset the object stream internal data after save
	 */
	void reset()
	{
		mNewObjList = null;
		mNewObjNumbers = null;
		mNewObjOffsets = null;
		mOldDataStream = null;
	}

	/**
	 * Generate the output data stream from the object stream's output list
	 * @throws IOException
	 */
	void writeObjectsToStream()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// Don't do anything if you have no objects to write
		if (mNewObjList == null || mNewObjList.isEmpty())
			return;

		// Reinitialize
		int numObjects = 0;
		int[] objNumbers = new int[mNewObjList.count()];
		int[] objOffsets = new int[mNewObjList.count()];

		// Create a new OutputByteStream
		OutputByteStream objos = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);

		// Write each object on the list to the OutputByteStream and store its offset in the info
		Iterator iterator = mNewObjList.iterator();
	
		while (iterator.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iterator.next();
			CosObject obj = info.getObject();
			info.setPos(objos.getPosition());
			info.setObjGen(0);
			objNumbers[numObjects] = info.getObjNum();
			objOffsets[numObjects] = (int)info.getPos();
			info.setWriteCompressed(true);
			obj.writeOut(objos);
			/**
			 *  Fix for Bug#3978861:  
			 *  Issue: While writing Cos Objects to the stream , no space
			 *  was added. So the consecutive booleans , or Cosnull were added without any spaces in between 
			 *  like truefalsenull
			 *  Due to this, acrobat was able to open the file as it uses spaces/ delim to parse the object stream.
			 *  So adding space as a delim after every Cos Object is written to facilitate the parsing in acrobat.   
			 */
			objos.write(' ');
			info.setStreamInfo(getInfo());
			info.setStreamNdx(numObjects);
			numObjects++;
		}

		// Acquire this object's data stream and write the ObjNum/Offset list
		OutputByteStream os = getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);
		Iterator iter = mNewObjList.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			os.write(StringOps.toByteArray(Integer.toString(info.getObjNum())));
			os.write(' ');
			os.write(StringOps.toByteArray(Long.toString(info.getPos())));
			os.write(' ');
		}

		// Get the position of the first object and store it as /First
		long objBase = os.getPosition();
		put(ASName.k_First, (int)objBase);
		for (int i = 0; i < numObjects; i++)
			objOffsets[i] += objBase;

		// Write the number of objects in the stream
		put(ASName.k_N, numObjects);

		// Append the objects stream to the data stream
		InputByteStream objis = objos.closeAndConvert();
		objos = null;
		IO.copy(objis, os);
		objis.close();
		objis = null;
		mOldDataStream = getDataStream();
		newDataDecoded(os.closeAndConvert());
		os = null;
		mNewObjNumbers = objNumbers;
		mNewObjOffsets = objOffsets;

		// Flate encode the data stream (done when this stream object is written)
		put(ASName.k_Filter, ASName.k_FlateDecode);
	}
}
