/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons.servlets;

import com.day.cq.commons.jcr.JcrConstants;
import org.apache.felix.scr.annotations.sling.SlingServlet;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.servlets.OptingServlet;
import org.apache.sling.api.servlets.SlingSafeMethodsServlet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import java.io.IOException;

/**
 * Simple proxy servlet that include the request to a proxied resource.
 * Dispatches ".html" requests to a "nt:file" that have a "sling:resourceType"
 * property set on their "jcr:content" node, to the "jcr:content" resource.
 */
@SlingServlet(resourceTypes = "nt:file", extensions = "html")
public class ProxyServlet extends SlingSafeMethodsServlet implements OptingServlet {

    private static final long serialVersionUID = -2132087866168663893L;
    /**
     * default log
     */
    private final Logger log = LoggerFactory.getLogger(ProxyServlet.class);

    @Override
    protected void service(SlingHttpServletRequest request,
                           SlingHttpServletResponse response)
            throws ServletException, IOException {
        Resource resource = request.getResource();
        RequestDispatcher crd = request.getRequestDispatcher(resource.getPath() + "/" + JcrConstants.JCR_CONTENT);
        if (crd != null) {
            crd.include(request, response);
        } else {
            log.error("Unable to dispatch proxy request.for {} referrer={}", request.getRequestURI(), request.getHeader("Referrer"));
            throw new ServletException("No Content");
        }
    }

    /**
     * {@inheritDoc}
     * <p>
     * Does not proxy for request to a nt:file that have no sling:resource type
     * set in the jcr:content node.
     */
    public boolean accepts(SlingHttpServletRequest request) {
        try {
            Resource resource = request.getResource();
            Node node = resource.adaptTo(Node.class);
            if (node == null) {
                if (log.isDebugEnabled()) {
                    log.debug("Unable to adapt resource to node: {}", resource.getPath());
                }
                return false;
            }
            if (!node.hasNode(JcrConstants.JCR_CONTENT)) {
                if (log.isDebugEnabled()) {
                    log.debug("resource has no jcr:content: {}", resource.getPath());
                }
                return false;
            }
            Node content = node.getNode(JcrConstants.JCR_CONTENT);
            if (!content.hasProperty("sling:resourceType")) {
                if (log.isDebugEnabled()) {
                    log.debug("resource content has no sling:resourceType property: {}", resource.getPath());
                }
                return false;
            }
            if (log.isDebugEnabled()) {
                log.debug("accepting request for {}", resource.getPath());
            }
            return true;
        } catch (RepositoryException e) {
            if (log.isDebugEnabled()) {
                log.warn("Error while evaluating accept.", e);
            } else {
                log.warn("Error while evaluating accept: {}", e.toString());
            }
            return false;
        }
    }
}