/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.video;

import com.day.cq.commons.jcr.JcrConstants;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import com.day.cq.dam.commons.util.DamUtil;
import com.day.cq.dam.commons.util.PrefixRenditionPicker;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.caconfig.resource.ConfigurationResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import java.awt.Dimension;

/**
 * Describes the transcoding of a video file. Utility wrapper around a video
 * profile page/resource.
 */
public class VideoProfile {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(VideoProfile.class);

    public static final String PROFILE_BASE_PATH = "dam/video";

    public static final String PROFILE_TYPE_CQ = "dam/components/video/profile";

    public static final String PROFILE_TYPE_S7 = "dam/components/video/s7profile";

    public static final String PROFILES_CACONFIG_BUCKET = "settings";
    public static final String PROFILES_LEGACY_BUCKET = "/etc";

    /**
     * @deprecated since release 6.4. With Video Transcoding profiles being resolved
     * by Sling ContentAware Configuration Resolver,
     * use {@link VideoProfile#get(ResourceResolver, ConfigurationResourceResolver, String)} instead
     * 
     * Resolves a video profile by name (relative to the video profile base
     * path) or absolute path.
     *
     * @param resolver to resolve the provided profile-name/profile-path
     * @param name can be a name or an absolute path
     * @return a video profile wrapper
     */
   @Deprecated
   public static VideoProfile get(ResourceResolver resolver, String name) {
        String path;
        Resource resource = null;
        if (name.startsWith("/")) {
            path = name;
            log.warn("Resolution of Video Transcoding Profiles with absolute paths is deprecated."
                    + " Use VideoProfile#get(ResourceResolver, ConfigurationResourceResolver, String) instead"
                    + " to resolve profiles-by-name using Sling Context-Aware Config Resolution");
            resource = resolver.getResource(path);
        } else {
            // a legacy profile... locate in '/etc'
            path = PROFILE_BASE_PATH + "/" + name;
            log.warn("Video Transcoding Profiles in '/etc/dam/video' are deprecated."
                    + " Create profiles under \"dam/video\" in \"settings\" bucket of Sling Context-Aware Configuration"
                    + " and use VideoProfile#get(ResourceResolver, ConfigurationResourceResolver, String) instead"
                    + " to resolve profiles-by-name using Sling Context-Aware Configuration Resolution");
            resource = resolver.getResource(PROFILES_LEGACY_BUCKET + "/" + path);
        }
        if (resource != null) {
            return new VideoProfile(resource);
        }
        return null;
    }


    /**
     * Resolves a video profile by name from Sling Context-Aware Configuration
     *
     * @param resolver to resolve the provided profile-name
     * @param configResolver to resolve provided profile-name using Sling Context-Aware Config Resolver
     * @param name of the profile (under the Context-Aware config's "settings" bucket)
     * @return a video profile wrapper
     */
    public static VideoProfile get(ResourceResolver resolver, ConfigurationResourceResolver configResolver, String name) {
        String path;
        path = PROFILE_BASE_PATH + "/" + name;

        Resource tenantAssetRootResource = resolver.getResource(DamUtil.getTenantAssetsRoot(resolver));
        Resource resource = configResolver.getResource(tenantAssetRootResource, PROFILES_CACONFIG_BUCKET, path);

        VideoProfile profile = null;
        if (resource != null) {
            profile = new VideoProfile(resource);
        } else {
            // for backwards compatibility
            profile = VideoProfile.get(resolver, name);
        }
        return profile;
    }

    private Resource resource;

    public VideoProfile(Resource resource) {
        if (JcrConstants.JCR_CONTENT.equals(resource.getName())) {
            this.resource = resource.getParent();
        } else {
            this.resource = resource;
        }
    }

    public String getName() {
        return resource.getName();
    }

    public String getPath() {
        return resource.getPath();
    }

    public Resource getResource() {
        return resource;
    }

    public Resource getContentResource() {
        return resource.getResourceResolver().getResource(resource, JcrConstants.JCR_CONTENT);
    }

    public ValueMap getProperties() {
        return ResourceUtil.getValueMap(getContentResource());
    }

    public Node getContentNode() {
        Resource content = getContentResource();
        return content == null ? null : content.adaptTo(Node.class);
    }

    public String getHtmlType() {
        return getProperties().get(VideoConstants.PN_HTML_VIDEO_TYPE, "");
    }

    public Rendition getRendition(Asset asset) {
        if (isCQProfile()) {
            ValueMap props = getProperties();
            String renditionSelector = (String)props.get(VideoConstants.PN_RENDITION_SELECTOR);
            if (StringUtils.isEmpty(renditionSelector)) {
                // fallback to profile name, for legacy reasons
                renditionSelector = getName();
            }
            String prefix = VideoConstants.RENDITION_PREFIX + renditionSelector;

            return asset.getRendition(new PrefixRenditionPicker(prefix));
        } else if (isS7Profile()) {
            String prefix = "cq5dam.video.s7." + getPresetHandle();
            return asset.getRendition(new PrefixRenditionPicker(prefix));
        } else {
            return null;
        }
    }

    public Dimension getOutputSize() {
        ValueMap props = getProperties();
        Integer width = props.get(VideoConstants.PN_VIDEO_WIDTH, Integer.class);
        Integer height = props.get(VideoConstants.PN_VIDEO_HEIGHT, Integer.class);
        if (width != null && height != null) {
            return new Dimension(width, height);
        }
        return null;
    }

    /**
     * Get the value for the 'src' attribute for the html &lt;source&gt; element, based on the given rendition.
     * @param rendition Rendition for which to get the src attribute
     * @return String with correct src attribute
     */
    public String getHtmlSource(Rendition rendition) {
        String htmlSource = "";
        if (rendition != null && isCQProfile()) {
            htmlSource = rendition.getPath();
        } else if (rendition != null && isS7Profile()) {
            htmlSource = getScene7Url(rendition);
        }
        return htmlSource;

    }

    public String getStrobeVideoSource(Rendition rendition) {
        String source = "";
        if (rendition != null && isCQProfile()) {
            source = "../../../../.." + rendition.getPath(); // strobe speciality: path must be relative to swf file
        } else if (rendition != null && isS7Profile()) {
            source = getScene7Url(rendition);
        }
        return source;
    }

    public String getFlvVideoSource(Rendition rendition) {
        String source = "";
        if (rendition != null && isCQProfile()) {
            source = rendition.getPath();
        } else if (rendition != null && isS7Profile()) {
            source = getScene7Url(rendition);
        }
        return source;
    }

    public String getCustomVideoSource(Rendition rendition) {
        return getFlvVideoSource(rendition);
    }

    private boolean isS7Profile() {
        return getResourceType().equals(PROFILE_TYPE_S7);
    }

    private boolean isCQProfile() {
        return getResourceType().equals(PROFILE_TYPE_CQ);
    }

    private String getResourceType() {
        String resourcType = "";
        Resource content = resource.getChild(JcrConstants.JCR_CONTENT);
        if (content != null) {
            resourcType = content.getResourceType();
        }
        return resourcType;
    }

    private String getPresetHandle() {
        String presetHandle = getProperties().get(VideoConstants.S7PN_PRESET_PATH, "");
        if (StringUtils.isNotBlank(presetHandle)) {
            int lastIndex = presetHandle.lastIndexOf("/");
            presetHandle = presetHandle.substring(lastIndex + 1, presetHandle.length());
            presetHandle = presetHandle.replaceFirst("ps-", "");
        }
        return presetHandle;
    }

    private String getScene7Url(Rendition rendition) {
        String scene7Url = "";
        if (rendition != null) {
            scene7Url = rendition.getProperties().get("scene7.url", "");
        }
        return scene7Url;
    }
}
