/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.commons;

/**
 * The <code>GlobPattern</code> implements matching operations that do a
 * pattern globbing.
 * <br>
 * This class is copied from <code>day-commons-text</code> and should not
 * be used anymore. switch to regexp patterns if possible.
 */
public class GlobPattern {

    /**
     * Pattern strings should be compared to.
     */
    private final String pattern;

    /**
     * if this is present, use for handle comparison (acl)
     */
    private final String handlePattern;

    /**
     * Class constructor that create a <code>GlobPattern</code> with the given
     * pattern. If the <code>isHandle</code> flag is <code>true</code> this
     * will construct a hierarchy matcher.
     *
     * @param pattern  pattern string
     * @param isHandle if <code>true</code> and the pattern contains no
     *                 wildcards, the {@link #matches(String)} returns true, if
     *                 the compared string is equal or starts with the
     *                 pattern+"/" (i.e. is a child page)
     */
    public GlobPattern(String pattern, boolean isHandle) {
        this.pattern = pattern;
        if (isHandle && !containsWildcards(pattern)) {
            handlePattern = (pattern.equals("/")) ? "/": pattern + '/';
        } else {
            handlePattern = null;
        }
    }

    /**
     * Class constructor that create a <code>GlobPattern</code> with the given
     * pattern.
     *
     * @param pattern pattern string
     */
    public GlobPattern(String pattern) {
        this(pattern, false);
    }

    /**
     * Returns a flag indicating whether a string matches this pattern.
     *
     * @param s string to be checked
     * @return <code>true</code> if <em>s</em> matches this pattern, else
     *         <code>false</code>.
     */
    public final boolean matches(String s) {
        if (handlePattern != null) {
            // do handle comparison
            return pattern.equals(s) || s.startsWith(handlePattern);
        }
        return recurseMatchPattern(pattern, s, 0, 0);
    }

    /**
     * Returns a flag indicating whether a string matches a pattern.
     *
     * @param pattern pattern used for comparison
     * @param s       string to be checked
     * @return <code>true</code> if <em>s</em> matches <em>pattern</em>,
     *         else <code>false</code>.
     */
    public static boolean matches(String pattern, String s) {
        return recurseMatchPattern(pattern, s, 0, 0);
    }

    /**
     * Returns a flag indicating whether a string matches a pattern. if the
     * <em>ishHandle</em> is <code>true</code> and the <em>pattern</em> contains
     * no wildcards, the method returns true, if the pattern is a hierarchical
     * father of the string.
     *
     * @param pattern  pattern used for comparison
     * @param s        string to be checked
     * @param isHandle flag, indicating, if a handle comparison has to be
     *                 performed
     * @return <code>true</code> if <em>s</em> matches <em>pattern</em>,
     *         else <code>false</code>.
     */
    public static boolean matches(String pattern, String s, boolean isHandle) {
        if (isHandle && !containsWildcards(pattern)) {
            return pattern.equals(s) || s.startsWith(pattern + '/');
        }
        return GlobPattern.matches(pattern, s);
    }

    /**
     * Returns a flag indicating whether a string matches a pattern. unlike the
     * <code>matches</code> methods, this matching is done shell-like.
     *
     * @param s string to be checked
     * @return <code>true</code> if the string matches shell-like;
     *         <code>false</code> otherwise.
     */
    public boolean shellMatches(String s) {
        return shellMatches(s, '/');
    }

    /**
     * Returns a flag indicating whether a string matches a pattern. unlike the
     * <code>matches</code> methods, this matching is done shell-like.
     *
     * @param s string to be checked
     * @param c character to be used as path delimiter
     * @return <code>true</code> if the string matches shell-like;
     *         <code>false</code> otherwise.
     */
    public boolean shellMatches(String s, char c) {
        if (pattern.equals("*")) return true;
        int sc = 0;
        for (int len = s.length() - 1; len >= 0; len--) {
            if (s.charAt(len) == c) sc++;
        }
        for (int len = pattern.length() - 1; len >= 0; len--) {
            if (pattern.charAt(len) == c) sc--;
        }
        return sc == 0 && matches(s);
    }

    /**
     * Returns <code>true</code> if the string contains wildcards.
     *
     * @param s string to be checked
     * @return <code>true</code> if <em>s</em> contains wildcards, else
     *         <code>false</code>.
     */
    public static boolean containsWildcards(String s) {
        return indexOfWildcard(s) >= 0;
    }

    /**
     * Returns the index of the first wildcard character in the string or
     * <code>-1</code> if the string does not contain a wild card character.
     *
     * @param s string to be checked
     * @return the index of the first wildcard.
     */
    public static int indexOfWildcard(String s) {
        for (int i = 0; i < s.length(); i++) {
            if ("*?[]".indexOf(s.charAt(i)) != -1) {
                return i;
            }
        }

        // not found, return -1
        return -1;
    }

    /**
     * An internal routine to implement expression matching.
     * This routine is based on a self-recursive algorithm.
     *
     * @param pattern The pattern
     * @param s       The string to be compared.
     * @param sIdx    The index of where we are in <em>string</em>.
     * @param pIdx    The index of where we are in <em>pattern</em>.
     * @return True if <em>string</em> matched pattern, else false.
     */
    private static boolean recurseMatchPattern(String pattern, String s,
                                               int sIdx, int pIdx) {
        int pLen = pattern.length();
        int sLen = s.length();

        for (; ;) {
            if (pIdx >= pLen) {
                return (sIdx >= sLen);
            }
            if (sIdx >= sLen && pattern.charAt(pIdx) != '*') {
                return false;
            }

            // Check for a '*' as the next pattern char.
            // This is handled by a recursive call for
            // each postfix of the name.
            if (pattern.charAt(pIdx) == '*') {
                if (++pIdx >= pLen) {
                    return true;
                }

                for (; ;) {
                    if (recurseMatchPattern(pattern, s, sIdx, pIdx)) {
                        return true;
                    }
                    if (sIdx >= sLen) {
                        return false;
                    }
                    ++sIdx;
                }
            }

            // Check for '?' as the next pattern char.
            // This matches the current character.
            if (pattern.charAt(pIdx) == '?') {
                ++pIdx;
                ++sIdx;
                continue;
            }

            // Check for '[' as the next pattern char.
            // This is a list of acceptable characters,
            // which can include character ranges.
            if (pattern.charAt(pIdx) == '[') {
                for (++pIdx; ; ++pIdx) {
                    if (pIdx >= pLen || pattern.charAt(pIdx) == ']') {
                        return false;
                    }
                    if (pattern.charAt(pIdx) == s.charAt(sIdx)) {
                        break;
                    }

                    if (pIdx < (pLen - 1)
                            && pattern.charAt(pIdx + 1) == '-') {
                        if (pIdx >= (pLen - 2)) {
                            return false;
                        }

                        char chStr = s.charAt(sIdx);
                        char chPtn = pattern.charAt(pIdx);
                        char chPtn2 = pattern.charAt(pIdx + 2);

                        if ((chPtn <= chStr) && (chPtn2 >= chStr)) {
                            break;
                        }
                        if ((chPtn >= chStr) && (chPtn2 <= chStr)) {
                            break;
                        }
                        pIdx += 2;
                    }
                }

                for (; pattern.charAt(pIdx) != ']'; ++pIdx) {
                    if (pIdx >= pLen) {
                        --pIdx;
                        break;
                    }
                }

                ++pIdx;
                ++sIdx;
                continue;
            }

            // Check for backslash escapes
            // We just skip over them to match the next char.
            if (pattern.charAt(pIdx) == '\\') {
                if (++pIdx >= pLen) {
                    return false;
                }
            }

            if (pIdx < pLen && sIdx < sLen) {
                if (pattern.charAt(pIdx) != s.charAt(sIdx)) {
                    return false;
                }
            }
            ++pIdx;
            ++sIdx;
        }
    }

    /**
     * Returns the pattern of this <code>GlobPattern</code>
     *
     * @return the pattern.
     */
    public String toString() {
        return pattern;
    }

    /**
     * Returns <code>true</code> if <code>this</code> <code>GlobPattern</code>
     * is equal to object <code>obj</code>.
     *
     * @param obj the object to compare for equality.
     * @return <code>true</code> if <code>this</code> <code>GlobPattern</code>
     *         is equal to object <code>obj</code>.
     */
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof GlobPattern) {
            GlobPattern other = (GlobPattern) obj;
            return (this.pattern.equals(other.pattern)
                    && this.isHandlePattern() == other.isHandlePattern());
        }
        return false;
    }

    /**
     * Returns the hashCode for this <code>GlobPattern</code>.
     *
     * @return the hashCode for this <code>GlobPattern</code>.
     */
    public int hashCode() {
        return this.pattern.hashCode() + ((isHandlePattern()) ? 1: 0);
    }

    /**
     * Returns <code>true</code> if this GlobPattern is a handle pattern.
     *
     * @return <code>true</code> if this GlobPattern is a handle pattern;
     *         <code>false</code> otherwise.
     */
    private boolean isHandlePattern() {
        return handlePattern != null;
    }
}
