package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.LayoutTable.CoverageConsumer;

final class GsubHarvester  extends LookupTableHarvester 
{
	private AltSubstHarvester altSubstHarvester;
	private LigaSubstHarvester ligaSubstHarvester;
	private MultSubstHarvester multSubstHarvester;
	private SingleSubstFormat1Harvester singleSubstFormat1Harvester;
	private SingleSubstFormat2Harvester singleSubstFormat2Harvester;

	GsubHarvester(Gsub gsub, int numGlyphs) 
	{
		super(gsub, numGlyphs);
		this.altSubstHarvester = new AltSubstHarvester();
		this.ligaSubstHarvester = new LigaSubstHarvester();
		this.multSubstHarvester = new MultSubstHarvester();
		this.singleSubstFormat1Harvester = new SingleSubstFormat1Harvester();
		this.singleSubstFormat2Harvester = new SingleSubstFormat2Harvester();

	}

	/**
	 * Gather the lookups that are relevant to the supplied gids. The reachable glyphs are added to gids. (So
	 * this method harvests 2 things.)
	 * @param gids The gids to harvest
	 * @return A map of lookup indices and subtable indices that can (possibly recursively) apply to the supplied gids.
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException 
	 */
	TreeMap<Integer /*lookup index*/, List /*subtables*/> gatherPossibleLookups(Subset gids) 
	throws InvalidFontException, UnsupportedFontException
	{
		TreeMap applicableLookups = new TreeMap();
		IntGrowableArray featuresInFeatureList = getAllFeatureListFeatures();
		int initialGIDListSize;

		do
		{
			initialGIDListSize = gids.getNumGlyphs();
			harvestAllFeatures(featuresInFeatureList, applicableLookups, gids);		   
		} while (gids.getNumGlyphs() != initialGIDListSize);
		return applicableLookups;
	}

	private abstract class GSubLookupHarvester implements CoverageLookupHarvester {
		public boolean keepGoing() {
			// GSub needs to keep going so that all of the glyphs can be harvested.
			return true;
		}
	}

	private class SingleSubstFormat1Harvester extends GSubLookupHarvester 
	{

		SingleSubstFormat1Harvester() {}

		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int delta = lookupTable.data.getint16(stOffset + 4);
			try {
				gids.getSubsetGid(coverageGlyph + delta);
			} catch (InvalidFontException e) {
				return false;
			}
			return true;
		}
	}

	private class SingleSubstFormat2Harvester extends GSubLookupHarvester 
	{
		SingleSubstFormat2Harvester() {}

		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int substitute = lookupTable.data.getuint16(stOffset + 6 + 2*coverageIndex);
			try {
				gids.getSubsetGid(substitute);
			} catch (InvalidFontException e) {
				return false;
			}
			return true;
		}
	}

	private class MultSubstHarvester extends GSubLookupHarvester 
	{
		MultSubstHarvester() { }

		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int sequenceOffset = lookupTable.data.getOffset (stOffset, 6 + 2*coverageIndex);
			int glyphCount = lookupTable.data.getuint16 (sequenceOffset);
			for (int g = 0; g < glyphCount; g++) {
				try {
					gids.getSubsetGid(lookupTable.data.getuint16(sequenceOffset + 2 +  2*g));
				} catch (InvalidFontException e) {
					return false;
				}
			}
			return true;
		}

	}

	private class AltSubstHarvester  extends GSubLookupHarvester  
	{
		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int alternateOffset = lookupTable.data.getOffset(stOffset, 6 + coverageIndex*2);
			int glyphCount = lookupTable.data.getuint16(alternateOffset);
			for (int i = 0; i < glyphCount; i++)
			{
				int glyphID = lookupTable.data.getuint16(alternateOffset + 2 + 2*i);
				try {
					gids.getSubsetGid(glyphID);
				} catch (InvalidFontException e) {
					return false;
				}
			}
			return true;
		}
	}

	private class LigaSubstHarvester  extends GSubLookupHarvester  
	{
		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int ligaSetOffset = lookupTable.data.getOffset(stOffset, 6 + coverageIndex*2);
			int ligaCount = lookupTable.data.getuint16(ligaSetOffset);
			boolean foundRelevantLiga = false;
			for (int i = 0; i < ligaCount; i++) {
				int ligaOffset = lookupTable.data.getOffset(ligaSetOffset, 2 + i*2);
				int componentCount = lookupTable.data.getuint16(ligaOffset+2);
				int j;
				for (j = 0; j < componentCount-1; j++)
				{
					int gid = lookupTable.data.getuint16(ligaOffset + 4 + 2*j);
					if (gids.getExistingSubsetGid(gid) == -1)
						break;
				}

				if (j == componentCount-1) {
					foundRelevantLiga = true;
					int gid = lookupTable.data.getuint16(ligaOffset);
					try {
						gids.getSubsetGid(gid);
					} catch (InvalidFontException e) {
						foundRelevantLiga = false;
					}
				}
			}

			return foundRelevantLiga;
		}
	}

	private class ChainingFormat3SubstHarvester  extends GSubLookupHarvester  implements CoverageConsumer 
	{
		private Map lookups;
		private boolean coverageContainsGid;

		ChainingFormat3SubstHarvester(Map lookups)
		{
			this.lookups = lookups;
		}

		private boolean checkCoverages(int stOffset, int coverageStartOffset, int numCoverages, Subset gids) 
		throws InvalidFontException, UnsupportedFontException
		{
			for (int i = 0; i < numCoverages; i++)
			{
				coverageContainsGid = false;

				lookupTable.iterateCoverage(lookupTable.data.getOffset(stOffset, coverageStartOffset + 2*i), gids, this);

				if (!coverageContainsGid)
					return false;
			}
			return true;
		}

		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException 
		{
			int backtrackCount = lookupTable.data.getuint16(stOffset + 2);
			int delta = 4;
			boolean ok = checkCoverages(stOffset, delta, backtrackCount, gids);
			if (!ok)
				return false;

			// the first input coverage was consumed gettting here.
			delta += 2*backtrackCount;
			int inputCount = lookupTable.data.getuint16(stOffset + delta);
			delta += 2;
			ok = checkCoverages(stOffset, delta + 2, inputCount-1, gids);
			if (!ok)
				return false;

			delta += 2*inputCount;
			int lookaheadCount = lookupTable.data.getuint16(stOffset + delta);
			delta += 2;
			ok = checkCoverages(stOffset, delta, lookaheadCount, gids);
			if (!ok)
				return false;

			delta += 2*lookaheadCount;
			int lookupCount = lookupTable.data.getuint16(stOffset+delta);

			delta += 2;

			for (int j = 0; j < lookupCount; j++)
			{
				int lookupIndex = lookupTable.data.getuint16(stOffset + delta + 4*j + 2);
				// Note that this is a little bit loose. If we wanted the tightest set, we'd
				// only harvest on the glyphs that this lookup applies to. However, I'll work on the
				// assumption that if this sublookup applies to other glyphs in 'gids', then there will be
				// a path that gets us there in most fonts, so we'll want it eventually anyway. (That's a
				// good sounding justification for matching cooltype's behavior anyway...)
				harvest(gids, lookupIndex, lookups);
			}

			return true;
		}

		public boolean glyphInfo(int gid, int coverageIndex) {
			coverageContainsGid = true;
			return false;
		}

	}

	private class ContextFormat3SubstHarvester  extends GSubLookupHarvester implements CoverageConsumer
	{
		private Map lookups;
		private boolean coverageContainsGid;

		ContextFormat3SubstHarvester(Map lookups)
		{
			this.lookups = lookups;
		}

		public boolean lookupApplies(int coverageGlyph, int stOffset, int coverageIndex, Subset gids) 
		throws InvalidFontException, UnsupportedFontException {

			int glyphCount = lookupTable.data.getuint16(stOffset+2);
			if (glyphCount == 1)
				coverageContainsGid = true;
			int i;

			for (i = 1; i < glyphCount; i++) // we already consumed the first coverage to get in this method...
			{
				coverageContainsGid = false;
				lookupTable.iterateCoverage(lookupTable.data.getOffset(stOffset, 6 + i*2), gids, this);

				if (!coverageContainsGid)
					break;
			}

			if (i == glyphCount) {
				int lookupCount = lookupTable.data.getuint16(stOffset+4);

				for (int j = 0; j < lookupCount; j++)
				{
					int lookupIndex = lookupTable.data.getuint16(stOffset + 6 + 2*glyphCount + 4*j + 2);
					// Note that this is a little bit loose. If we wanted the tightest set, we'd
					// only harvest on the glyphs that this lookup applies to. However, I'll work on the
					// assumption that if this sublookup applies to other glyphs in 'gids', then there will be
					// a path that gets us there in most fonts, so we'll want it eventually anyway. (That's a
					// good sounding justification for matching cooltype's behavior anyway...)
					harvest(gids, lookupIndex, lookups);
				}
			}

			return coverageContainsGid;
		}

		public boolean glyphInfo(int gid, int coverageIndex) {
			coverageContainsGid = true;
			return false;  
		}
	}


	private List gatherContextualGlyphs(int stOffset, Subset gids, Map lookups, List subtables, int subtableIndex) 
	throws InvalidFontException, UnsupportedFontException
	{	    
		int stFormat = lookupTable.data.getuint16(stOffset);
		switch (stFormat) {
		case 1:
			// Since I can't find any promise that these won't recurse to another contextual lookup, I don't
			// want to reuse the same instance of the harvester, so create a new one for each lookup.
			subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ContextFormat1Harvester(this, lookups, true), subtableIndex);
			break;
		case 2:
			subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ContextFormat2Harvester(this, lookups), subtableIndex);
			break;
		case 3:
			int numGlyphs = lookupTable.data.getuint16(stOffset+2);
			if (numGlyphs > 0)
				subtables = gatherLookupsForCoverage(stOffset, 6, gids, subtables, new ContextFormat3SubstHarvester(lookups), subtableIndex);
			break;
		default:
			throw new InvalidFontException("Invalid contextual lookup format (" + stFormat + ")");
		}
		return subtables;
	}

	private List gatherContextualGlyphsForAllSubtables(Subset gids, int offset, Map lookups) 
	throws InvalidFontException, UnsupportedFontException 
	{
		List subtables = null;
		int subtableCount = lookupTable.data.getuint16 (offset+ 4);

		for (int i = 0; i < subtableCount; i++)
		{
			int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
			subtables = gatherContextualGlyphs(stOffset, gids, lookups, subtables, i);
		}

		return subtables;
	}

	private List gatherSingleSubstGlyphsForAllSubtables(Subset gids, int offset, Map lookups) 
	throws InvalidFontException, UnsupportedFontException 
	{
		List subtables = null;
		int subtableCount = lookupTable.data.getuint16 (offset+ 4);

		for (int i = 0; i < subtableCount; i++)
		{
			int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
			int stFormat = lookupTable.data.getuint16(stOffset);
			switch (stFormat) {
			case 1:
			{
				subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, singleSubstFormat1Harvester, i);
				break;
			}
			case 2:
			{
				subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, singleSubstFormat2Harvester, i);
				break;
			}
			}
		}

		return subtables;
	}

	private List gatherChainingGlyphs(int stOffset, Subset gids, Map lookups, List subtables, int subtableIndex) 
	throws InvalidFontException, UnsupportedFontException
	{
		int stFormat = lookupTable.data.getuint16(stOffset);
		switch (stFormat) {
		case 1:
			// Since I can't find any promise that these won't recurse to another contextual lookup, I don't
			// want to reuse the same instance of the harvester, so create a new one for each lookup.
			subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ChainingFormat1Harvester(this, lookups, true), subtableIndex);
			break;
		case 2:
			subtables = gatherLookupsForCoverage(stOffset, 2, gids, subtables, new ChainingFormat2Harvester(this, lookups), subtableIndex);
			break;
		case 3:
			int backtrackCount = lookupTable.data.getuint16(stOffset + 2);
			int inputCount = lookupTable.data.getuint16(stOffset + 4 + 2*backtrackCount);
			if (inputCount > 0)
				subtables = gatherLookupsForCoverage(stOffset, 4 + 2*backtrackCount + 2, gids, subtables, new ChainingFormat3SubstHarvester(lookups), subtableIndex);
			break;
		default:
			throw new InvalidFontException("Invalid contextual lookup format (" + stFormat + ")");
		}
		return subtables;
	}

	private List gatherChainingGlyphsForAllSubtables(Subset gids, int offset, Map lookups) 
	throws InvalidFontException, UnsupportedFontException 
	{
		List subtables = null;
		int subtableCount = lookupTable.data.getuint16 (offset+ 4);

		for (int i = 0; i < subtableCount; i++)
		{
			int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
			subtables = gatherChainingGlyphs(stOffset, gids, lookups, subtables, i);
		}

		return subtables;
	}

	private List gatherExtensionGlyphsForAllSubtables(Subset gids, int offset, Map lookups, int lookupFlag) 
	throws InvalidFontException, UnsupportedFontException 
	{
		List subtables = null;
		List dummySubtables = null;
		int subtableCount = lookupTable.data.getuint16 (offset+ 4);

		for (int i = 0; i < subtableCount; i++)
		{
			int stOffset = lookupTable.data.getOffset (offset, 6 + 2*i);
			int format = lookupTable.data.getuint16(stOffset);
			if (format != 1) // we only understand format 1.
				throw new InvalidFontException("Invalid extension format (" + format + ")");
			int lookupType = lookupTable.data.getuint16(stOffset + 2);
			int extOffset = lookupTable.data.getuint32asint(stOffset + 4, "Only signed extension values supported");

			switch (lookupType) {
			case 1: { 
				int subFormat = lookupTable.data.getuint16(stOffset+extOffset);
				if (subFormat == 1) {
					dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, singleSubstFormat1Harvester, 0);
				} else {
					dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, singleSubstFormat2Harvester, 0);
				}
				break; }
			case 2: { 
				dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, multSubstHarvester, 0);
				break; }
			case 3: { 
				dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, altSubstHarvester, 0);
				break; }
			case 4: { 
				dummySubtables = gatherLookupsForCoverage(stOffset+extOffset, 2, gids, dummySubtables, ligaSubstHarvester, 0);
				break; }
			case 5: { 
				dummySubtables = gatherContextualGlyphs(stOffset+extOffset, gids, lookups, dummySubtables, 0);
				break; }
			case 6: { 
				dummySubtables = gatherChainingGlyphs(stOffset+extOffset, gids, lookups, dummySubtables, 0);
				break; }
			case 7: // Extension lookups can have extension sublookups, per the spec.
			default: {
				throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}

			if (dummySubtables != null) // if this sub-subtable applied, add the extension subtable to the list of relevant subtables.
			{
				if (subtables == null)
					subtables = new ArrayList();
				Integer j = new Integer(i);
				if (!subtables.contains(j))
					subtables.add(j);
			}
		}

		return subtables;
	}

	protected List harvestSubtables(Subset gids, int lookupOffset, int lookupType, int lookupFlag, Map lookups) 
	throws InvalidFontException, UnsupportedFontException
	{
		List subtables = null;
		switch (lookupType) {
		case 1: { 
			subtables = gatherSingleSubstGlyphsForAllSubtables(gids, lookupOffset, lookups);
			break; }
		case 2: { 
			subtables = gatherLookupsForCoverageAllSubtables(gids, lookupOffset, multSubstHarvester);
			break; }
		case 3: { 
			subtables = gatherLookupsForCoverageAllSubtables(gids, lookupOffset, altSubstHarvester);
			break; }
		case 4: { 
			subtables = gatherLookupsForCoverageAllSubtables(gids, lookupOffset, ligaSubstHarvester);
			break; }
		case 5: { 
			subtables = gatherContextualGlyphsForAllSubtables(gids, lookupOffset, lookups);
			break; }
		case 6: { 
			subtables = gatherChainingGlyphsForAllSubtables(gids, lookupOffset, lookups);
			break; }
		case 7: {
			subtables = gatherExtensionGlyphsForAllSubtables(gids, lookupOffset, lookups, lookupFlag);
			break;
		}
		default: {
			throw new InvalidFontException ("Invalid GSUB lookup type (" + lookupType + ")"); }}

		return subtables;
	}
}
