package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

abstract class LookupTableSubsetter extends LayoutTableSubsetter {
	protected final int origNumGlyphs;
	
	// If we are preserving features that have no lookups remaining in the subset, we add
	// a dummy lookup
	private int dummyLookupId = -1; 
	
	LookupTableSubsetter(LayoutTable origTable, OTByteArrayBuilder builder, int numGlyphs)
	{
		super(origTable, builder);
		this.origNumGlyphs = numGlyphs;
	}
	
	interface LookupSubtableGenerator {
		public int writeSubtable() throws InvalidFontException, UnsupportedFontException;
	}
	
	/* -------------------------------------------------------------------------------
	 * Methods subclasses must implement, since they understand the subtable formats 
	 */
	
	/**
	 * Gather the coverages associated with a subtable. Coverages associated
	 * with non-extension lookups are added to coverages. Coverages associated with extension lookups
	 * are added to extensionCoverages. 
	 */
	abstract void gatherCoveragesForSubtable(int stOffset, int lookupType, int lookupIndex, Integer subtableIndex, 
			Map coverages, Map extensionCoverages, Subset subset) 
	throws InvalidFontException, UnsupportedFontException;
	
	/**
	 * Returns the size of extension subtables?
	 */
	abstract int getExtensionSubtableSize();
	
	/**
	 * Write out the subset subtable (leaving holes for the final coverage offsets)
	 */
	abstract int writeSubtable(int origSTOffset, int newSTOffset, 
			Map /*<original Offset, NewCoverage > */ newCoverages, int lookupType, Subset subset,
			LookupSubset lookupSubset)	
	throws InvalidFontException, UnsupportedFontException;
	
	/**
	 * Patch the final coverage offsets into the subset subtable.
	 */
	abstract void patchSubtableCoverage(int origSTOffset,  
			int newSTOffset, Map coverageInfo, int lookupType) throws InvalidFontException;
	
	/**
	 * For a kind of LookupTable, this returns the extension lookup id.
	 */
	abstract int getExtensionLookupType();
	
	/**
	 * Write an extension subtable at newSTOffset. The referenced subtable will have the lookup type
	 * referenced by subTableLookupType.
	 */
	abstract int writeExtensionSubtable(int newSTOffset, int subTableLookupType);
	
	/**
	 * Put the offset to the referenced subtable into the extension subtable.
	 */
	abstract void patchExtensionOffset(int extensionSubtableOffset, int referencedSubtableOffset);
	
	/**
	 * Given the offset to an original extension subtable, this method returns the
	 * lookup type associated with the referenced subtable.
	 */
	abstract int getReferencedLookupType(int origExtSubOffset) throws InvalidFontException;
	
	/**
	 * Given the offset to an original extension subtable, get the offset to the referenced subtable.
	 */
	abstract int getReferencedSubtableOffset(int origExtensionSTOffset) throws InvalidFontException, UnsupportedFontException;
	
	abstract boolean mustKeepFeature(int featureTagOffset) throws InvalidFontException;
	
	/* ------------------------------------------------------------------------------- */

	protected void gatherCoverages(int lookupIndex, Map coverages, Map extensionCoverages, Subset subset, List subtablesToKeep) throws InvalidFontException, UnsupportedFontException {
	     // Convert to lookup offsets
	      int lookupListOffset = ((LookupTable)origTable).getLookupListOffset ();
	      int offset = origTable.data.getOffset (lookupListOffset, 2 + 2*lookupIndex); 
	      
		  for (int i = 0; i < subtablesToKeep.size(); i++)
		  {
		      int lookupType = origTable.data.getuint16 (offset);
			  Integer subtable = (Integer)subtablesToKeep.get(i);
			  int stOffset = origTable.data.getOffset (offset, 6 + 2*subtable.intValue());
			  gatherCoveragesForSubtable(stOffset, lookupType, lookupIndex, subtable, coverages, extensionCoverages, subset);
		  }	    	
	}
	
	protected static class NewCoverage {
		int newOffset = -1;
		OTByteArray newCoverage = null;
		int glyphCount = 0;
		Map /*<Integer, Set <Integer>>*/ referringOrigLookups = new HashMap(); /* the lookups and subtables that refer to this coverage */
		
		int getMaxReferringOrigLookup()
		{
			Iterator iter = referringOrigLookups.keySet().iterator();
			int max = 0;
			while (iter.hasNext())
			{
				int thisLookup = ((Integer)iter.next()).intValue();
				if (max < thisLookup)
					max = thisLookup;
			}
			return max;
		}
		
		int getMaxSubtable(int lookup)
		{
			Set subtables = (Set)referringOrigLookups.get(new Integer(lookup));
			int max = 0;
			Iterator iter = subtables.iterator();
			while (iter.hasNext())
			{
				int theValue = ((Integer)iter.next()).intValue();
				if (max < theValue)
					max = theValue;
			}
			
			return max;
		}
	}
	
	/**
	 * Not a traditional Subset, but Subset has what we need: a way to map
	 * input IDs (in this case feature indices) to subset IDs.
	 */
	static class FeatureSubset extends SubsetDefaultImpl {
		FeatureSubset(LookupTableSubsetter subsetter, OTByteArray origTable, int origFeatureListOffset, Map lookupsToKeep) throws InvalidFontException, UnsupportedFontException
		{
			// Add features to the subset iff they have lookups that will be kept. Because we look at the features in feature index order,
			// the new subset feature indices will have that order preserved.
			super(origTable.getuint16(origFeatureListOffset), true);
			int count = origTable.getuint16(origFeatureListOffset);
			for (int i = 0; i < count; i++)
			{
				int origFeatureOffset = origTable.getOffset(origFeatureListOffset, 2 + 6*i + 4);
				if (featureContainsLookups(origTable, origFeatureOffset,lookupsToKeep) ||
						subsetter.mustKeepFeature(origFeatureListOffset + 2 + 6*i))
				{
					getSubsetGid(i);
				}
			}
		}
		
	
	
		// look at all of the lookups contained in the feature and see if any of them are in lookupsToKeep. If so, return true. Else return false.
		private boolean featureContainsLookups(OTByteArray origTable, int origFeatureOffset, Map lookupsToKeep) throws InvalidFontException
		{
			int lookupCount = origTable.getuint16(origFeatureOffset + 2);
			for (int i = 0; i < lookupCount; i++)
			{
				int index = origTable.getuint16(origFeatureOffset + 4 + 2*i);
				if (lookupsToKeep.containsKey(new Integer(index)))
					return true;
			}
			return false;
		}
	}

	/**
	 * Not a traditional Subset, but Subset has what we need: a way to map
	 * input IDs (in this case lookup ids) to subset IDs.
	 */
	static class LookupSubset extends SubsetDefaultImpl {
		final TreeMap lookupsToKeep;
		
		LookupSubset(TreeMap lookupsToKeep, int origLookupCount) 
		throws InvalidFontException, UnsupportedFontException
		{
			// Since the lookups are sorted in the TreeMap, the subset gids will be sorted correctly.
			
			super (origLookupCount, true);
			this.lookupsToKeep = lookupsToKeep;
			Iterator iter = lookupsToKeep.keySet().iterator();
			while (iter.hasNext())
			{
				Integer lookup = (Integer)iter.next();
				getSubsetGid(lookup.intValue());
			}
		}
		
		List getSubtables(int origGID)
		{
			return (List)lookupsToKeep.get(new Integer(origGID));
		}
	}

	private void moveCoveragesForLookup(int origLookupId, Map fromCoverages, Map toCoverages)
	{
		Iterator coverageIter = fromCoverages.keySet().iterator();
		while (coverageIter.hasNext())
		{
			Integer origCoverageOffset = (Integer)coverageIter.next();
			NewCoverage newCoverage = (NewCoverage)fromCoverages.get(origCoverageOffset);
			Integer oLID = new Integer(origLookupId);
			
			// if this coverage was referred to by origLookupId, move it to the extensions Map.
			if (newCoverage.referringOrigLookups.containsKey(oLID))
			{
				// If the extensions Map already has this coverage, just add this lookup.
				if (toCoverages.containsKey(origCoverageOffset))
				{
					NewCoverage extensionCopy = (NewCoverage)toCoverages.get(origCoverageOffset);
					extensionCopy.referringOrigLookups.put(oLID, newCoverage.referringOrigLookups.get(oLID));
				} else
				{

					NewCoverage extensionCopy = new NewCoverage();
					extensionCopy.glyphCount = newCoverage.glyphCount;
					extensionCopy.newCoverage = newCoverage.newCoverage;
					extensionCopy.referringOrigLookups.put(oLID, newCoverage.referringOrigLookups.get(oLID));
					
					toCoverages.put(origCoverageOffset, extensionCopy);
				}
	
				newCoverage.referringOrigLookups.remove(oLID);
			}
			
		}
	}

	static protected void addToCoveragesMap(LayoutTable origTable, int absoluteCoverageOffset, Map coverages, int lookupIndex, Integer stIndex, Subset subset) 
	throws InvalidFontException, UnsupportedFontException
	{
		Integer coverageOffsetInteger = new Integer(absoluteCoverageOffset);
		if (!coverages.containsKey(coverageOffsetInteger))
		{
			NewCoverage newCoverage = new NewCoverage();
			Set subtables = new HashSet();
			subtables.add(stIndex);
			newCoverage.referringOrigLookups.put(new Integer(lookupIndex), subtables);
			
			CoverageGenerator generator = CoverageGenerator.newInstance(origTable, absoluteCoverageOffset, subset);
			newCoverage.newCoverage = generator.generateCoverage().toOTByteArray();
			newCoverage.glyphCount = generator.numGlyphsFound;
			coverages.put(coverageOffsetInteger, newCoverage);
		} else
		{
			NewCoverage newCoverage = (NewCoverage)coverages.get(coverageOffsetInteger);
			Integer lookup = new Integer(lookupIndex);
			if (newCoverage.referringOrigLookups.containsKey(lookup))
			{
				Set referringSubtables = (Set)newCoverage.referringOrigLookups.get(lookup);
				referringSubtables.add(stIndex);
			} else 
			{
				Set subtables = new HashSet();
				subtables.add(stIndex);
				newCoverage.referringOrigLookups.put(new Integer(lookupIndex), subtables);
			}
		}
	}
	
	
	private int getLookupOffset(int origLLOffset, int origLookupID) throws InvalidFontException
	{
	    return origTable.data.getOffset(origLLOffset, 2 + 2*origLookupID); 	
	}
	
	// For all of the subtables in subtablesToKeep, patch in the offsets to the coverages referenced by them.
	private void patchCoverages(int origLLOffset, int origLookupID,  
			List subtablesToKeep, List newSubtableOffsets, Map regularCoverages)
	throws InvalidFontException
	{
		 int origLookupOffset = getLookupOffset(origLLOffset, origLookupID);
		 int lookupType = origTable.data.getuint16(origLookupOffset);
		 for (int i = 0; i < subtablesToKeep.size(); i++)
		 {
			 int subtable = ((Integer)subtablesToKeep.get(i)).intValue();
			 patchSubtableCoverage(origTable.data.getOffset(origLookupOffset, 6 + 2*subtable),
					 ((Integer)newSubtableOffsets.get(i)).intValue(), regularCoverages, lookupType);
		 }
	}
	
	// The subtables in subtablesToKeep either originally were extensions or they have been converted to extensions.
	// Patch all of the subtables referenced from those extensions so that they have the offsets to their coverages.
	private void patchExtensionCoverages(int origLookupOffset, List subtablesToKeep, 
			List newSubtableOffsets, Map extensionCoverages) throws InvalidFontException, UnsupportedFontException
	{
		for (int i = 0; i < subtablesToKeep.size(); i++)
		{
			int subtable = ((Integer)subtablesToKeep.get(i)).intValue();
			int origExtensionSTOffset = origTable.data.getOffset(origLookupOffset, 6 + 2*subtable);
			int lookupType = origTable.data.getuint16(origLookupOffset);
			
			if (lookupType == getExtensionLookupType())
			{
				lookupType = getReferencedLookupType(origExtensionSTOffset);
				origExtensionSTOffset = getReferencedSubtableOffset(origExtensionSTOffset);
			}

			patchSubtableCoverage(origExtensionSTOffset, ((Integer)newSubtableOffsets.get(i)).intValue(), extensionCoverages, lookupType);
		 }
	}
		
	/**
	 * Write out the new lookup and all of the subtables in subtablesToKeep.
	 */
	private int writeLookup(int newLookupOffset,
			Map newCoverages, int origLookupOffset, List subtablesToKeep, Subset subset,
			List newSubtableOffsets, LookupSubset lookupSubset)
	throws InvalidFontException, UnsupportedFontException
	{	      
	    int subtableCount = subtablesToKeep.size();
	    int size = 6 + 2*subtableCount;
	    builder.ensureCapacity(newLookupOffset + size);
	    
	    int lookupType = origTable.data.getuint16(origLookupOffset);
	    builder.setuint16(newLookupOffset, lookupType);
	    builder.setuint16(newLookupOffset + 2, origTable.data.getuint16(origLookupOffset+2));
	    builder.setuint16(newLookupOffset + 4, subtableCount);
		
	    for (int i = 0; i < subtableCount; i++)
		{
	    	int subtable = ((Integer)subtablesToKeep.get(i)).intValue();
	    	builder.setuint16(newLookupOffset + 6 + 2*i, size);
	    	newSubtableOffsets.add(i, new Integer(newLookupOffset + size));
	    	
	    	size += writeSubtable(origTable.data.getOffset(origLookupOffset, 6 + 2*subtable), 
	    			newLookupOffset+size, newCoverages, lookupType, subset, lookupSubset);
			
		}
	    
	    return size;
	}
	
	/**
	 * Whether or not the original lookup was an extension, the new lookup needs to be. 
	 * Handle writing out either situation and write out all of the subtables that we want to keep as
	 * extension subtables.
	 */
	private int writeLookupAsExtension(int newLookupOffset, int origLookupOffset, List subtablesToKeep, List newSubtableOffsets)
	throws InvalidFontException, UnsupportedFontException
	{	      

	    int subtableCount = subtablesToKeep.size();
	    int size = 6 + 2*subtableCount;
	    int lookupType = origTable.data.getuint16(origLookupOffset);
	    boolean lookupIsExtension = (lookupType == getExtensionLookupType());
	    
		builder.setuint16(newLookupOffset, getExtensionLookupType());
		builder.setuint16(newLookupOffset + 2, origTable.data.getuint16(origLookupOffset+2));
		builder.setuint16(newLookupOffset + 4, subtableCount);
			
		for (int i = 0; i < subtableCount; i++)
		{
			int subtable = ((Integer)subtablesToKeep.get(i)).intValue();
		 	builder.setuint16(newLookupOffset + 6 + 2*i, size);
		   	newSubtableOffsets.add(i, new Integer(newLookupOffset + size));
		    	
		   	if (lookupIsExtension)
		   		size += writeSubtable(origTable.data.getOffset(origLookupOffset, 6 + 2*subtable), 
		   			newLookupOffset+size, null, lookupType, null, null);
		   	else
		   		size += writeExtensionSubtable(newLookupOffset+size, lookupType);
				
		}
	    
	    return size;
	    
	}
	
	/**
	 * Write the subtables referenced by extension lookup subtables and patch the offset in the extension
	 * lookup subtables. The original subtables may have been extension lookups or it could be that we had
	 * to convert non-extension lookups to extension lookups. This code handles both cases. 
	 */
	private int writeExtensionSubtables(int newExtensionStartOffset, List newSubtableOffsets, 
			Map newCoverages, int origLookupOffset, List subtablesToKeep, Subset subset,
			LookupSubset lookupSubset, int origLookupType, int origLookupID) throws InvalidFontException, UnsupportedFontException
	{
		int extensionSubtablesSize = 0;
		for (int i = 0; i < subtablesToKeep.size(); i++)
		{
			int extensionSubtableOffset = ((Integer)newSubtableOffsets.get(i)).intValue();
			int origSTOffset = origTable.data.getOffset(origLookupOffset, 6 + 2*((Integer)subtablesToKeep.get(i)).intValue());
			patchExtensionOffset(extensionSubtableOffset, newExtensionStartOffset + extensionSubtablesSize);
			newSubtableOffsets.set(i, new Integer(newExtensionStartOffset + extensionSubtablesSize));
			int subLookupType = origLookupType;
			
			if (origLookupType == getExtensionLookupType())
			{
				subLookupType = getReferencedLookupType(origSTOffset);
				origSTOffset = getReferencedSubtableOffset(origSTOffset);
			}
			
			extensionSubtablesSize += writeSubtable(origSTOffset, newExtensionStartOffset + extensionSubtablesSize, 
					newCoverages, subLookupType, subset, lookupSubset);
			

			/*
			 * If we have written out all of the lookups/subtables that use this coverage, write out the coverages so that it can
			 * (probably) be within a 16-bit offset of all referring lookups.
			 */
			Iterator coverageIter = newCoverages.keySet().iterator();
			while (coverageIter.hasNext())
			{
				Integer origCoverageOffset = (Integer)coverageIter.next();
				NewCoverage newCoverage = (NewCoverage)newCoverages.get(origCoverageOffset);
				if (newCoverage.getMaxReferringOrigLookup() == origLookupID && newCoverage.getMaxSubtable(origLookupID) == ((Integer)subtablesToKeep.get(i)).intValue())
				{
					newCoverage.newOffset = newExtensionStartOffset + extensionSubtablesSize; // record where we are writing this coverage, so we can patch it into the referring subtables
					extensionSubtablesSize += writeByteArrayAtOffset(newExtensionStartOffset + extensionSubtablesSize, newCoverage.newCoverage);
				}
			}
		}
		
		return extensionSubtablesSize;
	}
	
	private int writeDummyLookup(int newDummyOffset)
	{
		builder.ensureCapacity(newDummyOffset + 6);
		builder.setuint16(newDummyOffset, 0); // doesn't matter what this is so long as it is legal for all LookupTables
		builder.setuint16(newDummyOffset+2, 0); // doesn't matter what this is so long as it is legal for all LookupTables
		builder.setuint16(newDummyOffset+2, 0); // most important...no subtables.
		return 6;
	}
	
	/**
	 * Entry point to get a new lookup list built and emitted at newLLOffset. It only
	 * preserves lookups that are specified in lookupToKeep.
	 */
	protected int buildLookupList(int origLLOffset,
			int newLLOffset, 
			LookupSubset lookupToKeep, Subset subset)
	throws InvalidFontException, UnsupportedFontException
	{
		int llSize = 2 + 2*lookupToKeep.getNumGlyphs() + ((dummyLookupId == -1) ? 0:2);
		Map regularCoverages = new HashMap();
		Map extensionCoverages = new HashMap();
		
		builder.ensureCapacity(newLLOffset + llSize);
		builder.setuint16(newLLOffset, lookupToKeep.getNumGlyphs() + ((dummyLookupId == -1) ? 0:1));
		
		int subtableCount = 0;
		
		/* We want to preserve shared coverages for the space savings. Gather up all
		 * of the information for all of the coverages we want to preserve.
		 */
		for (int i = 0; i < lookupToKeep.getNumGlyphs(); i++)
		{
			List subtablesToKeep = lookupToKeep.getSubtables(lookupToKeep.getFullGid(i));
			subtableCount += subtablesToKeep.size();
				
			gatherCoverages(lookupToKeep.getFullGid(i), regularCoverages, 
					extensionCoverages, subset, subtablesToKeep);
		}

		/* 
		 * Figure out how much room all of the non-extension coverages take so that
		 * we can make sure to fit them in the non-extension region of the GSUB.
		 */
		int coverageSize = 0;
		Iterator iter = regularCoverages.keySet().iterator();
		while (iter.hasNext())
		{
			Object key = iter.next();
			NewCoverage thisCoverage = (NewCoverage)regularCoverages.get(key);
			coverageSize += thisCoverage.newCoverage.getSize();
		}
		
		/*
		 * Write out lookups until we can no longer fit them in the non-extension region
		 * of the GSUB. Keep track of where all of the new subtables get written so we can
		 * more easily patch in the final coverage offsets once we know them.
		 */
		int subtablesWritten = 0;
		int i;
		int spaceTaken = 0;
		List /* Integers */[] newSubtableOffsets = new List[lookupToKeep.getNumGlyphs()];
		List extensionList = new ArrayList();
		
		for (i = 0; i < lookupToKeep.getNumGlyphs(); i++)
		{
			newSubtableOffsets[i] = new ArrayList();
			List subtablesToKeep = lookupToKeep.getSubtables(lookupToKeep.getFullGid(i));
			int originalLookupOffset = getLookupOffset(origLLOffset, lookupToKeep.getFullGid(i));
			
			builder.setuint16(newLLOffset + 2 + 2*i, llSize);
			
			if (origTable.data.getuint16(originalLookupOffset) == getExtensionLookupType())
			{
				extensionList.add(new Integer(i));
			}
			
			int thisLookupSize = writeLookup(newLLOffset + llSize, 
					regularCoverages, originalLookupOffset, subtablesToKeep, subset, 
					newSubtableOffsets[i], lookupToKeep);
			int thisNumSubtables = subtablesToKeep.size();
			
			int roomNeeded = spaceTaken + thisLookupSize + // the room we've used so far
				(lookupToKeep.getNumGlyphs() - (i+1))*6 + // room to write the remaining lookup tables (the offset to the subtable is counted in the + 2 below)
				((dummyLookupId == -1) ? 0:6) + // room to write out dummy lookup
				(subtableCount - (subtablesWritten + thisNumSubtables))*(getExtensionSubtableSize() + 2) // room to write remaining subtables as extensions
				+ coverageSize; // room to write the coverages
			
			// if everything after this lookup were an extension, would this lookup fit? If so, keep the lookup and
			// keep going. Otherwise, backup, and move this lookup (and all remaining) to be an extension
			// This is overly conservative. We really only need offsets to be within maxOffset from the relevant subtable. 
			// But balancing that with sharing coverages between lookups gets very complicated...
			if (roomNeeded > maxOffset)
			{
				break;
			}
			subtablesWritten += thisNumSubtables;
			spaceTaken += thisLookupSize;
			llSize += thisLookupSize;
		}
	
		if (i < lookupToKeep.getNumGlyphs())
		{
			for (; i < lookupToKeep.getNumGlyphs(); i++)
			{
				newSubtableOffsets[i] = new ArrayList();
				int originalLookupOffset = getLookupOffset(origLLOffset, lookupToKeep.getFullGid(i));
				List subtablesToKeep = lookupToKeep.getSubtables(lookupToKeep.getFullGid(i));
				builder.setuint16(newLLOffset + 2 + 2*i, llSize);
				llSize += writeLookupAsExtension(newLLOffset + llSize, originalLookupOffset, subtablesToKeep, newSubtableOffsets[i]);
				extensionList.add(new Integer(i));

				// coverages that used to be for regular lookups are now for extension lookups
				moveCoveragesForLookup(lookupToKeep.getFullGid(i), regularCoverages, extensionCoverages);
			}
		}
		
		/*
		 * Write out the non-extension coverages.
		 */
		Iterator coverageIter = regularCoverages.keySet().iterator();
		while (coverageIter.hasNext())
		{
			Integer origCoverageOffset = (Integer)coverageIter.next();
			NewCoverage newCoverage = (NewCoverage)regularCoverages.get(origCoverageOffset);
			if (newCoverage.referringOrigLookups.isEmpty()) // if no non-extension lookups refer to this coverage anymore, don't write it out here.
				continue;
			newCoverage.newOffset = newLLOffset + llSize; // record where we are writing this coverage, so we can patch it into the referring subtables
			llSize += writeByteArrayAtOffset(newLLOffset + llSize, newCoverage.newCoverage);
		}
		
		/*
		 * Patch in the offsets to the non-extension coverages.
		 */
		for (i = 0; i < newSubtableOffsets.length; i++)
		{
			if (extensionList.contains(new Integer(i)))
				continue;
			patchCoverages(origLLOffset, lookupToKeep.getFullGid(i),  
					lookupToKeep.getSubtables(lookupToKeep.getFullGid(i)), newSubtableOffsets[i], regularCoverages);
		}
		
		if (dummyLookupId != -1)
		{
			builder.setuint16(newLLOffset + 2 + 2*dummyLookupId, llSize);
			llSize += writeDummyLookup(newLLOffset + llSize);
		}
		
		/*
		 * Write out the subtables referenced by the extension lookups
		 */
		for (i = 0; i < extensionList.size(); i++)
		{
			int subsetLookup = ((Integer)extensionList.get(i)).intValue();
			List subtablesToKeep = lookupToKeep.getSubtables(lookupToKeep.getFullGid(subsetLookup));
			int originalLookupOffset = getLookupOffset(origLLOffset, lookupToKeep.getFullGid(subsetLookup));
			int origLookupType = origTable.data.getuint16(originalLookupOffset);
			llSize += writeExtensionSubtables(newLLOffset + llSize, newSubtableOffsets[subsetLookup], extensionCoverages, 
					originalLookupOffset, subtablesToKeep, subset,
					lookupToKeep, origLookupType, lookupToKeep.getFullGid(subsetLookup));
		}
		
		/*
		 * Patch in the offsets to the extension coverages.
		 */
		for (i = 0; i < extensionList.size(); i++)
		{
			int subsetLookup = ((Integer)extensionList.get(i)).intValue();
			int originalLookupOffset = getLookupOffset(origLLOffset, lookupToKeep.getFullGid(subsetLookup));
			List subtablesToKeep = lookupToKeep.getSubtables(lookupToKeep.getFullGid(subsetLookup));
			patchExtensionCoverages(originalLookupOffset, subtablesToKeep, newSubtableOffsets[subsetLookup], extensionCoverages);
		}
		
		return llSize;
	}

	/**
	 * Subset a LookupTable so that it only contains the lookups in 'lookupToKeep' and glyphs in 'subset'.
	 * @param lookupToKeep A TreeMap containing the lookup indices to keep along with the subtables in those lookups to keep.
	 * @return the new lookup table
	 * @throws InvalidFontException
	 * @throws UnsupportedFontException
	 */
	OTByteArrayBuilder subsetAndStream(TreeMap /*<Integer, List<Integer>*/ lookupToKeep, Subset subset) throws InvalidFontException, UnsupportedFontException {
		
		int origFLOffset = origTable.data.getOffset(0, 6);
		LookupSubset lookupSubset = new LookupSubset(lookupToKeep, origTable.data.getuint16(((LookupTable)origTable).getLookupListOffset()));
		FeatureSubset featureSubset = new FeatureSubset(this, origTable.data, origFLOffset, lookupToKeep);
		
		builder.setuint32(0, 0x00010000); //emit ths header.version
		int origOffset = origTable.data.getOffset(0, 4);
		int gsubSize = 10;
		builder.setuint16(4, gsubSize); //emit the ScriptList offset
		gsubSize += buildScriptList(origOffset, gsubSize, featureSubset, lookupSubset);
		
		builder.setuint16(6, gsubSize); // emit the FeatureList offset.
		gsubSize += buildFeatureList(origFLOffset, gsubSize, featureSubset, lookupSubset);
		
	
		origOffset = origTable.data.getOffset(0, 8);
		builder.setuint16(8, gsubSize);  // emit the LookupList offset.
		buildLookupList(origOffset, gsubSize, lookupSubset, subset);
		
		return builder;
	}
	
	/*
	 * Entry point to build a new script list at the offset newSLOffset. It must preserve all of the original scripts, 
	 * or the behavior of the font could be changed (suddenly default features are applied when they wouldn't have previously).
	 */
	private int buildScriptList(int origSLOffset,  
			int newSLOffset, 
			FeatureSubset featuresToKeep, LookupSubset lookupToKeep) throws InvalidFontException, UnsupportedFontException
	{
		int origScriptCount = origTable.data.getuint16(origSLOffset);
		int slSize = 2;

		slSize +=  origScriptCount*6;
		builder.ensureCapacity(newSLOffset + slSize);
		builder.setuint16(newSLOffset, origScriptCount); // emit the number of scripts
		for (int i = 0; i < origScriptCount; i++)
		{
			for (int k = 0; k < 4; k++)
				builder.setuint8(newSLOffset + 2 + 6*i + k, origTable.data.getint8(origSLOffset + 2 + 6*i + k)); // emit the ScriptTag
			builder.setuint16(newSLOffset + 2 + 6*i + 4, slSize); // emit the Script offset.
			slSize += writeScript(origTable.data.getOffset(origSLOffset, 2 + 6*i + 4), featuresToKeep, newSLOffset + slSize, lookupToKeep);
		}
		
		
		return slSize;
	}

	private int writeScript(int origScriptOffset, FeatureSubset featuresToKeep, int newScriptOffset, LookupSubset lookupToKeep) 
	throws InvalidFontException, UnsupportedFontException
	{
		int scriptSize = 0;
		int origLangSysCount = origTable.data.getuint16(origScriptOffset + 2);
		
		scriptSize += 4 + origLangSysCount*6;
		builder.ensureCapacity(newScriptOffset + scriptSize);
		builder.setuint16(newScriptOffset+2, origLangSysCount); // emit the LangSysCount
		
		int origDefaultLangSysOffset = origTable.data.getOffset(origScriptOffset, 0);
		if (origDefaultLangSysOffset != 0)
		{
			builder.setuint16(newScriptOffset, scriptSize); // emit the DefaultLangSys offset.
			scriptSize += writeLangSys(origDefaultLangSysOffset, featuresToKeep, newScriptOffset + scriptSize, lookupToKeep);
		} else
		{
			builder.setuint16(newScriptOffset, 0);
		}
		
		// emit the LangSysRecords
		for (int i = 0; i < origLangSysCount; i++)
		{
			builder.setuint16(newScriptOffset + 4 + 6*i + 4, scriptSize); // emit the LangSys offset
			for (int k = 0; k < 4; k++) // emit the LangSysTag
				builder.setuint8(newScriptOffset + 4 + 6*i + k, origTable.data.getint8(origScriptOffset + 4 + 6*i + k));
			scriptSize += writeLangSys(origTable.data.getOffset(origScriptOffset, 4 + 6*i + 4), featuresToKeep, newScriptOffset + scriptSize, lookupToKeep);
		}
		
		
		return scriptSize;
	}
	
	private int writeLangSys(int origLSOffset, FeatureSubset featuresToKeep, int newLSOffset, LookupSubset lookupToKeep)
	throws InvalidFontException, UnsupportedFontException
	{
		int origFeatureCount = origTable.data.getuint16(origLSOffset + 4);
		List featureToKeepList = new ArrayList();
		int langSysSize = 0;
		for (int i = 0; i < origFeatureCount; i++)
		{
			int featureIndex = origTable.data.getuint16(origLSOffset + 6 + 2*i);
			if (featuresToKeep.getExistingSubsetGid(featureIndex) != -1)
			{
				featureToKeepList.add(new Integer(i));
			}
		}

		langSysSize += 6 + 2*featureToKeepList.size();
		builder.ensureCapacity(newLSOffset + langSysSize);
		builder.setuint16(newLSOffset, 0); // emit the LookupOrder
		
		int requiredFeature = origTable.data.getuint16(origLSOffset + 2);
		if (requiredFeature != 0xffff && featuresToKeep.getExistingSubsetGid(requiredFeature) != -1)
		{
			builder.setuint16(newLSOffset + 2, featuresToKeep.getExistingSubsetGid(requiredFeature));   // emit the new required feature index
		} else
		{
			builder.setuint16(newLSOffset + 2, 0xffff);
		}
		
		builder.setuint16(newLSOffset + 4, featureToKeepList.size());  // emit the new feature count.
			
		for (int i = 0; i < featureToKeepList.size(); i++)
		{
			int j = ((Integer)featureToKeepList.get(i)).intValue();
			int featureIndex = origTable.data.getuint16(origLSOffset + 6 + 2*j);
			int newFeatureIndex = featuresToKeep.getExistingSubsetGid(featureIndex);
			builder.setuint16(newLSOffset + 6 + 2*i, newFeatureIndex); // emit the new feature index.
		}
		return langSysSize;
	}
	

	
	/*
	 * Entry point to get a new feature list built and emitted at offset newFLOffset. 
	 * It only preserves features that are specified and that have preserved lookups.
	 */
	private int buildFeatureList(int origFLOffset, 
			int newFLOffset, 
			LookupTableSubsetter.FeatureSubset featuresToKeep, LookupTableSubsetter.LookupSubset lookupSubset) 
	throws InvalidFontException
	{
		int flSize = 2 + 6*featuresToKeep.getNumGlyphs();
		builder.ensureCapacity(newFLOffset + flSize);
		builder.setuint16(newFLOffset, featuresToKeep.getNumGlyphs()); // emit the new feature count.
		
		// emit the feature records for the features we want to keep
		for (int i = 0; i < featuresToKeep.getNumGlyphs(); i++)
		{
			int origFeature = featuresToKeep.getFullGid(i);
			for (int j = 0; j < 4; j++) // emit the feature Tag
			{
				builder.setuint8(newFLOffset + 2 + 6*i + j, origTable.data.getint8(origFLOffset + 2 + 6*origFeature + j));
			}
			
			builder.setuint16(newFLOffset + 2 + 6*i + 4, flSize); // emit the feature offset
			flSize += writeFeature(origTable.data.getOffset(origFLOffset, 2 + 6*origFeature + 4),
					newFLOffset + flSize, lookupSubset);
		}
		
		return flSize;
	}
	
	private int writeFeature(int origFeatOffset, 
			int newFeatOffset, LookupSubset lookupSubset)
	throws InvalidFontException
	{
		int featureSize = 0;
		int origLookupCount = origTable.data.getuint16(origFeatOffset + 2);
		int newLookupCount = 0;
		for (int i = 0; i < origLookupCount; i++)
		{
			int origLookupIndex = origTable.data.getuint16(origFeatOffset + 4 + 2*i);
			if (lookupSubset.getExistingSubsetGid(origLookupIndex) != -1)
				newLookupCount++;
		}
		
		if (origLookupCount > 0 && newLookupCount == 0)
		{
			// we are preserving a feature and need a dummy lookup.
			newLookupCount++;
			if (dummyLookupId == -1)
				dummyLookupId = lookupSubset.getNumGlyphs();
		}
		
		featureSize += 4 + 2*newLookupCount;
		
		builder.ensureCapacity(newFeatOffset + featureSize);
		builder.setuint16(newFeatOffset, 0); // emit the FeatureParams
		builder.setuint16(newFeatOffset + 2, newLookupCount); // emit the new LookupCount
		int j = 0;
		for (int i = 0; i < origLookupCount; i++)
		{
			int origLookupIndex = origTable.data.getuint16(origFeatOffset + 4 + 2*i);
			int newLookupIndex = lookupSubset.getExistingSubsetGid(origLookupIndex);
			if (newLookupIndex != -1)
			{
				builder.setuint16(newFeatOffset + 4 + 2*j, newLookupIndex); // emit the new feature indices.
				j++;
			}
		}
		
		if (origLookupCount > 0 && newLookupCount == 0)
		{
			builder.setuint16(newFeatOffset + 4, dummyLookupId);
		}
		
		return featureSize;
	}
	
	static protected void patchCoverageAtOffset(LayoutTable origTable, OTByteArrayBuilder builder, int origSTOffset, 
			int newSTOffset, int coverageDelta, Map coverageInfo) throws InvalidFontException {
		Integer coverageOffset = new Integer(origTable.data.getOffset(origSTOffset, coverageDelta));
		NewCoverage coverage = (NewCoverage)coverageInfo.get(coverageOffset);
		if (coverage.newOffset - newSTOffset > 0xffff)
			throw new RuntimeException("New coverage offset is out of range!");
		else if (coverage.newOffset < newSTOffset)
			throw new RuntimeException("New coverage written out before subtable!");
		builder.setuint16(newSTOffset + coverageDelta, coverage.newOffset - newSTOffset);
	}
	
	
}
