/*
*
*	File: Vmtx.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'vmtx' table.
 *
 * <h4>Version handling</h4>
 * 
 * <p>'vmtx' tables are not versionned.</p>
 *
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 **/

final public class Vmtx extends Table {
  
  // The 'vmtx' table does not stand on its own, as 
  // one needs the 'hhea.numberOfHMetrics' field to interpret it.
  
  // So this class collects all these pieces and gives the illusion
  // of a standalone 'vmtx' table.
  
  protected final int numberOfVMetrics;
  protected final VmtxRaw vmtx;
  
  protected Vmtx (VmtxRaw vmtx, int numberOfVHMetrics) {
    this.vmtx = vmtx;
    this.numberOfVMetrics = numberOfVHMetrics;
  }
  
  /** Get the left side bearing of the glyph <code>glyphID</code>. */
  public int getTopSideBearing (int glyphID) throws InvalidFontException {
    return vmtx.getTopSideBearing (glyphID, numberOfVMetrics);
  }

  /** Get the horizontal advance of the glyph <code>glyphID</code>. */
  public int getHorizontalAdvance (int glyphID) throws InvalidFontException {
    return vmtx.getVerticalAdvance (glyphID, numberOfVMetrics);
  }
  
  
  public void subsetAndStream (Subset subset, Map tables)
  throws InvalidFontException {
	OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(subset.getNumGlyphs () * 4);
    int o = 0;
    for (int i = 0; i < subset.getNumGlyphs (); i++) {
      int fullId = subset.getFullGid (i);
      newData.setuint16 (o, getHorizontalAdvance (fullId));
      newData.setint16 (o + 2, getTopSideBearing (fullId));
      o += 4; }
      
    tables.put (new Integer (Tag.table_vmtx), newData); 
  }  
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = vmtx.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_vmtx), newData);
  }
}

