/*
 *
 *	File: CSSPDF16Atribute.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.inlineformatting.css20;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import com.adobe.fontengine.inlineformatting.AttributedRun;
import com.adobe.fontengine.inlineformatting.ElementAttribute;

/**
 * This class holds the CSS attributes that can be used to tag an {@link AttributedRun}.  Objects
 * of this class are used as the value entries of the {@link ElementAttribute#CSS20Attribute} tag.
 * 
 * <h4>Concurrency</h4>
 * 
 * Instances of this class are immutable after construction and contain no mutable
 * static data. Therefore, they are threadsafe.
 */
final public class CSS20Attribute
{
	private final List			familyNames;   
	private final CSSWeightValue	weight;
	private final CSSStyleValue	style;
	private final CSSVariantValue	variant;   
	private final CSSStretchValue	stretch;
	private final double			pointSize;
	private final double			opticalSize;

	/**
	 * Constructor.
	 * It is highly recommended that the {@link java.lang.String String} objects have had the
	 * <code>java.lang.String.intern()</code> method called on them.  This will <b>greatly</b> improve
	 * the performance of the formatting operations.
	 * @param familyNames array of the font family names - this must not be <code>null</code>
	 * @param style the font style
	 * @param variant the font variant
	 * @param weight the font weight
	 * @param stretch the font stretch
	 * @param pointSize the point size (this must be equal to or greater than zero)
	 * @param opticalSize the optical size used for selecting optical font variants
	 */
	public CSS20Attribute(String[] familyNames, CSSStyleValue style, CSSVariantValue variant,
			CSSStretchValue stretch, CSSWeightValue weight, double pointSize, double opticalSize) 
	{
		this.familyNames = new ArrayList(familyNames.length);
		for (int i = 0; i < familyNames.length; i++)
		{
			this.familyNames.add(i, familyNames[i]);
		}
		this.weight = weight;
		this.style = style;
		this.variant = variant;
		this.stretch = stretch;
		this.pointSize = pointSize;
		this.opticalSize = opticalSize;
	}

	/**
	 * Constructor.
	 * This is a convenience constructor that assumes that the optical size is equal to
	 * the point size.  This is likely to the be the case for most uses.
	 * 
	 * It is highly recommended that the {@link java.lang.String String} objects have had the
	 * <code>java.lang.String.intern()</code> method called on them.  This will <b>greatly</b> improve
	 * the performance of the formatting operations.
	 * @param familyNames array of the font family names - this must not be <code>null</code>
	 * @param style the font style
	 * @param variant the font variant
	 * @param weight the font weight
	 * @param stretch the font stretch
	 * @param pointSize the point size (this must be equal to or greater than zero)
	 */
	public CSS20Attribute(String[] familyNames, CSSStyleValue style, CSSVariantValue variant,
			CSSStretchValue stretch, CSSWeightValue weight, double pointSize) 
	{
		this(familyNames, style, variant, stretch, weight, pointSize, pointSize);
	}

	/**
	 * Copy constructor.
	 * @param properties
	 */
	public CSS20Attribute(CSS20Attribute properties)
	{
		this.familyNames = new ArrayList(properties.familyNames);
		this.weight = properties.weight;
		this.style = properties.style;
		this.variant = properties.variant;
		this.stretch = properties.stretch;
		this.pointSize = properties.pointSize;
		this.opticalSize = properties.opticalSize;
	}

	/**
	 * @return - the familyNames.
	 */
	public List getFamilyNamesList()
	{
		return new ArrayList(this.familyNames);
	}

	public String[] getFamilyNames()
	{
		String[] array = new String[familyNames.size()];
		return (String[]) this.familyNames.toArray(array);
	}

	/**
	 * @return - the style.
	 */
	public CSSStyleValue getStyle()
	{
		return this.style;
	}

	/**
	 * @return - the weight after resolving for "bolder" and "lighter".
	 */
	public int getWeight()
	{            
		return this.weight.getValue();
	}

	/**
	 * @return - the variant.
	 */
	public CSSVariantValue getVariant()
	{
		return this.variant;
	}

	/**
	 * @return - the stretch.
	 */
	public CSSStretchValue getStretch()
	{
		return this.stretch;
	}

	/**
	 * @return - the point size.
	 */
	public double getPointSize()
	{
		return this.pointSize;
	}

	/**
	 * @return - the optical size.
	 */
	public double getOpticalSize()
	{
		return this.opticalSize;
	}

	@Override
	public String toString() 
	{
		return new String("[" + this.familyNames + "; " 
				+ this.pointSize + "; " + this.opticalSize + "; " + this.weight + "; " 
				+ this.style + "; " + this.stretch + "; " + this.style + "]");
	}

	/**
	 * This class provides an enumeration of the possible values for the CSS Stretch attribute.
	 * 
	 * <h4>Concurrency</h4>
	 * 
	 * Instances of this class are immutable after construction and contain no mutable static
	 * data. Therefore, they are threadsafe.
	 */
	public static final class CSSStretchValue implements Serializable
	{
		public static final CSSStretchValue ULTRACONDENSED = new CSSStretchValue("ultracondensed", 100);
		public static final CSSStretchValue EXTRACONDENSED = new CSSStretchValue("extracondensed", 200);
		public static final CSSStretchValue CONDENSED = new CSSStretchValue("condensed", 300);
		public static final CSSStretchValue SEMICONDENSED = new CSSStretchValue("semicondensed", 400);
		public static final CSSStretchValue NORMAL = new CSSStretchValue("normal", 500);
		public static final CSSStretchValue SEMIEXPANDED = new CSSStretchValue("semiexpanded", 600);
		public static final CSSStretchValue EXPANDED = new CSSStretchValue("expanded", 700);
		public static final CSSStretchValue EXTRAEXPANDED = new CSSStretchValue("extraexpanded", 800);
		public static final CSSStretchValue ULTRAEXPANDED = new CSSStretchValue("ultraexpanded", 900);

		private static final CSSStretchValue[] stretchValues = {
			ULTRACONDENSED, EXTRACONDENSED, CONDENSED, SEMICONDENSED, NORMAL,
			SEMIEXPANDED, EXPANDED, EXTRAEXPANDED, ULTRAEXPANDED
		};

		/* Serialization signature is explicitly set and should be 
		 * incremented on each release to prevent compatibility.
		 */
		static final long serialVersionUID = 1;

		private String name;
		private int value;

		private CSSStretchValue (String name, int value) {
			// this class provides an enum
			this.name = name;
			this.value = value;
		}

		// This is required to maintain the enum property through serialization
		private Object readResolve()
		{
			return getStretchValue(this.value);
		}

		/**
		 * Gets the internal numerical stretch value.
		 * @return the internal stretch numerical value
		 */
		int getValue()
		{
			return this.value;
		}

		/**
		 * Parse the text and return the CSSStretchValue object which is represented if any.
		 * 
		 * @param text textual representation of the stretch
		 * @return CSSStretchValue represented by the text value
		 */
		public static CSSStretchValue parse(String text)
		{
			for (int i = 0; i < stretchValues.length; i++)
			{
				if (stretchValues[i].name.equalsIgnoreCase(text))
				{
					return stretchValues[i];
				}
			}
			return null;
		}

		/**
		 * @param value the value of the stretch object to return
		 * @return CSSStretchValue object with the stretch specified or <code>null</code> if none exists
		 */
		public static CSSStretchValue getStretchValue(int value)
		{
			for (int i = 0; i < stretchValues.length; i++)
			{
				if (stretchValues[i].value == value)
				{
					return stretchValues[i];
				}
			}
			return null;
		}
		
		@Override
		public String toString() 
		{
			return this.name;
		}
	}

	/**
	 * This class provides an enumeration of the possible values for the CSS Style attribute.
	 * 
	 * <h4>Concurrency</h4>
	 * 
	 * Instances of this class are immutable after construction and contain no mutable static
	 * data. Therefore, they are threadsafe.
	 */
	public static final class CSSStyleValue implements Serializable
	{
		public static final CSSStyleValue NORMAL = new CSSStyleValue ("normal", 1);     
		public static final CSSStyleValue OBLIQUE = new CSSStyleValue ("oblique", 2);
		public static final CSSStyleValue ITALIC = new CSSStyleValue ("italic", 3);

		private static final CSSStyleValue[] styleValues = {
			NORMAL, OBLIQUE, ITALIC
		};

		/* Serialization signature is explicitly set and should be 
		 * incremented on each release to prevent compatibility.
		 */
		static final long serialVersionUID = 1;

		private String name;
		private int value;

		private CSSStyleValue (String name, int style) { 
			// this class provides an enum
			this.name = name;
			this.value = style;
		}

		// This is required to maintain the enum property through serialization
		private Object readResolve()
		{
			return getStyleValue(this.value);
		}

		/**
		 * Parse the text and return the CSSStyleValue object which is represented if any.
		 * 
		 * @param text textual representation of the style
		 * @return CSSStyleValue represented by the text value
		 */
		public static CSSStyleValue parse(String text)
		{
			for (int i = 0; i < styleValues.length; i++)
			{
				if (styleValues[i].name.equalsIgnoreCase(text))
				{
					return styleValues[i];
				}
			}
			return null;
		}

		/**
		 * @param value the value of the stretch object to return
		 * @return CSSStyleValue object with the style specified or <code>null</code> if none exists
		 */
		protected static CSSStyleValue getStyleValue(int value)
		{
			for (int i = 0; i < styleValues.length; i++)
			{
				if (styleValues[i].value == value)
				{
					return styleValues[i];
				}
			}
			return null;
		}
		
		@Override
		public String toString() 
		{
			return this.name;
		}
	}

	/**
	 * This class provides an enumeration of the possible values for the CSS Variant attribute.
	 * 
	 * <h4>Concurrency</h4>
	 * 
	 * Instances of this class are immutable after construction and contain no mutable static
	 * data. Therefore, they are threadsafe.
	 */
	public static final class CSSVariantValue implements Serializable
	{
		public static final CSSVariantValue NORMAL = new CSSVariantValue ("normal", 1);
		public static final CSSVariantValue SMALL_CAPS = new CSSVariantValue ("small-caps", 2);

		private static final CSSVariantValue[] variantValues = {
			NORMAL, SMALL_CAPS
		};

		/* Serialization signature is explicitly set and should be 
		 * incremented on each release to prevent compatibility.
		 */
		static final long serialVersionUID = 1;

		private String name;
		private int value;

		private CSSVariantValue (String name, int value) 
		{ 
			// this class provides an enum
			this.name = name;
			this.value = value;
		}

		// This is required to maintain the enum property through serialization
		private Object readResolve()
		{
			return getVariantValue(this.value);
		}

		/**
		 * Parse the text and return the CSSVariantValue object which is represented if any.
		 * 
		 * @param text textual representation of the variant
		 * @return CSSVariantValue represented by the text value
		 */
		public static CSSVariantValue parse(String text)
		{
			for (int i = 0; i < variantValues.length; i++)
			{
				if (variantValues[i].name.equalsIgnoreCase(text))
				{
					return variantValues[i];
				}
			}
			return null;
		}

		/**
		 * @param value the value of the stretch object to return
		 * @return CSSStretchValue object with the stretch specified or <code>null</code> if none exists
		 */
		protected static CSSVariantValue getVariantValue(int value)
		{
			for (int i = 0; i < variantValues.length; i++)
			{
				if (variantValues[i].value == value)
				{
					return variantValues[i];
				}
			}
			return null;
		}   
		
		@Override
		public String toString() 
		{
			return this.name;
		}
	}

	/**
	 * This class provides an enumeration of the possible values for the CSS Weight attribute.
	 * 
	 * <h4>Concurrency</h4>
	 * 
	 * Instances of this class are immutable after construction and contain no mutable static
	 * data. Therefore, they are threadsafe.
	 */
	public static final class CSSWeightValue implements Serializable
	{
		public static final CSSWeightValue W100 	= new CSSWeightValue(100);
		public static final CSSWeightValue W200 	= new CSSWeightValue(200);
		public static final CSSWeightValue W300 	= new CSSWeightValue(300);
		public static final CSSWeightValue W400 	= new CSSWeightValue(400);
		public static final CSSWeightValue W500 	= new CSSWeightValue(500);
		public static final CSSWeightValue W600 	= new CSSWeightValue(600);
		public static final CSSWeightValue W700 	= new CSSWeightValue(700);
		public static final CSSWeightValue W800 	= new CSSWeightValue(800);
		public static final CSSWeightValue W900 	= new CSSWeightValue(900);

		/**
		 * The weight value of <code>NORMAL</code> is identical to a weight value of
		 * 400 as per the CSS specifications. 
		 */
		public static final CSSWeightValue NORMAL 	= W400;

		/**
		 * The weight value of <code>BOLD</code> is identical to a weight value of
		 * 700 as per the CSS specifications. 
		 */
		public static final CSSWeightValue BOLD 	= W700;

		final static CSSWeightValue[] weightValues = {
			W100, W200, W300, W400, W500, W600, W700, W800, W900
		};

		/* Serialization signature is explicitly set and should be 
		 * incremented on each release to prevent compatibility.
		 */
		static final long serialVersionUID = 1;

		private int value;

		private CSSWeightValue (int value) 
		{ 
			// this class provides an enum
			this.value = value;
		}

		// This is required to maintain the enum property through serialization
		private Object readResolve()
		{
			return getWeightValue(this.value);
		}

		/**
		 * @param value the value of the stretch object to return
		 * @return CSSStretchValue object with the stretch specified or <code>null</code> if none exists
		 */
		protected static CSSWeightValue getWeightValue(int value)
		{
			for (int i = 0; i < weightValues.length; i++)
			{
				if (weightValues[i].value == value)
				{
					return weightValues[i];
				}
			}
			return null;
		}

		/**
		 * Return the numerical value of the weight.
		 * @return the numerical weight value
		 */
		public int getValue()
		{
			return this.value;
		}

		/**
		 * Parse the text and return the CSSWeightValue object which is represented if any.
		 * 
		 * @param text textual representation of the weight
		 * @return CSSWeightValue represented by the text value
		 */
		public static CSSWeightValue parse(String text)
		{
			if (text.equalsIgnoreCase("bold"))
			{
				return CSSWeightValue.BOLD;
			}
			if (text.equalsIgnoreCase("normal"))
			{
				return CSSWeightValue.NORMAL;
			}
			return CSSWeightValue.getWeightValue(Integer.parseInt(text));
		}

		@Override
		public String toString() 
		{
			return Integer.toString(this.value);
		}
	}
}
