/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.activitystreams;

import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ProviderType;

/**
 * <code>ActivityObject</code> specifies a general object use with activities.
 *
 * Note that property names and descriptions were reused from http://activitystrea.ms/specs/json/1.0/. Although that
 * document specifies how activities are serialized in JSON the names and concepts can be reused. Additional properties
 * and behaviors are added where the JSON abstraction is not suitable.
 */
@ProviderType
public interface ActivityObject extends JsonAdaptable {

    /**
     * Provides a permanent, universally unique identifier for the object in the form of an absolute IRI [RFC3987].
     * An object SHOULD contain a single id property. If an object does not contain an id property, consumers MAY use
     * the value of the url property as a less-reliable, non-unique identifier.
     */
    String getId();

    /**
     * An IRI [RFC3987] identifying a resource providing an HTML representation of the object.
     * An object MAY contain a url property
     */
    String getURL();

    /**
     * Identifies the type of object. An object MAY contain an objectType property whose value is a String that is
     * non-empty and matches either the "isegment-nz-nc" or the "IRI" production in [RFC3987].
     * Note that the use of a relative reference other than a simple name is not allowed.
     * If no objectType property is contained, the object has no specific type.
     */
    String getObjectType();

    /**
     * Natural-language description of the object encoded as a single String containing HTML markup.
     * Visual elements such as thumbnail images MAY be included. An object MAY contain a content property.
     */
    String getContent();

    /**
     * A natural-language, human-readable and plain-text name for the object. HTML markup MUST NOT be included. An
     * object MAY contain a displayName property. If the object does not specify an objectType property, the object
     * SHOULD specify a displayName.
     */
    String getDisplayName();

    /**
     * Natural-language summarization of the object encoded as a single String containing HTML markup.
     * Visual elements such as thumbnail images MAY be included. An activity MAY contain a summary property.
     */
    String getSummary();

    /**
     * Description of a resource providing a visual representation of the object, intended for human consumption.
     * An object MAY contain an image property whose value is a Media Link.
     */
    MediaLink getImage();

    /**
     * Describes the entity that created or authored the object. An object MAY contain a single author property whose
     * value is an Object of any type. Note that the author field identifies the entity that created the object and
     * does not necessarily identify the entity that published the object. For instance, it may be the case that an
     * object created by one person is posted and published to a system by an entirely different entity.
     *
     * Note that the author object might by dynamically generated based on the user id of the actual author, if present
     * in the system.
     */
    ActivityObject getAuthor();

    /**
     * Returns the user id of the author of this activity object or <code>null</code> if this object was not created
     * by a user present in this system.
     */
    String getAuthorUserId();

    /**
     * The date and time at which the object was published. An object MAY contain a published property.
     */
    long getPublished();

    /**
     * The date and time at which a previously published object has been modified. An Object MAY contain an updated property.
     */
    long getUpdated();

    /**
     * An array of one or more additional, associated objects, similar to the concept of attached files in an email
     * message. An object MAY have an attachments property.
     */
    ActivityObject[] getAttachments();

    /**
     * Am array of one or more absolute IRI's [RFC3987] identifying objects that duplicate this object's content.
     * An object SHOULD contain a downstreamDuplicates property when there are known objects, possibly in a different
     * system, that duplicate the content in this object. This MAY be used as a hint for consumers to use when
     * resolving duplicates between objects received from different sources.
     */
    String[] getDownstreamDuplicates();

    /**
     * An array of one or more absolute IRI's [RFC3987] identifying objects that duplicate this object's content.
     * An object SHOULD contain an upstreamDuplicates property when a publisher is knowingly duplicating with a new ID
     * the content from another object. This MAY be used as a hint for consumers to use when resolving duplicates
     * between objects received from different sources.
     */
    String[] getUpstreamDuplicates();

    /**
     * Returns a read-only value map of the raw properties underlying this activity object.
     */
    ValueMap getProperties();

    /**
     * Returns a copy of this activity object that is mutable. If this instance is already mutable <code>this</code> is
     * returned.
     * @return a mutable copy of this activity object.
     */
    MutableActivityObject getMutableActivityObject();

}