/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.rest.converter;

import java.util.Collections;
import java.util.Map;

import org.apache.sling.api.request.RequestPathInfo;
import org.apache.sling.api.resource.Resource;

import com.adobe.granite.rest.filter.Filter;

/**
 * The {@code ResourceConverterContext} provides context information which are necessary or useful for converting a
 * {@link Resource resource}.
 */
public class ResourceConverterContext {

    /**
     * Default page size limit.
     */
    public static final int DEFAULT_LIMIT = 20;

    /**
     * Default indicator if absolute URIs should be used.
     */
    public static final boolean DEFAULT_USE_ABSOLUTE_URIS = true;

    /**
     * Default show all properties flag.
     */
    public static final boolean DEFAULT_SHOW_ALL_PROPERTIES = false;

    private int offset = 0;

    private int limit = -1;

    private Filter<Resource> filter;

    private String scheme;

    private String serverName;

    private int serverPort;

    private String contextPath;

    private RequestPathInfo requestPathInfo;

    private Map<String, String[]> parameters;

    private String query;

    private boolean useAbsoluteURI = DEFAULT_USE_ABSOLUTE_URIS;

    private boolean showAllProperties = DEFAULT_SHOW_ALL_PROPERTIES;

    private String[] showProperties;

    private String rootContextPath;

    private Map<String, String[]> modelFilters;

    /**
     * Create a new {@link ResourceConverterContext}.
     */
    public ResourceConverterContext() {
    }

    /**
     * Get the Root Context Path Value as defined in OSGI config for ApiEndpointResourceProviderFactory
     * @return The root context path.
     */
    public String getRootContextPath() {
        return rootContextPath;
    }

    /**
     * Set the root context path
     * @param rootContextPath The root context path
     */
    public void setRootContextPath(String rootContextPath) {
        this.rootContextPath = rootContextPath;
    }

    /**
     * Returns the offset.
     * @return The offset value
     */
    public int getOffset() {
        return offset;
    }

    /**
     * Set the offset.
     * @param offset Offset value
     * @return reference to self
     */
    public ResourceConverterContext setOffset(int offset) {
        this.offset = offset;
        return this;
    }

    /**
     * Returns the limit.
     * @return The limit value
     */
    public int getLimit() {
        return limit;
    }

    /**
     * Sets the limit.
     * @param limit Limit value
     * @return reference to self
     */
    public ResourceConverterContext setLimit(int limit) {
        this.limit = limit;
        return this;
    }

    /**
     * Returns the filter.
     * @return The filter
     */
    public Filter<Resource> getFilter() {
        return filter;
    }

    /**
     * Sets the filter.
     * @param filter A filter
     * @return reference to self
     */
    public ResourceConverterContext setFilter(Filter<Resource> filter) {
        this.filter = filter;
        return this;
    }

    /**
     * Returns the name of the scheme.
     * @return Scheme name
     */
    public String getScheme() {
        return scheme;
    }

    /**
     * Sets the name of the scheme.
     * @param scheme Scheme name
     * @return reference to self
     */
    public ResourceConverterContext setScheme(String scheme) {
        this.scheme = scheme;
        return this;
    }

    /**
     * Returns the host name of the server.
     * @return Server host name
     */
    public String getServerName() {
        return serverName;
    }

    /**
     * Sets the host name of the server.
     * @param serverName Server host name
     * @return reference to self
     */
    public ResourceConverterContext setServerName(String serverName) {
        this.serverName = serverName;
        return this;
    }

    /**
     * Returns the port number.
     * @return Server port
     */
    public int getServerPort() {
        return serverPort;
    }

    /**
     * Sets the port number.
     * @param serverPort Server port
     * @return reference to self
     */
    public ResourceConverterContext setServerPort(int serverPort) {
        this.serverPort = serverPort;
        return this;
    }

    /**
     * Returns the portion of the request URI that indicates the context.
     * @return Context path
     */
    public String getContextPath() {
        return contextPath;
    }

    /**
     * Sets the portion of the request URI that indicates the context.
     * @param contextPath Context path
     * @return reference to self
     */
    public ResourceConverterContext setContextPath(String contextPath) {
        this.contextPath = contextPath;
        return this;
    }

    /**
     * Returns the {@link org.apache.sling.api.request.RequestPathInfo}.
     * @return RequestPathInfo
     */
    public RequestPathInfo getRequestPathInfo() {
        return requestPathInfo;
    }

    /**
     * Sets the {@link org.apache.sling.api.request.RequestPathInfo}.
     * @param requestPathInfo A RequestPathInfo
     * @return reference to self
     */
    public ResourceConverterContext setRequestPathInfo(RequestPathInfo requestPathInfo) {
        this.requestPathInfo = requestPathInfo;
        return this;
    }

    /**
     * Returns the request parameters.
     * @return A {@link java.util.Map} of request parameter keys and values
     */
    public Map<String, String[]> getParameters() {
        return parameters;
    }

    /**
     * Sets the request parameters.
     * @param parameters Request parameters
     * @return reference to self
     */
    public ResourceConverterContext setParameters(Map<String, String[]> parameters) {
        this.parameters = parameters;

        return this;
    }

    /**
     * Returns the query.
     * @return Query value
     */
    public String getQuery() {
        return query;
    }

    /**
     * Sets the query.
     * @param query Query value
     * @return reference to self
     */
    public ResourceConverterContext setQuery(String query) {
        this.query = query;
        return this;
    }

    /**
     * Indicator if absolute URI's are used.
     * @return {@code true} if URIs are absolute, {@code false} otherwise
     */
    public boolean isAbsolutURI() {
        return useAbsoluteURI;
    }

    /**
     * Sets the property to use absolute URI's.
     * @param useAbsoluteURI Indicator to use absolute URIs
     * @return reference to self
     */
    public ResourceConverterContext setAbsolutURI(boolean useAbsoluteURI) {
        this.useAbsoluteURI = useAbsoluteURI;
        return this;
    }

    /**
     * Indicator if all properties should be shown.
     * @return the showAllProperties
     */
    public boolean isShowAllProperties() {
        return showAllProperties;
    }

    /**
     * Sets the property to show all properties.
     * @param showAllProperties Indicator to show all properties
     * @return reference to self
     */
    public ResourceConverterContext setShowAllProperties(boolean showAllProperties) {
        this.showAllProperties = showAllProperties;
        return this;
    }

    /**
     * Retrieves the properties to show.
     * @return A list of property names to show.
     */
    public String[] getShowProperties() {
        return this.showProperties == null ? new String[0] : this.showProperties;
    }

    /**
     * Sets the properties to show.
     * @param showProperties Property names to show.
     * @return references to self
     */
    public ResourceConverterContext setShowProperties(String[] showProperties) {
        this.showProperties = showProperties;
        return this;
    }

    public Map<String, String[]> getModelFilters() {
        if (this.modelFilters == null) {
            return Collections.<String, String[]>emptyMap();
        }
        return this.modelFilters;
    }

    public void setModelFilters(Map<String, String[]> filters) {
        this.modelFilters = Collections.unmodifiableMap(filters);
    }
}
