/* ****************************************************************************
 *
 *	File: CosToken.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.SoftReference;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException.CosParseErrorType;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASObject;
import com.adobe.internal.pdftoolkit.core.types.ASString;
import com.adobe.internal.pdftoolkit.core.util.BooleanHolder;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;
import com.adobe.internal.pdftoolkit.core.util.ParseOps;

/**
 * The PDF parser.
 *
 */
public final class CosToken
{
	
	static final long DEFAULT_NEXT_OBJ_POS = Long.MAX_VALUE;
	private static final int INITIAL_NAME_BUFFER_SIZE = 1024;
	private static ThreadLocal<SoftReference<byte[]>> tlBuffer = new ThreadLocal<SoftReference<byte[]>>(){
		@Override
		protected synchronized SoftReference<byte[]> initialValue() {
			return new SoftReference<byte[]>(new byte[INITIAL_NAME_BUFFER_SIZE]);
		}
	};
	
	//refer to table C.1 ï¿½ Architectural limits of ISO 32000-1
	public static final BigDecimal MIN_FLOAT_APPROACHING_TO_ZERO = BigDecimal.valueOf(1.175E-38); 
	public static final BigDecimal MAX_REAL = BigDecimal.valueOf(Float.MAX_VALUE);
	/**
	 * The PDF parser. The ByteArrayUtility class provides additional
	 * parsing support.
	 *
	 * @see ByteArrayUtility
	 *
	 */
	private CosToken(){}

	/**
	 * Reads an object id and generation from the ByteStream. Throws ParseException
	 * if the ByteStream is not positioned at num num 'obj'. This used to be part of
	 * readIndirectObject, but is needed separately for object stream implementation
	 * @param buf
	 * @throws PDFCosParseException
	 */
	static int[] readObjID(CosDocument doc, InputByteStream buf, byte b)
		throws PDFCosParseException, IOException
	{
		int[] values = new int[2];
		values[0] = (readNumber(buf, b)).intValue();
		values[1] = (readNumber(buf, skipWhitespace(buf))).intValue();
		if (!(skipWhitespace(buf) == 'o' && buf.read() == 'b' && buf.read() == 'j')) {
			throw new PDFCosParseException("Expected 'obj' : " + Long.toString(buf.getPosition() - 1));
		}
		return values;
	}

	/**
	 * Skips an object id from PDF byte stream.
	 * @param doc
	 * @param buf
	 * @param b
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	static void skipObjID(CosDocument doc, InputByteStream buf, byte b)
		throws PDFCosParseException, IOException
	{
		while (buf.read() != 'o');
		if (!(buf.read() == 'b' && buf.read() == 'j')) {
			throw new PDFCosParseException("Expected 'obj' : " + Long.toString(buf.getPosition() - 1));
		}
	}

	/**
	 * Parses an indirect object and returns the underlying object.
	 * This recognizes and returns CosStream and CosObjectStream objects.
	 * Indirect objects are defined in section 3.2.9 of the PDF Reference
	 * Manual version 1.4.
	 *
	 * @param doc			Document containing the indirect object
	 * @param buf			PDF data buffer to parse
	 * @param info			Info of the object reference
	 *
	 * @return Underlying COS object.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	static CosObject readIndirectObject(CosDocument doc, InputByteStream buf, CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		CosObject retVal = null;
		try {
			long nextObjPos = info != null && info.getNextObjPos() != Long.MAX_VALUE? (buf.getPosition() + (info.getNextObjPos() - info.getPos())) : CosToken.DEFAULT_NEXT_OBJ_POS;
			retVal = readObject(doc, buf, skipWhitespace(buf), info, null, true, nextObjPos);
		} finally {
			if (retVal == null && info != null)
				info.clearObject();
		}
		return retVal;
	}

	/**
	 * Parses a COS object from the PDF byte stream.
	 *
	 * @param doc		Document containing the object
	 * @param buf		PDF data buffer to parse
	 * @param info		Info to assign to the parsed COS object
	 *
	 * @return Newly allocated COS object parsed from the buffer.
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	static CosObject readObject(CosDocument doc, InputByteStream buf, CosObjectInfo info)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		long nextObjPos = info != null && info.getNextObjPos() != Long.MAX_VALUE? (buf.getPosition() + (info.getNextObjPos() - info.getPos())) : CosToken.DEFAULT_NEXT_OBJ_POS;
		return readObject(doc, buf, skipWhitespace(buf), info, null, false, nextObjPos);
	}
	
	/**
	 * Skips next a COS object from the PDF byte stream.
	 *
	 * @param doc		Document containing the object
	 * @param buf		PDF data buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	static void skipObject(CosDocument doc, InputByteStream buf)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		skipObject(doc, buf, skipWhitespace(buf));
	}

	/**
	 * Parses a PDF array from the PDF byte stream and returns a list of
	 * the elements in the array. Typically, the list returned by this method
	 * would be passed to the CosArray constructor.
	 *
	 * @param doc		Document containing the array
	 * @param buf		PDF data buffer to parse
	 *
	 * @return List of elements in the array.
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws IOException
	 * @throws PDFIOException
	 */
	private static ArrayList readArray(CosDocument doc, InputByteStream buf, long nextObjPos)
		throws PDFCosParseException, PDFSecurityException, IOException, PDFIOException
	{
		return readObjectList(doc, buf, (byte)']', nextObjPos);
	}
	
	/**
	 * Parses a PDF array from the PDF byte stream and skips a list of
	 * the elements in the array.
	 *
	 * @param doc		Document containing the array
	 * @param buf		PDF data buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws IOException
	 * @throws PDFIOException
	 */
	private static void skipArray(CosDocument doc, InputByteStream buf)
		throws PDFCosParseException, PDFSecurityException, IOException, PDFIOException
	{
		skipObjectList(doc, buf, (byte)']');
	}

	/**
	 * Parses a PDF dictionary from the PDF byte stream and returns a map of
	 * the entries in the dictionary. Typically, the map returned by this
	 * method would be passed to the COSDictionary constructor.
	 *
	 * @param doc		Document containing the dictionary
	 * @param buf		PDF data buffer to parse	
	 *
	 * @return Map of dictionary entries
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private static Map readDictionary(CosDocument doc, InputByteStream buf, long nextObjPos)
		throws PDFCosParseException, IOException, IOException, PDFSecurityException, PDFIOException
	{
		// Parse the dictionary entries into a list consisting
		// of key followed by value followed by key, etc.
		ArrayList<CosObject> content = readObjectList(doc, buf, (byte)'>', nextObjPos);
		if(content == null)
			return null;
		byte b = (byte)buf.read();
		if (b != (byte)'>')
			throw new PDFCosParseException("Expected '>' but got '" + b + "'.");

		// Load a map with the dictionary entries by iterating over
		// the list.
		LinkedHashMap rslt = new LinkedHashMap();
		Iterator<CosObject> contentIter = content.iterator();
		while (contentIter.hasNext()) {
			CosObject cosName = contentIter.next();
			if (cosName instanceof CosName) {
				ASName key = ((CosName)cosName).nameValue();
				Object value = contentIter.next();
				if (!(value instanceof CosNull)) {
					rslt.put(key, value);
				}
			}	
			else
			{
				rslt.put(null, cosName);
			}
		}
		return rslt;
	}
	
	/**
	 * Parses a PDF dictionary from the PDF byte stream and skips a map of
	 * the entries in the dictionary.
	 *
	 * @param doc		Document containing the dictionary
	 * @param buf		PDF data buffer to parse	
	 *
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private static void skipDictionary(CosDocument doc, InputByteStream buf)
		throws PDFCosParseException, IOException, IOException, PDFSecurityException, PDFIOException
	{
		// Parse the dictionary entries into a list consisting
		// of key followed by value followed by key, etc.
		skipObjectList(doc, buf, (byte)'>');
		buf.read();
	}

	/**
	 * Parse a hexidecimal digit from the PDF byte stream.
	 *
	 * @param buf		PDF data buffer to parse
	 *
	 * @return A single hexidecimal digit.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static byte readHexDigit(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		return toHexDigit((byte)buf.read());
	}

	private static byte[] readHexString(InputByteStream buf, boolean[] oddBall)
		throws PDFCosParseException, IOException
	{
		int bufSize = 16;
		int digitPosition = 0;
		byte[] bytes = new byte[bufSize];
		for (byte b = (byte)buf.read(); b != InputByteStream.EOF; b = (byte)buf.read()) {
			while (b != InputByteStream.EOF && ByteOps.isWhitespace(b))
				b = (byte)buf.read();
			if (b == '>' || b == InputByteStream.EOF)
				break;
			int h = CosToken.toHexDigit(b);
			b = (byte)buf.read();
			while (b != InputByteStream.EOF && ByteOps.isWhitespace(b))
				b = (byte)buf.read();
			int l = 0;
			if (b != '>' && b != InputByteStream.EOF)
				l = CosToken.toHexDigit(b);
			if (digitPosition >= bufSize) {
				int newBufSize = 4096;
				if (newBufSize <= bufSize)
					newBufSize = bufSize * 2;
				byte[] newBytes = new byte[newBufSize];
				System.arraycopy(bytes, 0, newBytes, 0, bufSize);
				bytes = newBytes;
				bufSize = newBufSize;
			}
			bytes[digitPosition++] = (byte)(h * 16 + l);
			if (b == '>' || b == InputByteStream.EOF) {
				oddBall[0] = true;
				break;
			}

		}
		if (digitPosition < bytes.length) {
			byte[] copyBytes = new byte[digitPosition];
			System.arraycopy(bytes, 0, copyBytes, 0, digitPosition);
			bytes = copyBytes;
		}
		return bytes;
	}
	
	/**
	 * Skips a hexstring from PDF byte stream
	 * @param buf
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void skipHexString(InputByteStream buf)
	throws PDFCosParseException, IOException
	{
		for (byte b = (byte)buf.read(); b != InputByteStream.EOF; b = (byte)buf.read()) {
			while (ByteOps.isWhitespace(b))
				b = (byte)buf.read();
			if (b == '>')
				break;
			b = (byte)buf.read();
			if (b == '>') {
				break;
			}
		}
	}

	/**
	 * Converts the specified hexidecimal character to a byte value. For
	 * example, the hex character 'A' is converted to the value 10.
	 *
	 * @param b		Hex character to convert
	 * @return Byte value corresponding to the specified hex character.
	 * @throws PDFCosParseException
	 */
	public static byte toHexDigit(byte b)
		throws PDFCosParseException
	{
		int rslt = 0;
		if (b >= '0' && b <= '9')
			rslt = (b - '0');
		else if (b >= 'A' && b <= 'F')
			rslt = (b - 'A') + 10;
		else if (b >= 'a' && b <= 'f')
			rslt = (b - 'a') + 10;
		else
			throw new PDFCosParseException("Expected Hex Digit - instead got " + Byte.toString(b));
		return (byte)rslt;
	}

	/**
	 * Parses an atom from the PDF byte stream. An atom can be thought
	 * of as a COS name as defined in section 3.2.4 of the PDF Reference
	 * Manual version 1.4.
	 *
	 * @param buf			PDF data buffer to parse
	 *
	 * @return ASName parsed from the buffer
	 *
	 *
	 * TODO: In PDF 1.1 the "#" was not an escape character.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static ASName readName(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		byte[] readBuffer = tlBuffer.get().get();
		if(readBuffer == null){
			tlBuffer.remove();
			// will be automatically re-initialized here.
			readBuffer = tlBuffer.get().get();
		}
		int end = 0;
		int cur = buf.read();
		while (cur != InputByteStream.EOF && ByteOps.isRegular((byte)cur)) {
			if(end == readBuffer.length){
				// buffer reached its maximum capacity.
				byte[] newBuffer = new byte[readBuffer.length*2];
				System.arraycopy(readBuffer, 0, newBuffer, 0, readBuffer.length);
				readBuffer = newBuffer;
				tlBuffer.set(new SoftReference<byte[]>(readBuffer));
			}
			// Handle escaped names.
			if (cur == '#') {
				int next = buf.read();
				buf.unget();
				if (ByteOps.isHexDigit((byte)next))
					cur = readHexDigit(buf) * 16 + readHexDigit(buf);
			}

			// Build up the name.
			readBuffer[end++] = (byte)cur;
			cur = buf.read();
		}
		if(cur != InputByteStream.EOF)
			buf.unget();

		// Create the atom if it does not already exist.
		return ASName.getName(readBuffer, 0, end);
	}
	
	/**
	 * Skips an atom from the PDF byte stream. An atom can be thought
	 * of as a COS name as defined in section 3.2.4 of the PDF Reference
	 * Manual version 1.4.
	 *
	 * @param buf			PDF data buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void skipName(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		int cur = buf.read();
		while (cur != InputByteStream.EOF && ByteOps.isRegular((byte)cur)) {
			// Handle escaped names.
			if (cur == '#') {
				int next = buf.read();
				buf.unget();
				if (ByteOps.isHexDigit((byte)next))
					cur = readHexDigit(buf) * 16 + readHexDigit(buf);
			}
			cur = buf.read();
		}
		if(cur != InputByteStream.EOF)
			buf.unget();

		// Return as the asname has been skiped
		return;
	}

	/**
	 * Parses a numerical value from the PDF byte stream. A numeric object
	 * is defined in section section 7.3.3 of ISO 32000-1.
	 *
	 * @param buf		Byte stream to parse
	 * @param b		Starting byte to parse
	 *
	 * @return Parsed number.
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	static Number readNumber(InputByteStream buf, byte b)
		throws PDFCosParseException, IOException
	{
		int sign = 1;
		long left = 0;
		long right = 0;
		long divisor = 1;
		boolean decimal = false;
		Number rslt;
		long initPos = buf.getPosition();
		byte initByte = b;
		long numberOfDigitsBeforeDecimal = 1;// number of digits before decimal excluding leading zeros.
		if (b == '-') {
			sign = -1;
			numberOfDigitsBeforeDecimal = 0;
			b = (byte)buf.read();
		}
		else if (b == '+') {
			sign = 1;
			numberOfDigitsBeforeDecimal = 0;
			b = (byte)buf.read();
		}
		else if (b == '0'){
			numberOfDigitsBeforeDecimal = 0;
			b = (byte)buf.read();
		}
		
		while (ByteOps.isDigit(b) && b == '0') {// skipping leading zeros.
			initPos++;
			b = (byte)buf.read();
		}
		while (ByteOps.isDigit(b)) {
			left *= 10;
			left += b - '0';
			b = (byte)buf.read();
		}
		if (b == '.') {
			decimal = true;
			//Annex C.2 in ISO 32000-1 restricts the range of integer in PDF from -2^31 to 2^31 -1 which comes under the range of java long.
			//but the range of real can be ï¿½3.403 ï¿½ 10^38 which is very much out of the range of long which is up to 2^63 -1.
			// so if there is a decimal present (which represents a real number) then we should be careful.
			numberOfDigitsBeforeDecimal += (buf.getPosition() - initPos) - 1;
			int dataLengthToParse = (int) (buf.getPosition() - initPos);// there should not be any problem with this cast because biggest real number allowed is
																		// 3.403 ï¿½ 10^38 which shall have 39 digits.
			long numberOfDigitsAfterDecimal = 0;// number of digits after decimal excluding trailing zeros.
			b = (byte)buf.read();
			
			while (ByteOps.isDigit(b) && numberOfDigitsAfterDecimal < 16) {/// iterate over the 16 digits after decimal. 
				right *= 10;
				right += b - '0';
				divisor *= 10;
				numberOfDigitsAfterDecimal++;
				b = (byte)buf.read();
			}
			if(numberOfDigitsAfterDecimal == 16){//// we have 16 digits after decimal for sure. Let's find out here if we have more non-zero digits after that.
				while(ByteOps.isDigit(b) && b == '0'){//skipping all zeros.
					numberOfDigitsAfterDecimal++;
					b = (byte)buf.read();
				}
				if(ByteOps.isDigit(b)){///after skipping if we get a non-zero digit then count them also.
					while(ByteOps.isDigit(b)){
						numberOfDigitsAfterDecimal++;
						b = (byte)buf.read();
					}
				}else
					numberOfDigitsAfterDecimal = 16;// if no non-zero digit after skipping zeros then let's stick to 16 digits.
			}
			if (ByteOps.isRegular(b))// if last digit is a regular byte then it's problem.
				throw new PDFCosParseException("Unexpected character at position " + Long.toString(buf.getPosition() - 1));
			buf.unget();
			
			if(numberOfDigitsBeforeDecimal > 16 || numberOfDigitsAfterDecimal > 16){///if it's a big real number or very small real number 
																				///then let's java handle it.
				InputByteStream slicedByteStream = null;
				InputStream inputStream = null;
				try {
					dataLengthToParse += numberOfDigitsAfterDecimal;
					byte[] data = new byte[dataLengthToParse + 1];
					slicedByteStream = buf.slice(initPos, dataLengthToParse);
					inputStream = slicedByteStream.toInputStream();
					inputStream.read(data, 1, dataLengthToParse);
					data[0] = initByte;
					BigDecimal bigNumber = new BigDecimal(new String(data));
					// checking for architectural limits of PDF reference 1.7
					if(bigNumber != null && (bigNumber.compareTo(MAX_REAL) == 1 || bigNumber.compareTo(MAX_REAL.negate()) == -1)){
							PDFCosParseException exceptionOccured = new PDFCosParseException("Number in document is out of the range ï¿½3.403 ï¿½ 10^38");
							exceptionOccured.addErrorType(CosParseErrorType.NumberParseError);
							throw exceptionOccured;
					}
					if (bigNumber != null && bigNumber.compareTo(MIN_FLOAT_APPROACHING_TO_ZERO) == -1 && 
							bigNumber.compareTo(MIN_FLOAT_APPROACHING_TO_ZERO.negate()) == 1)
						return new Double(sign * 0.0);
					return Double.valueOf(bigNumber.doubleValue());
				}finally{
					try{
						if(inputStream != null) inputStream.close();
					}finally{
						if(slicedByteStream != null) slicedByteStream.close();
					}
				}
			}
		}
		if (!decimal) {
			if ((sign == 1 && left <= Integer.MAX_VALUE) || (sign == -1 && left <= -1L*Integer.MIN_VALUE))
				rslt = Integer.valueOf((int)left * sign);
			else {
				PDFCosParseException exceptionOccured = new PDFCosParseException("Number in document is either greater than 2,147,483,647 OR less than -2,147,483,648");
				exceptionOccured.addErrorType(CosParseErrorType.NumberParseError);
				throw exceptionOccured;
			}
		} else
			rslt = new Double((left + ((double) right) / divisor) * sign);
		if (ByteOps.isRegular(b))
			throw new PDFCosParseException("Unexpected character at position " + Long.toString(buf.getPosition() - 1));
		buf.unget();
		return rslt;
	}

	/**
	 * 
	 * Parses a numerical value from the PDF byte stream. A numeric object
	 * is defined in section 7.3.3 of ISO 32000-1.
	 *
	 * @param buf		Byte array to parse
	 *
	 * @return Parsed number.
	 * @throws PDFCosParseException
	 */
	static Number readNumber(byte[] buf, BooleanHolder wasRepaired)
		throws PDFCosParseException
	{
		return readNumber(buf, false, wasRepaired);
	}
	private static Number readNumber(byte[] buf, boolean isExponent, BooleanHolder wasRepaired)
			throws PDFCosParseException
		{
		int index = 0;
		if (index == buf.length)
			throw new PDFCosParseException("Empty number string");
		byte b = buf[index++];
		int sign = 1;
		long left = 0;
		long right = 0;
		long divisor = 1;
		boolean decimal = false;
		long numberOfDigitsBeforeDecimal = 0;// number of digits before decimal excluding leading zeros.
		Number rslt;
		if (b == '-') {
			sign = -1;
			if (index == buf.length)
				throw new PDFCosParseException("Non-numeric number string");
			b = buf[index++];
		} else if (b == '+') {
			sign = 1;
			if (index == buf.length)
				throw new PDFCosParseException("Non-numeric number string");
			b = buf[index++];
		}
		while (ByteOps.isDigit(b) && b == '0') {/// skipping all leading zeros.
			if (index == buf.length) {
				b = ' ';
				break;
			}
			b = buf[index++];
		}
		while (ByteOps.isDigit(b)) {
			left *= 10;
			left += b - '0';
			numberOfDigitsBeforeDecimal++;
			if (index == buf.length) {
				b = ' ';
				break;
			}
			b = buf[index++];
		}
		if (b == '.') {
			//Annex C.2 in ISO 32000-1 restricts the range of integer in PDF from -2^31 to 2^31 -1 which comes under the range of java long.
			//but the range of real can be ï¿½3.403 ï¿½ 10^38 which is very much out of the range of long which is up to 2^63 -1.
			// so if there is a decimal present (which represents a real number) then we should be careful.
			decimal = true;
			long numberOfDigitsAfterDecimal = 0;// number of digits after decimal excluding trailing zeros.
			b = ' ';// this is to handle the case when real number if of format "xxxxxxxx." means no digits after decimal
			
			int indexOfLastTrailingZero = buf.length - 1;
			if (index < buf.length){
				b = buf[indexOfLastTrailingZero];
				while (b == 0) {// skipping all trailing null bytes. 
					b = buf[--indexOfLastTrailingZero];
				}
				while (ByteOps.isDigit(b) && b == '0') {// skipping all trailing zeros after decimal. 
					b = buf[--indexOfLastTrailingZero];
				}

				b = buf[index++];
				while (ByteOps.isDigit(b)) {
					right *= 10;
					right += b - '0';
					divisor *= 10;
					numberOfDigitsAfterDecimal++;
					if (index > indexOfLastTrailingZero) {
						b = ' ';
						break;
					}
					b = buf[index++];
				}
			}
			if (numberOfDigitsAfterDecimal == 0 && numberOfDigitsBeforeDecimal == 0 && !ByteOps.isDigit(b) && ByteOps.isRegular(b))
				throw new PDFCosParseException("Non-numeric number string");
			
			//Annex C.2 in ISO 32000-1 restricts the range of integer in PDF from -2^31 to 2^31 -1 which comes under the range of java long.
			//but the range of real can be ï¿½3.403 ï¿½ 10^38 which is very much out of the range of long which is up to 2^63 -1.
			// so if there is a decimal present (which represents a real number) then we should be careful.
			if(numberOfDigitsBeforeDecimal > 16 || numberOfDigitsAfterDecimal > 16){///if it's a big real number or very small number then let's java handle it.
				
				BigDecimal bigNumber = new BigDecimal(new String(buf).substring(0, indexOfLastTrailingZero));
				// checking for architectural limits of PDF reference 1.7
				if(bigNumber != null && (bigNumber.compareTo(MAX_REAL) == 1 || bigNumber.compareTo(MAX_REAL.negate()) == -1)){
					PDFCosParseException exceptionOccured = new PDFCosParseException("Number in document is out of the range ï¿½3.403 ï¿½ 10^38");
					exceptionOccured.addErrorType(CosParseErrorType.NumberParseError);
					throw exceptionOccured;
				}
				if (bigNumber != null && bigNumber.compareTo(MIN_FLOAT_APPROACHING_TO_ZERO) == -1 && 
						bigNumber.compareTo(MIN_FLOAT_APPROACHING_TO_ZERO.negate()) == 1)
					return new Double(sign * 0.0);
				return Double.valueOf(bigNumber.doubleValue());
			}
		}
		if (!decimal) {
			if ((sign == 1 && left <= Integer.MAX_VALUE) || (sign == -1 && left <= -1L*Integer.MIN_VALUE))
				rslt = Integer.valueOf((int)left * sign);
			else {
				PDFCosParseException exceptionOccured = new PDFCosParseException("Number in document is either greater than 2,147,483,647 OR less than -2,147,483,648");
				exceptionOccured.addErrorType(CosParseErrorType.NumberParseError);
				throw exceptionOccured;
			}
		} else
			rslt = new Double((left + ((double) right) / divisor) * sign);
		if(!isExponent && b == (byte)'e' && wasRepaired != null){
			// the number is in exponential form like 4567e+10.
			// PDF reference doesn't allow such format but we are supporting that to fix bug #3296757.
			// However, we shall mark the document as repaired if we ultimately succeed to get the in-range number. 
			byte[] bufCopy = new byte[buf.length - index];
			System.arraycopy(buf, index, bufCopy, 0, bufCopy.length);
			Number pow = null;
			try{
				pow = readNumber(bufCopy, true, wasRepaired);// get the number after exponential which shall be raised as power to 10.
			}catch(Exception e){
				throw new PDFCosParseException("Non-numeric number string");
			}
			rslt = rslt.doubleValue() * Math.pow(10.0, pow.doubleValue());
			double rsltDouble = rslt.doubleValue();
			if(rsltDouble != rslt.intValue()){// It's an integer so check for integer range allowed.
				if(rsltDouble > MAX_REAL.doubleValue() || rsltDouble < MAX_REAL.negate().doubleValue()){
					throw new PDFCosParseException("Non-numeric number string");
				}
			}else{// it's real number so check for real number range allowed.
				if(rsltDouble > Integer.MAX_VALUE || rsltDouble < Integer.MIN_VALUE){
					throw new PDFCosParseException("Non-numeric number string");
				}
			}
			wasRepaired.setValue(true);// mark the cosobject and document as repaired.
			b = buf[buf.length-1];
		}
		if (!ByteOps.isDigit(b) && ByteOps.isRegular(b))
			throw new PDFCosParseException("Non-numeric number string");
		return rslt;
	}

	/**
	 * Parses a single line of a PDF file. A line is consider to be terminated
	 * by a '\r', '\n', or '\r\n' sequence.
	 *
	 * @param buf		    Byte stream to parse
	 * @param breakOnSpace  if <code>true</code> then a space would also mean line end.
	 *
	 * @return String containing the parsed line without the line termination character(s).
	 * @throws IOException
	 */
	static String readLine(InputByteStream buf, boolean breakOnSpace)
			throws IOException
	{
		StringBuilder rslt = new StringBuilder();
		int b = (byte)buf.read();
		while (b != '\r' && b != '\n') {
			if(breakOnSpace && b == ' ')
			{ 
				break;
			}
			rslt.append((char) b);
			b = buf.read();
		}
		if (b == '\r' && buf.read() != '\n')
			buf.unget();
		return rslt.toString();
	}
	
	/**
	 * 
	 * Skips over whitespace in the input stream. Whitespace is defined
	 * in section 3.1.1 of the PDF Reference Manual version 1.4. For the
	 * purposes of this method, comments are also considered whitespace.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @return The first non-whitespace byte encountered.
	 * @throws IOException
	 * @throws PDFCosParseException 
	 */
	public static byte skipWhitespace(InputByteStream buf)
		throws IOException, PDFCosParseException
	{
		while (true) {
			byte b = (byte)buf.read();
			if (!ByteOps.isWhitespace(b)) {
				if (b == (byte)'%') {
					while (b != (byte)'\r' && b != (byte)'\n') {
						b = (byte)buf.read();
						if (b == -1 && buf.eof())
							throw new PDFCosParseException("Read past EOF");
					}
				} else {
					return b;
				}
			}
		}
	}

	/**
	 * Reads digits from stream and copies them to byte array starting from specified offset.
	 * Returns the current offset of byte array.
	 * @param output
	 * @param buf
	 * @param b
	 * @param offset
	 * @return int
	 * @throws IOException
	 */
	private static int readDigits(byte[] output, InputByteStream buf, byte b, int offset) throws IOException{
		do {
			output[offset++] = b;
			b = (byte)buf.read();
		} while ((offset<output.length) && (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.'));
		if(b != InputByteStream.EOF)
			buf.unget();
		if(ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.')
		{
			do{
				b = (byte)buf.read();			
			}while(ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.');
			if(b != InputByteStream.EOF)
				buf.unget();
		}
		return offset;
	}
	
	/**
	 * This is the core COS object creation method. Parses the byte buffer
	 * to determine the type of COS object to create and then creates the
	 * appropriate COS object.
	 *
	 * @param doc		Document containg the COS object
	 * @param buf		Buffer to parse
	 * @param b		Starting byte
	 * @param info		Info for the created object
	 *
	 * @return Newly allocated COSObject
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFIOException
	 */
	private static CosObject readObject(CosDocument doc, InputByteStream buf, byte b, CosObjectInfo info, ArrayList parseArray, boolean doStms, long nextObjPos)
		throws PDFCosParseException, IOException, PDFSecurityException, IOException, PDFIOException
	{
		CosObject rslt = null;
		if (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.') {
			byte[] rep = new byte[64];
			int pos = readDigits(rep, buf, b, 0);
			b = (byte)buf.read();
			if(b == (byte)'e' && !buf.eof()){
				b = (byte)buf.read();// read further if this number is in exponential format.
				if (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-'){
					rep[pos++] = (byte)'e';
					readDigits(rep, buf, b, pos);
					b = (byte)buf.read();
				}else
					buf.unget();
			}
			if(b != InputByteStream.EOF)
				buf.unget();
			rslt = new CosNumeric(doc, rep, info);
		} else if (b == (byte)'R') {
			if (parseArray != null) {
				int generation = ((CosNumeric)parseArray.remove(parseArray.size() - 1)).intValue();
				int objectNumber = ((CosNumeric)parseArray.remove(parseArray.size() - 1)).intValue();
				CosLinearization cosLin = doc.getLinearization();
				if (cosLin != null) {
					objectNumber = cosLin.mapOldToNewObjNum(objectNumber);
					generation = 0;
				}
				// Watson#525559 The method getObjectInfo adds objects to cosList. If object number is huge addition to coslist
				// takes large memory and sometimes can result in Out of memory as well. So while parsing an cos object, ignore
				// it if its object number is large than largest object number defined by xref sections.
				// This check does not apply if the cosdoc is for FDFDocument or if we are still initialising Xref tables.
				if(doc.isFDF() || ! doc.getXRef().isXrefIntialized() || objectNumber < doc.getXRef().getNumObjectsDefinedInXRefEntries())
				{
					info = doc.getObjectInfo(objectNumber, generation);
				}					
				
				if (info != null)
					rslt = doc.getObjectRef(info);
				if (rslt == null)
					rslt = doc.createCosNull();
				
			} else {
				rslt = doc.createCosNull();
			}
		} else if (b == (byte)'/') {
			ASName name = readName(buf);
			rslt = new CosName(doc, name, info);
		} else if (b == (byte)'(') {
			byte[] str = ByteArrayUtility.readLiteral(doc, buf, nextObjPos);
			if(str == null)
				return null;
			rslt = new CosString(doc, str, 0, str.length, true, info);
		} else if (b == (byte)'<') {
			b = (byte)buf.read();
			if (b == (byte)'<') {
				Map map = readDictionary(doc, buf, nextObjPos);
				if(map == null){
					rslt = doc.createCosNull();// couldn't read the dictionary.
					if(info != null){
						rslt.setInfo(info);
						doc.setRepairedValue(info.getObjNum(), rslt);
					}
					return rslt;
				}
				if (doStms) {
					b = skipWhitespace(buf);
					// FIXME - replace with generic parsing code
					if (b == 's') {
						if (!(buf.read()=='t' && buf.read()=='r' && buf.read()=='e' && buf.read()=='a' && buf.read()=='m')) {
							throw new PDFCosParseException("Expected 'stream' at position " + Long.toString(buf.getPosition() - 1));
						}
						b = (byte)buf.read();	// see section 3.2.7 of pdf spec (we are handling extra cases here)
						while (b == ' ')
							b = (byte)buf.read();
						if (b == '\r') {
							b = (byte)buf.read();
							if (b != '\n') buf.unget();	// if \r but not \r\n
						} else if (b != '\n') buf.unget();	// if no \r or \n
						boolean ostream = (map.containsKey(ASName.k_Type) && (((CosName)map.get(ASName.k_Type)).nameValue() == ASName.k_ObjStm));
						long pos = buf.getPosition();
						if (ostream && info != null)
							rslt = new CosObjectStream(doc, map, info, pos);
						else
							rslt = new CosStream(doc, map, info, pos);
					}
				}
				if (rslt == null)
					rslt = new CosDictionary(doc, map, info);
			} else {
				buf.unget();
				boolean[] oddBall = new boolean[1];
				byte[] str = readHexString(buf, oddBall);
				rslt = new CosString(doc, str, 0, str.length, true, info, true, oddBall[0]);
			}
		} else if (b == (byte)'[') {
			ArrayList objectsList = readArray(doc, buf, nextObjPos);
			if(objectsList == null){
				rslt = doc.createCosNull();// couldn't read the array.
				if(info != null){
					rslt.setInfo(info);
					doc.setRepairedValue(info.getObjNum(), rslt);
				}
			}else
				rslt = new CosArray(doc, objectsList, info);
		} else if (b == (byte)'t') {
			rslt = new CosBoolean(doc, readTrue(buf).booleanValue(), info);
		} else if (b == (byte)'f') {
			rslt = new CosBoolean(doc, readFalse(buf).booleanValue(), info);
		} else if (b == (byte)'n') {
			readNull(buf);
			rslt = doc.createCosNull();
		} else {
			throw new PDFCosParseException("Unexpected token at position " + Long.toString(buf.getPosition() - 1));
		}
		return rslt;
	}
	
	/**
	 * This method tokenizes the document stream to reach the position where object definition
	 * is completed.
	 * @param doc		Document containing the COS object
	 * @param buf		Buffer to skip
	 * @param b		    Starting byte
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 * @throws PDFIOException
	 */
	public static void skipObject(CosDocument doc, InputByteStream buf, byte b)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		if (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.') {
			do {
				b = (byte)buf.read();
			} while (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.');
			if(b != InputByteStream.EOF)
				buf.unget();
			// A cosnumeric is skipped
		} else if (b == (byte)'R') {
			// End of indirect object reference
		} else if (b == (byte)'/') {
			skipName(buf);
		} else if (b == (byte)'(') {
			ByteArrayUtility.skipLiteral(buf);
		} else if (b == (byte)'<') {
			b = (byte)buf.read();
			if (b == (byte)'<') {
				long dictStartPos = buf.getPosition();
				skipDictionary(doc, buf);
				
				b = skipWhitespace(buf);
				if (b == 's') {
					if (buf.read()=='t' && buf.read()=='r' && buf.read()=='e' && buf.read()=='a' && buf.read()=='m')
					{
						// No option but to parse the dictionary to get the value of length entry
						buf.seek(dictStartPos);
						Map<ASName, CosObject> map = readDictionary(doc, buf, CosToken.DEFAULT_NEXT_OBJ_POS);
						b = (byte)buf.read();	// see section 3.2.7 of pdf spec (we are handling extra cases here)
						while (b == ' ')
							b = (byte)buf.read();
						if (b == '\r') {
							b = (byte)buf.read();
							if (b != '\n') buf.unget();	// if \r but not \r\n
						} else if (b != '\n') buf.unget();	// if no \r or \n
						long length = -1;
						CosObject obj = map.get(ASName.k_Length);
						if(obj instanceof CosNumeric)
						{
							length = ((CosNumeric) obj).longValue();
						}
						else if(obj instanceof CosObjectRef)
						{
							// In case consumeric existed as an indirect object
							if(((CosObjectRef) obj).getValueType() == CosObject.t_Numeric)
							{
								length = ((Number) ((CosObjectRef)obj).getValue()).longValue();
							}
						}
						if(length != -1)
						{
							// If could not find length it means that PDF could be invalid. Still do not
							// throw an exception as we are just skipping data used for object definition
							// and not validating it.
							buf.seek(buf.getPosition() + length);
						}
					}
				}else
				{
					buf.unget();
				}
				
			} else {
				buf.unget();
				skipHexString(buf);
			}
		} else if (b == (byte)'[') {
			skipArray(doc, buf);
		} else if (b == (byte)'t') {
			skipTrue(buf);
		} else if (b == (byte)'f') {
			skipFalse(buf);
		} else if (b == (byte)'n') {
			skipNull(buf);			
		}
		return;
	}

	/**
	 * Parse the byte stream to create a list from a PDF array or dictionary.
	 *
	 * @param doc			Document containing the object being parsed
	 * @param buf			Byte stream to parse
	 * @param delimiter		Termination character
	 *
	 * @return List of array elements or dictionary entries (key, value, ...)
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private static ArrayList<CosObject> readObjectList(CosDocument doc, InputByteStream buf,
						byte delimiter, long nextObjPos)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		ArrayList<CosObject> rslt = new ArrayList<CosObject>();
		byte b = skipWhitespace(buf);
		while (b != delimiter && b != InputByteStream.EOF) {
			CosObject obj = readObject(doc, buf, b, null, rslt, false, nextObjPos);
			if(obj == null)
				return null;
			rslt.add(obj);
			b = skipWhitespace(buf);
		}
		return rslt;
	}
	
	/**
	 * Parse the byte stream and skips a list from a PDF array or dictionary.
	 *
	 * @param doc			Document containing the object being parsed
	 * @param buf			Byte stream to parse
	 * @param delimiter		Termination character
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 * @throws PDFSecurityException
	 * @throws PDFIOException
	 */
	private static void skipObjectList(CosDocument doc, InputByteStream buf,
						byte delimiter)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		byte b = skipWhitespace(buf);
		while (b != delimiter && b != InputByteStream.EOF) {
			skipObject(doc, buf, b);
			b = skipWhitespace(buf);
		}
		return;
	}
	

	/**
	 * Parses t"rue" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @return Boolean.TRUE
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static Boolean readTrue(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		if (buf.read() != 'r')
			throw new PDFCosParseException("Expected 'true' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'u')
			throw new PDFCosParseException("Expected 'true' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'e')
			throw new PDFCosParseException("Expected 'true' - " + Long.toString(buf.getPosition() - 1));
		return Boolean.TRUE;
	}
	
	/**
	 * Skips t"rue" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void skipTrue(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		buf.read();// 'r'
		buf.read();// 'u'
		buf.read();// 'e'
	}
	
	/**
	 * Skips f"alse" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void skipFalse(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		buf.read();// 'a'
		buf.read();// 'l'
		buf.read();// 's'
		buf.read();// 'e'
	}
	
	/**
	 * Skips n"ull" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void skipNull(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		buf.read();// 'u'
		buf.read();// 'l'
		buf.read();// 'l'
	}

	/**
	 * Parses f"alse" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @return Boolean.FALSE
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static Boolean readFalse(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		if (buf.read() != 'a')
			throw new PDFCosParseException("Expected 'false' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'l')
			throw new PDFCosParseException("Expected 'false' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 's')
			throw new PDFCosParseException("Expected 'false' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'e')
			throw new PDFCosParseException("Expected 'false' - " + Long.toString(buf.getPosition() - 1));
		return Boolean.FALSE;
	}

	/**
	 * Parses n"ull" from the PDF byte stream.
	 *
	 * @param buf		Buffer to parse
	 *
	 * @throws PDFCosParseException
	 * @throws IOException
	 */
	private static void readNull(InputByteStream buf)
		throws PDFCosParseException, IOException
	{
		if (buf.read() != 'u')
			throw new PDFCosParseException("Expected 'null' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'l')
			throw new PDFCosParseException("Expected 'null' - " + Long.toString(buf.getPosition() - 1));
		if (buf.read() != 'l')
			throw new PDFCosParseException("Expected 'null' - " + Long.toString(buf.getPosition() - 1));
	}

	static ASObject readPrimitive(InputByteStream buf)
		throws IOException, PDFCosParseException
	{
		long savePos;
		byte b = skipWhitespace(buf);
		if (buf.eof())
			return null;
		if (ByteOps.isDigit(b) || b == (byte)'+' || b == (byte)'-' || b == (byte)'.') {
			savePos = buf.getPosition();
			try {
				return ParseOps.readNumber(b, buf);
			} catch (PDFParseException e) {
			}
			buf.seek(savePos);
		}
		if (b == (byte)'/') {
			savePos = buf.getPosition();
			try {
				return ParseOps.readName(buf);
			} catch (PDFParseException e) {
			}
			buf.seek(savePos);
		}
		if (ByteOps.isRegular(b)) {
			StringBuilder strBuf = new StringBuilder();
			strBuf.append((char)b);
			while (!buf.eof() && ByteOps.isRegular(b = (byte)buf.read()))
				strBuf.append((char)b);
			buf.unget();
			return new ASString(strBuf.toString());
		}
		if (b == (byte)'<') {
			if ((b = (byte)buf.read()) == (byte)'<')
				return new ASString("<<");
			buf.unget();
		}
		if (b == (byte)'>') {
			if ((b = (byte)buf.read()) == (byte)'>')
				return new ASString(">>");
			buf.unget();
		}
		byte[] bytes = new byte[1];
		bytes[0] = b;
		return new ASString(bytes);
	}
}
