/*
 * File: TIFFOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.OutputStream;
import java.io.FilterOutputStream;
import java.io.IOException;

/**
 * TIFFOutputStream
 *  Performs TIFF image filtering of an output stream
 */
public class TIFFOutputStream extends FilterOutputStream
{
	// Variables
	private TIFFEngine	tiff;

	// Constructors
	/**
	 * Creates an output stream filter.
	 * @param out	the output stream
	 */
	public TIFFOutputStream(OutputStream  out, FilterParams p)
	{
		super(out);
		tiff = new TIFFEngine(out, p);
	}

	public TIFFOutputStream(OutputStream  out)
	{
		this(out, null);
	}

	/**
	 * Writes a byte. Will block until the byte is actually
	 * written.
	 * @param b the byte
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(int b)
		throws IOException{
		tiff.put(b & 0x0ff);
	}
	
	
	/**
	 * Writes a subarray of bytes.
	 * @param b	the data to be written
	 * @param off	the start offset in the data
	 * @param len	the number of bytes that are written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[], int  off, int  len)
		throws IOException
	{
		int	maxWrite = b.length-off;
		if (maxWrite > len)
			maxWrite = len;
		
		if(tiff.getBitsPerComponent()==16){
			while (maxWrite > 0) {
				short channel =0;
	            int newChar = (short) (b[off++] & 0x00FF);
	            channel = (short) (channel | (newChar<<8));
	            newChar = (short) (b[off++] & 0x00FF);
	            channel = (short) (channel | newChar);
	            tiff.put(channel);
	            maxWrite-=2;
			}
		}else{
			while (maxWrite-- > 0) {
				write(b[off++] & 0x0ff);
			}
		}
	}

	/**
	 * Writes an array of bytes. Will block until the bytes
	 * are actually written.
	 * @param b	the data to be written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte  b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	
	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return tiff.getTotalOut();
	}
}
