/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.form;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EnumValue;
import com.adobe.xfa.EventManager;
import com.adobe.xfa.LogMessage;
import com.adobe.xfa.Schema;
import com.adobe.xfa.XFA;
import com.adobe.xfa.template.TemplateModel;
import com.adobe.xfa.template.containers.Container;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;

/**
 * @execute from public api.
 */
class ExecuteDispatcher extends com.adobe.xfa.Dispatcher {
	
	private final Element mExecuteNode;
	private final boolean mbValidate;


	ExecuteDispatcher(Element scriptContextNode,
					  String sEventContext,
					  int nEventID,
					  EventManager eventManager,
					  Element executeNode,
					  boolean bValidate) {
		super(scriptContextNode,
			  sEventContext, 
			  nEventID, 
			  eventManager);
		mExecuteNode = executeNode;
		mbValidate = bValidate;
	}

	public void dispatch() {
		Element node = getActionContextNode();

		if (node == null)
			return;

		int eRunAt = getRunAt();
		String sConnection = getConnection();
		int eExecuteType = getExecuteType();

		assert( node instanceof FormField ||
				node instanceof FormSubform ||
				node instanceof FormExclGroup);
				
		// note would be nice to directly access mpoNodeImpl.mpModel
		FormModel formModel = (FormModel)node.getModel();
		int eModelRunAtSetting = formModel.getRunScripts();
		
		// As of //https://zerowing.corp.adobe.com/display/xtg/InactivePresenceXFAProposal,
		// event processing associated with inactive containers must not occur.
		// To mitigate performance on older forms, only do this check for XFA 3.0 documents and higher.
		AppModel appModel = node.getAppModel();
		if (appModel != null) {
			TemplateModel templateModel = TemplateModel.getTemplateModel(appModel, false);
			if ((templateModel != null) &&
				(templateModel.getOriginalXFAVersion() >= Schema.XFAVERSION_30) &&
		        (node instanceof Container)) {
				Container container = (Container)node;
				int ePresence = container.getRuntimePresence(EnumAttr.UNDEFINED);
				if (EnumAttr.PRESENCE_INACTIVE == ePresence)
					return;
			}
		}
		
		boolean bValid = true;
		
		if (mbValidate)
			bValid = formModel.performPreEventValidations();
			
		if (bValid) {

			// If we're set to be the client (i.e. eModelRunAtSetting == EnumAttr.RUNSERVER_CLIENT)
			// and we're provided with a ServerExchange, send a "click" event to the server.
			if (eRunAt != EnumAttr.RUNAT_CLIENT) {
				if (eModelRunAtSetting == EnumAttr.RUNSCRIPTS_CLIENT)
						formModel.serverExchange(node, "click");
			}

			FormModel.Execute execute = formModel.getExecute();
			if (execute == null)
				return;
			
			// Fix for Watson  2384257, if dataConnection is supposed to run at Server.
			// It should not be executed at client. Protected the change using XFA V3.2 Scripting Legacy flag.
			///*
			boolean bLegacyScripting = true;
			TemplateModel oTemplateModel = TemplateModel.getTemplateModel(appModel, true);		
			if( null != oTemplateModel) 
			{
				if(!oTemplateModel.getLegacySetting(AppModel.XFA_LEGACY_V32_SCRIPTING))
					bLegacyScripting = false;
			}
			if(bLegacyScripting)
				execute.execute(sConnection, eRunAt, eExecuteType);
			else if (eRunAt != EnumAttr.RUNAT_SERVER)
				execute.execute(sConnection, eRunAt, eExecuteType);
			//*/			
			//execute.execute(sConnection, eRunAt, eExecuteType);			
		} 
		else {
			//One or more validations failed - do not execute
			ExFull exFull = new ExFull(ResId.FormExecuteCancelled);
			formModel.addErrorList(exFull, LogMessage.MSG_WARNING, getActionContextNode());
		}
	}

	int getRunAt() {
		if (mExecuteNode != null) {
			Attribute runAt = mExecuteNode.getAttribute(XFA.RUNATTAG);
			int eRunAt = ((EnumValue)runAt).getInt();

			return eRunAt;
		}

		return EnumAttr.RUNAT_BOTH;
	}

	String getConnection() {
		if (mExecuteNode != null)
		{
			Attribute connectionAttr = mExecuteNode.getAttribute(XFA.CONNECTIONTAG);
			String sConnection = connectionAttr.toString();

			return sConnection;
		}

		return "";
	}

	int getExecuteType() {
		if (mExecuteNode != null)
		{
			Attribute executeType = mExecuteNode.getAttribute(XFA.EXECUTETYPETAG);
			int eExecuteType = ((EnumValue)executeType).getInt();

			return eExecuteType;
		}

		return EnumAttr.EXECUTETYPE_IMPORT;
	}
	
	public String getDispatcherType() {
		return "execute";
	}
}
