/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.form;

import com.adobe.xfa.AppModel;
import com.adobe.xfa.Attribute;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EventManager;
import com.adobe.xfa.LogMessage;
import com.adobe.xfa.Node;
import com.adobe.xfa.Packet;
import com.adobe.xfa.Schema;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.template.TemplateModel;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;

import java.util.ArrayList;
import java.util.List;


/**
 * @exclude from public api.
 */
class SubmitDispatcher extends com.adobe.xfa.Dispatcher {
	
	private final Element mSubmitNode;
	private final boolean mbValidate;
	

	SubmitDispatcher(Element submitContextNode,
					 String sEventContext,
					 int nEventID,
					 EventManager eventManager,
					 Element submitNode,
					 boolean bValidate) {
		super(submitContextNode, 
				sEventContext, 
				nEventID, 
				eventManager);
		
		mSubmitNode = submitNode;
		mbValidate = bValidate;
	}

	public void dispatch() {
		Element node = getActionContextNode();

		if (node == null)
			return;

		assert( node instanceof FormField ||
				node instanceof FormSubform ||
				node instanceof FormExclGroup);
				
		// note would be nice to directly access mpoNodeImpl.mpModel
		FormModel formModel = (FormModel)node.getModel();
		
		FormModel.Submit submit = formModel.getSubmit();
		if (submit == null)
			return;

		// set up preSubmit/postSubmit events
		String sPreSubmit = EnumAttr.getString(EnumAttr.ACTIVITY_PRESUBMIT);
		String sPostSubmit = EnumAttr.getString(EnumAttr.ACTIVITY_POSTSUBMIT);
		
		// dispatch preSubmit event
		if (sPreSubmit.length() != 0)
			formModel.eventOccurred(sPreSubmit, formModel);

		EventManager poEventManager = getEventManager();
		if (poEventManager != null && !poEventManager.cancelAction(XFA.SUBMIT)) {
			
			//Watson 1317097 - only include the <execEvent> packet if 
			//config.present.submitUrl is set in configuration file.
			String sSubmitURL = formModel.getSubmitURL();
	
			Packet execEvent = null;
			if (!StringUtils.isEmpty(sSubmitURL)) {
				AppModel appModel = (AppModel)formModel.getXFAParent();
				
				execEvent = (Packet)appModel.createElement(XFA.PACKETTAG, "execEvent");
	
				// We have to use validate=false to append the packet to the AppModel.
				// It's allowed; the schema just doesn't know it.
				appModel.appendChild(execEvent, false);
			
				//bug 2426847
				submit.setPacketToIgnore(execEvent);
				
				// Watson 1875703/1874913
				// The context node for preSubmit should be the root of the form dom,
				// not the node that contains the <submit/> element. This is so that
				// preSubmit will fire for the entire form dom, and not just one node.
				TemplateModel oTemplateModel = TemplateModel.getTemplateModel(appModel, false);
				int nOriginalVersion = oTemplateModel.getOriginalXFAVersion();
				if (nOriginalVersion >= Schema.XFAVERSION_30) {	// Acrobat 9.1
					execEvent.setAttribute("$form", "context");
				}
				else {
					// Set the context attribute (relative to the form model)
					// watson bug 1463330 and 1466334
					execEvent.setAttribute(node.getSOMExpression(formModel, false), "context");
				}
				
				// Set the activity attribute
				execEvent.setAttribute(sPreSubmit, "activity");
			}
	
			//If this is a submit event:
			//0) Recalcs the form if recalcs were missed
			//   because of the calculationsEnabled flag.
			//1) Ensures form is in valid state, if we are to perform validations (I.e.,mbValidate=true).
			//   (Means all validations pass and mandatory fields have data) 
			//2) Submit
	
			boolean bValid = true;
			if (mbValidate)
				bValid = formModel.performPreEventValidations();			
	
			if (bValid) {
				// dig the submit info. We do it here every time because user might change the 
				// the submit's attributes at run-time (like target)
				int eFormat = getFormat();
				String sPackets = getXDPContent();
				boolean bEmbedPDF = getEmbedPDF();
				String sTarget = getTarget();
				String sTextEncoding = getTextEncoding();
				String sCertificate = getCertificate();
	
				// ensure we have the right case for the encoding
				if (sTextEncoding.startsWith("UTF"))
					sTextEncoding = sTextEncoding.toLowerCase();
	
				if (StringUtils.isEmpty(sPackets))
					sPackets = "datasets pdf xfdf";	// default
	
				if (execEvent != null) {
					if (!sPackets.contains("execEvent"))
						sPackets += " execEvent";
				}
	
				if (bEmbedPDF) {
					// force embed of PDF
					if (!sPackets.contains("pdf"))
						sPackets += " pdf";
				}
	
				String[] packets = sPackets.split(" ");			
				List<SignDispatcher> signDispatchers = getSignDispatchers();
				
				FormModel.Submit.SubmitParams params = new FormModel.Submit.SubmitParams(packets, sTarget, eFormat, sTextEncoding, bEmbedPDF);
				params.setCertificate(sCertificate);
				params.setSignDispatchers(signDispatchers);
				
				try {
					submit.submit(params);
				}
				catch (ExFull oError) {					
					submit.setPacketToIgnore(null);
					// remove the exec event node 
					if (execEvent != null && execEvent.getXFAParent() != null)
						execEvent.remove();
	
					// dispatch postSubmit event even if there is an error
					if (!StringUtils.isEmpty(sPostSubmit))
						formModel.eventOccurred(sPostSubmit, formModel);
	
					throw oError;
				}
			}				
			else {
				// One or more validations failed - do not submit
				ExFull exFull = new ExFull(ResId.FormSubmitCancelled);
				formModel.addErrorList(exFull, LogMessage.MSG_WARNING, getActionContextNode());
			}
			
			submit.setPacketToIgnore(null);
			// remove the exec event node 
			if (execEvent != null && execEvent.getXFAParent() != null)
				execEvent.remove();
		}
		
		// dispatch postSubmit event
		if (sPostSubmit.length() != 0)
			formModel.eventOccurred(sPostSubmit, formModel);
	}

	public String getDispatcherType() {
		return "submit";
	}

	public String getParameter(int nParam) {
		if (nParam == 0)
			return getTarget();

		return "";
	}

	int getFormat() {
		if (mSubmitNode != null)
			return mSubmitNode.getEnum(XFA.FORMATTAG);

		return EnumAttr.UNDEFINED;
	}

	String getXDPContent() {
		String sXDPContent = "";
		if (mSubmitNode != null) {
			Attribute content = mSubmitNode.getAttribute(XFA.XDPCONTENTTAG);
			sXDPContent = content.toString();
		}

		return sXDPContent;
	}
			
	boolean getEmbedPDF() {
		boolean bEmbedPDF = false;
		if (mSubmitNode != null) {
			int eEmbedPDF = mSubmitNode.getEnum(XFA.EMBEDPDFTAG);
			if (eEmbedPDF == EnumAttr.BOOL_TRUE)
				bEmbedPDF = true;	
		}
		return bEmbedPDF;
	}

	String getTarget() {
		String sTarget = "";
		if (mSubmitNode != null) {
			Attribute targetAttr = mSubmitNode.getAttribute(XFA.TARGETTAG);
			sTarget = targetAttr.toString();
		}

		return sTarget;
	}

	String getTextEncoding() {
		String sTextEncoding = "";
		if (mSubmitNode != null) {
			Attribute textEncoding = mSubmitNode.getAttribute(XFA.TEXTENCODINGTAG);
			sTextEncoding = textEncoding.toString();			
		}

		return sTextEncoding;
	}

	String getCertificate() {	
		String sCert = null;
		if (mSubmitNode != null) {
			Element encrypt = mSubmitNode.peekElement(XFA.ENCRYPTTAG, false, 0);
			if (encrypt != null)  {
				Element certificate = encrypt.peekElement(XFA.CERTIFICATETAG, false, 0);
				if (certificate != null) {
					TextNode contents = certificate.getText(false, false, false);
					if (contents != null)
						sCert = contents.getValue();
				}
			}
		}

		return sCert;
	}

	List<SignDispatcher> getSignDispatchers() {
		List<SignDispatcher> signDispatchers = new ArrayList<SignDispatcher>();

		if (mSubmitNode != null) {
			for (Node child = mSubmitNode.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				if (child.isSameClass(XFA.SIGNDATATAG)) {				
					EventManager eventManager = getEventManager();
					
					SignDispatcher sd = new SignDispatcher(getActionContextNode(),
															 (Element)child,
															 getEventContext(),
															 getEventID(),
															 eventManager);
					//Obj.addRef(pSD);
					signDispatchers.add(sd);
				}
			}
		}

		return signDispatchers;
	}
}