package com.adobe.xfa.text;

/**
 * As an abstract base class, TextMarkupBase defines the semantics for
 * building a representation of rich text from a series of method calls.
 * It was designed as the base class for output markup engines, which
 * translate those calls into their respective markup languages.  In
 * addition, serves as the base class for the text position classes,
 * because a position can also build up (add) rich text in its
 * underlying stream through a series of method calls.
 * <p>
 * For more information, please see the extenral documentation.
 * </p>
 * @exclude from published api.
 */

public abstract class TextMarkupBase {

/**
 * Add text to the rich text representation.
 * @param sText - Text to be added.
 */
	abstract public void text (String sText);

/**
 * Change attributes in the rich text representation.
 * <p>
 * Because the rich text representation is built up serially, this
 * method sets the attributes for subsequent text, until called again.
 * @param oAttr - Attributes to set.  The implementation must honour
 * enabled attributes and carry forward disabled ones.
 */
	abstract public void attr (TextAttr oAttr);

/**
 * Output a paragraph break to the rich text representation.
 */
	abstract public void para ();

/**
 * Add a field to the rich text representation.
 */
	public void field (TextField poField) {
	}

	public void openScopedBlock () {
	}

	public void closeScopedBlock () {
	}

	public boolean issueFirstPara () {
		return false;
	}

	public boolean legacyPositioning () {
		return false;
	}

/**
 * Append a position marker to the markup output.
 * @param poMarker - Pointer to marker to append.  Note that markers are
 * always cloned on insertion, so a copy actually gets inserted.  The
 * caller continues to retain ownership of the instance referred to by
 * this parameter, and can delete it any time after the call.
 */
	public void marker (TextMarker poMarker) {
	}

/**
 * Start a range marker in the markup output.
 * <p>
 * There must be a corresponding call to MarkerEnd() sometime later in
 * the translation for the markup operation to be considered valid.
 * @param poMarker - Pointer to marker to append.  Note that markers are
 * always cloned on insertion, so a copy actually gets inserted.  The
 * caller continues to retain ownership of the instance referred to by
 * this parameter, and can delete it any time after the call.
 * @return Pointer to a marker object that represents the range marker
 * in the output.  While this is a non-const pointer, it is owned by
 * AXTE and must not be modified by the client.  The client must hang on
 * to this pointer to complete the range.  The client can perform one of
 * two mutually-exclusive actions with this pointer.  Normally, it
 * passes it to a subsequent call to MarkerEnd() to complete the range.
 * Alternatively, it may subsequently remove its reference, cancelling
 * the marker range.  The client must not perform both these actions on
 * a single pointer, nor may it perform any other actions on the
 * pointer.
 */
	public TextMarker markerStart (TextMarker poMarker) {
		return null;
	}

/**
 * End a range marker in the markup output.
 * <p>
 * This must occur some time after a corresponding call to MarkerStart()
 * earlier in the translation for the markup operation to be considered
 * valid.
 * @param poMarker - Pointer to the marker object returned by the
 * earlier call to MarkerStart().  The client must use the return value
 * of that earlier call, not the value passed into it.	In making this
 * call, the client transfers ownership if its reference to AXTE and
 * must not subsequenty attempt to remove that reference on this marker.
 */
	public void markerEnd (TextMarker poMarker) {
	}

/**
 * Query the attributes currently in effect.
 * @return A pointer to the attributes currently in effect.  The default
 * implementation returns NULL.
 */
	public TextAttr attributePtr () {
		return null;
	}
}
