/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.testandtarget;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.analytics.testandtarget.impl.serializer.PerformanceReportSerializer;

/**
 * The {@code PerformanceReport} contains performance reporting data about a single campaign
 *
 */
public class PerformanceReport {

    private static final Logger LOG = LoggerFactory.getLogger(PerformanceReport.class);

    private List<PerformanceReportItem> items = new ArrayList<PerformanceReportItem>();

    private PerformanceReportItem totals = null;

    private String winningExperienceName = "";

    /**
     * Adds a {@link PerformanceReportItem}
     * @param item the item instance to add, must not be <code>null</code>
     */
    public void addItem(PerformanceReportItem item) {

        if (item == null) {
            throw new IllegalArgumentException("item may not be null");
        }

        if (PerformanceReportSerializer.ACTIVITY_TOTALS_NAME.equalsIgnoreCase(item.getExperienceName())) {
            totals = item;
        } else {
            items.add(item);
        }
    }

    /**
     * Returns a list of {@link PerformanceReportItem}
     * @return an unmodifiable view over the experiences contained in this report
     */
    public List<PerformanceReportItem> getItems() {

        return Collections.unmodifiableList(items);
    }

    /**
     * Sets the winning experience local id.
     * @param experienceName a <code>String</code> value representing the name of the winning experience
     */
    public void setWinningExperienceName(String experienceName) {
        winningExperienceName = experienceName;
    }

    /**
     * Retrieves the name of the winning experience or an empty string if there is no winning experience in
     * the report
     * @return a string representation of the winning experience
     */
    public String getWinningExperienceName() {
        return winningExperienceName;
    }
    /**
     * Returns the total performance for all experience contained in this report
     * 
     * <p>
     * The {@code name} field will always be set to null
     * </p>
     * 
     * @return the total performance for all experience contained in this report
     */
    public PerformanceReportItem getTotalPerformance() {

        int conversionCount = 0;
        double engagement = 0;
        int engagementCount = 0;
        double engagementTotal = 0;
        int entryCount = 0;

        for (PerformanceReportItem item : items) {
            conversionCount += item.getConversionCount();
            engagementCount += item.getEngagementCount();
            engagementTotal += item.getEngagementTotal();
            entryCount += item.getEntryCount();
        }

        engagement = engagementCount != 0
                ? engagementTotal / engagementCount
                : 0;

        if (totals == null) {
            totals = new PerformanceReportItem();

            totals.setEntryCount(entryCount);
            totals.setConversionRate(conversionCount / (double) entryCount);
        }

        totals.setConversionCount(conversionCount);
        totals.setEngagement(engagement);
        totals.setEngagementTotal(engagementTotal);
        return totals;
    }
}
