/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.analytics.testandtarget;

import com.day.cq.wcm.webservicesupport.Configuration;
import java.util.List;
import org.osgi.annotation.versioning.ProviderType;

/**
 * The {@code TestandtargetCommandService} is the entry point for all the operations which require
 * to relay the commands to Adobe Target server without any interpretation.
 *
 * <p>This service is able to perform operations on Adobe Target entry points via REST only.
 * This service doesn't interpret the commands at all, it simply acts like a proxy.
 * The onus of creating the correct body of the request lies with the caller.
 * Similarly, wherever mentioned, this command doesn't interpret the data returned
 * by Target server and simply returns it to the caller.</p>
 *
 */

@ProviderType
public interface TestandtargetCommandService {

    /**
     * Fetches the list of audiences from Target server
     * @param configuration a reference to the Cloud Service configuration
     * @return a list of TestandtargetAudience
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    List<TestandtargetAudience> getAudienceList(Configuration configuration) throws TestandtargetException;

    /**
     * Deletes the campaign and the offers it is referring to.
     * @param configuration a reference to the Cloud Service configuration
     * @param campaignID campaign ID of the campaign
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    void deleteCampaign(Configuration configuration, long campaignID) throws TestandtargetException;

    /**
     * Creates the visual offer
     * @param configuration a reference to the Cloud Service configuration
     * @param offerName the name of the offer
     * @param content The content that is to be delivered
     * @return the byte stream of the JSON returned by the server
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String createVisualOffer(Configuration configuration, String offerName, String content) throws TestandtargetException;

    /**
     * Creates the segment from the requestBody provided by the caller
     * @param configuration a reference to the Cloud Service configuration
     * @param segmentName The name of the offer
     * @param segmentJson the request body for the creation of the segment
     * @param isReusable  {@link Boolean} true if the audience is to be reused
     * @param modifiedByUser User creating the audience
     * @return byte stream of the JSON returned by the server
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String createSegment(Configuration configuration, String segmentName, String segmentJson, boolean isReusable, String modifiedByUser) throws TestandtargetException;

    /**
     * Creates the campaign from the requestBody provided by the caller.
     * @param configuration a reference to the Cloud Service configuration
     * @param campaignName The name of the campaign. Target doesn't fail in case of duplicate campaign name.
     * @param campaignJson the body of the campaign creation command
     * @return The byte stream of the JSON returned by the Target server
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String createCampaign(Configuration configuration, String campaignName, String campaignJson) throws TestandtargetException;

    /**
     * Deletes the segment
     * @param configuration a reference to the Cloud Service configuration
     * @param segmentID The segment ID to be deleted
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    void deleteSegment(Configuration configuration, long segmentID) throws TestandtargetException ;

    /**
     * Deletes the offer
     * @param configuration a reference to the Cloud Service configuration
     * @param offerId The offer to be deleted
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    void deleteOffer(Configuration configuration, long offerId) throws TestandtargetException ;

    /**
     * Updates the campaign to the new requestBody provided by the caller
     * @param configuration a reference to the Cloud Service configuration
     * @param campaignId The id of the campaign to be updated
     * @param campaignJson The updated campaign
     * @return The byte stream of the JSON returned by the Target server
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String updateCampaign(Configuration configuration, long campaignId, String campaignJson) throws TestandtargetException;

    /**
     * Fetches the details of the campaign
     * @param configuration a reference to the Cloud Service configuration
     * @param campaignId The id of the campaign to be updated
     * @return The byte stream of the JSON returned by the Target server
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String getCampaignDetails(Configuration configuration, long campaignId) throws TestandtargetException;

    /**
     * Given a campaign ID, fetches the summary report from Target server
     * @param configuration a reference to the Cloud Service configuration
     * @param campaignId campaign ID of the campaign
     * @return current report for the campaign ID
     * @throws TestandtargetException {@link TestandtargetException} on all errors
     */
    String getReportSummary(Configuration configuration, long campaignId) throws TestandtargetException;
}
