/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.widget;

import javax.servlet.ServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Doctypes Available in the W3C Recommendations
 *
 * @since 5.5
 */
public enum Doctype {

    /**
     * HTML 5 doctype.
     * The DOCTYPE declaration is {@code <!DOCTYPE html>} and is case-insensitive in the
     * HTML syntax. DOCTYPEs from earlier versions of HTML were longer because
     * the HTML language was SGML-based and therefore required a reference to a
     * DTD. With HTML5 this is no longer the case and the DOCTYPE is only needed
     * to enable standards mode for documents written using the HTML syntax.
     * Browsers already do this for {@code <!DOCTYPE html>}..
     */
    HTML_5("<!DOCTYPE HTML>"),

    /**
     * HTML 4.01 Strict
     * This DTD contains all HTML elements and attributes, but does not include
     * presentational or deprecated elements (like font).
     * Framesets are not allowed.
     */
    HTML_401_STRICT("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" " +
            "\"http://www.w3.org/TR/html4/strict.dtd\">"),

    /**
     * HTML 4.01 Transitional
     * This DTD contains all HTML elements and attributes, including
     * presentational and deprecated elements (like font).
     * Framesets are not allowed.
     */
    HTML_401_TRANS("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" " +
            "\"http://www.w3.org/TR/html4/loose.dtd\">"),

    /**
     * HTML 4.01 Frameset
     * This DTD is equal to HTML 4.01 Transitional, but allows the use of
     * frameset content.
     */
    HTML_401_FRAMESET("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\" " +
            "\"http://www.w3.org/TR/html4/frameset.dtd\">"),

    /**
     * XHTML 1.0 Strict
     * This DTD contains all HTML elements and attributes, but does not include
     * presentational or deprecated elements (like font). Framesets are not
     * allowed. The markup must also be written as well-formed XML.
     */
    XHTML_10_STRICT("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"),

    /**
     * XHTML 1.0 Transitional
     * This DTD contains all HTML elements and attributes, including
     * presentational and deprecated elements (like font). Framesets are not
     * allowed. The markup must also be written as well-formed XML.
     */
    XHTML_10_TRANS("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">"),

    /**
     * XHTML 1.0 Frameset
     * This DTD is equal to XHTML 1.0 Transitional, but allows the use of
     * frameset content.
     */
    XHTML_10_FRAMESET("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Frameset//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-frameset.dtd\">"),

    /**
     * XHTML 1.1
     * This DTD is equal to XHTML 1.0 Strict, but allows you to add modules
     * (for example to provide ruby support for East-Asian languages).
     */
    XHTML_11("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" " +
            "\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">");

    /**
     * name of the Doctype request attribute
     */
    private static final String REQUEST_ATTRIBUTE_NAME = Doctype.class.getName();

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(Doctype.class);

    /**
     * declaration
     */
    private final String declaration;

    private Doctype(String declaration) {
        this.declaration = declaration;
    }

    /**
     * Returns the declaration string of this doctype.
     * @return declaration string
     */
    public String getDeclaration() {
        return declaration;
    }

    /**
     * Checks if this doctype is XHTML.
     * @return <code>true</code> if this doctype is XHTML.
     */
    public boolean isXHTML() {
        return name().startsWith("XHTML_");
    }

    /**
     * Convenience method that retieves the doctype from the request attribute
     * and checks if the doctype is XHTML.
     *
     * @param req servlet request
     * @return <code>true</code> if the doctype is XHTML.
     */
    public static boolean isXHTML(ServletRequest req) {
        Doctype d = fromRequest(req);
        return d != null && d.isXHTML();
    }

    /**
     * Returns the current doctype of this request.
     * @param req servlet request
     * @return current doctype or <code>null</code> if not set.
     */
    public static Doctype fromRequest(ServletRequest req) {
        Object o = req.getAttribute(REQUEST_ATTRIBUTE_NAME);
        if (o == null) {
            return null;
        } else if (o instanceof Doctype) {
            return (Doctype) o;
        } else  {
            log.error("Object stored in request is not of an instance of Doctype: {}. Returning null.", o);
            return null;
        }
    }

    /**
     * Sets the current doctype of this request.
     * @param req servlet request
     */
    public void toRequest(ServletRequest req) {
        Doctype prev = fromRequest(req);
        if (prev != null && !this.equals(prev)) {
            log.warn("Overwriting existing request doctype ({}) with different value {}", prev, this);
        }
        req.setAttribute(REQUEST_ATTRIBUTE_NAME, this);
        if (log.isDebugEnabled()) {
            log.debug("Setting request doctype to {}", this);
        }
    }

}
