/*
*
*	File: FontFactory.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Creates font objects for OpenType fonts */
final public class FontFactory {

  // A OpenType container is a series of blocks, each block being
  // either a TTC Header, an Offset table for a font, or an OpenType table.
  //
  // The specification does not impose that these various blocks be 
  // disjoint: for example, if two blocks are such that last <i>n</i> bytes
  // of the first are identical to the first <i>n</i> bytes of the 
  // second, then it is possible to "share" those bytes 
  // in the file. We support this, although the data will be replicated in-memory for each table
  // (unless the 2 tables are the exact same range of bytes, in which case they will be shared).
  // 
  // Our loading strategy is make a single pass through the byte stream, and
  // to lift the blocks as we go. The purpose of this strategy is to
  // avoid having to buffer the whole file when the actual source does not
  // support fast random access (e.g. fonts stored in base64 in an SVG 
  // document, or fonts accessed via HTTP). 
  //
  // The class BlockToLoad represents one block we need to read, as
  // well as a pointer to the next block (in offset order). We then have
  // one subclass for each kind of block.
  
  private static abstract class BlockToLoad {
    public BlockToLoad next;
    public long offset;
    public int length;
    
    public BlockToLoad (long offset, int length) {
      super ();
      this.offset = offset;
      this.length = length;
      this.next = null;
    }
    
    public abstract void load (FontInputStream in, BlockToLoad firstBlock)
        throws IOException, InvalidFontException, UnsupportedFontException;
    
    protected abstract OTByteArray getByteArray();
    
    public BlockToLoad add (BlockToLoad b) throws InvalidFontException {
      if (next == null) {
    	next = b;
        return b; }
      else if (b.offset < next.offset || 
    		  (b.offset == next.offset && b.length < next.length)) {
        b.next = next;
        next = b;
        return b; }
      else if (! b.equals (next)) {
        return next.add (b);    }
      else {
    	  return next; }
    }   
    
    public void buildFonts (List v, byte[] digest, String base14CSSName, String base14PSName) 
    throws UnsupportedFontException, InvalidFontException
    {
      if (next != null) {
        next.buildFonts (v, digest, base14CSSName, base14PSName); }
    }
  }
      
  private static class TTCToLoad extends BlockToLoad {
    private OTByteArray ttcHeader;
     
    public TTCToLoad (OTByteArray header) {
      super (0,0);
      ttcHeader = header;
    }
    
    public OTByteArray getByteArray() { return null; } // not supported yet.
    
    public void load (FontInputStream in, BlockToLoad firstBlock)
    throws IOException, InvalidFontException, UnsupportedFontException {

      if (ttcHeader == null) {
        in.skipTo (offset);
        FontByteArray array = new FontByteArray(in, 12);
        ttcHeader = new OTByteArray (array); }
      else {
        in.skipTo (offset + 12); }
 
      if (Tag.misc_ttcf != ttcHeader.getuint32 (0)) {
        throw new InvalidFontException ("TTC header does not start with 'ttcf'"); }

      int majorVersion = ttcHeader.getuint16 (4);
      if (majorVersion != 1 && majorVersion != 2) {
        throw new UnsupportedFontException ("TTC major version " 
                    + majorVersion + " is not supported"); }

      int numFonts = ttcHeader.getuint32asint (8,
              "TTC fonts with 2^31 fonts or more not supported"); 
      FontByteArray array = new FontByteArray(in, 4 * numFonts);
      OTByteArray fontOffsets = new OTByteArray (array);
      for (int i = 0; i < numFonts; i++) {
        add (new FontToLoad (fontOffsets.getuint32 (4 * i))); }
      
      if (next != null) {
        next.load (in, next); }
     }
  }
  
  private static class FontToLoad extends BlockToLoad {
    java.util.ArrayList myTablesToLoad = new java.util.ArrayList /*TableToLoad*/ ();
    OTByteArray fontHeader;
    private final static int headerLength = 12;
    
    public FontToLoad (long offset) {
      super (offset, headerLength);
    }
    
    public FontToLoad (long offset, OTByteArray header) {
      super (offset, header.getSize());
      fontHeader = header;
    }
    
    public OTByteArray getByteArray() {return fontHeader;}
    
    // Fonts such FreesiaUPC (from MS) have tables of length 0, with
    // the same offset as some other table. Right now, we take care of these
    // below by not loading tables of length 0, but it means that we loose
    // the presence of the table (not a big deal, because those tables are 
    // anonymous, but still). 
    
    public void load (FontInputStream in, BlockToLoad firstBlock)
    throws IOException, InvalidFontException, UnsupportedFontException {

      if (fontHeader == null) {
        in.skipTo (offset);
        FontByteArray array = new FontByteArray(in, headerLength);
        fontHeader = new OTByteArray (array); }
      else {
        in.skipTo (offset + headerLength); }

      int sfntVersion = fontHeader.getint32 (0);
      if (   Tag.misc_OTTO != sfntVersion 
          && Tag.misc_1_0  != sfntVersion 
          && Tag.misc_true != sfntVersion) {
        throw new InvalidFontException ("FontData header does not have correct version (is: 0x" 
            + Integer.toHexString (sfntVersion) + ")"); }
      
      int numTables = fontHeader.getuint16 (4);

      FontByteArray array = new FontByteArray(in, 16*numTables);
      OTByteArray tableOffsets = new OTByteArray (array);
      for (int i = 0; i < numTables; i++) {
		long start = tableOffsets.getuint32 (16*i + 8);
        int size = tableOffsets.getuint32asint (16*i + 12,
                              "OT tables larger than 2^31 bytes not supported");
        int tag = tableOffsets.getint32 (i*16);
	    if (! Table.knownTable(tag)) {
			continue; } 
        if (size == 0) {
			// For now, reject tables of size zero. 
			// (part of the problem is fonts such as FreesiaUPC)
			continue; }
		if (! in.isValidOffset(start, size)) {
			continue; }

        TableToLoad ttl = new TableToLoad (start, size, tag);
        ttl = (TableToLoad) add (ttl);
        myTablesToLoad.add (ttl); }
      
      if (next != null) {
        next.load (in, firstBlock); }
    }

    public void buildFonts (List v, byte[] digest, String base14CSSName, String base14PSName) 
    throws InvalidFontException, UnsupportedFontException {
      java.util.Map /*int, Table */ myTables = new java.util.HashMap ();
      for (Iterator it = myTablesToLoad.iterator (); it.hasNext (); ) {
        TableToLoad ttl = (TableToLoad) it.next ();
        myTables.put (new Integer (ttl.tag), ttl.table); }
      OpenTypeFont f = new OpenTypeFont (myTables, digest, base14CSSName, base14PSName);
      v.add (f);
      if (next != null) {
        next.buildFonts (v, digest, base14CSSName, base14PSName); }
    }
  }
  
  private static class TableToLoad extends BlockToLoad {
    public int length;
    public Table table;
    public int tag;
    
    public TableToLoad (long offset, int length, int tag) {
      super (offset, length);
      this.length = length;
      this.tag = tag;
    }
    
    public OTByteArray getByteArray() { return table.data;}
    
    public void load (FontInputStream in, BlockToLoad firstTable)
    throws IOException, InvalidFontException, UnsupportedFontException {
    	FontByteArray theArray = null;
    	if (in.getCurrentOffset() <= offset) {
			in.skipTo (offset);
			theArray = new FontByteArray(in, length);
    	}
    	else
    	{
    		
    		BlockToLoad blockToCopy = firstTable;
    		OTByteArrayBuilder builder = OTByteArray.getOTByteArrayBuilderInstance(length);
    		
    		int previousBufferCount = (int)(in.getCurrentOffset() - offset);
    		if (length < previousBufferCount)
    			previousBufferCount = length;
    			
    		int offsetCopied = 0;
    		
    		// find the first block from which we need to copy
    		while (blockToCopy.offset + blockToCopy.length < offset) {
    			if (blockToCopy.next == this){
    				 if (Table.knownTable(tag)) {
    					 throw new InvalidFontException("Cannot find overlapped data");}
    				 else{
    					 // The needed data was previously tossed on the floor, but this table is one
    					 // we don't care about. Continue on, throwing this whole table on the floor.
    					 blockToCopy = null; 
    					 break; }}
    			else {
    				blockToCopy = blockToCopy.next;}}
    		
    		if (blockToCopy != null) {
	    		// Copy bytes from other tables.
	    		while (offsetCopied < previousBufferCount && blockToCopy != null && blockToCopy.offset <= offsetCopied + offset)
	    		{
	    			OTByteArray copyFrom = blockToCopy.getByteArray();
	    			if (copyFrom == null)
	        			throw new InvalidFontException("Overlapping data cannot be accessed.");
	    			int start = (int)(offsetCopied + offset- blockToCopy.offset);
	    			int fromLength = Math.min(previousBufferCount, blockToCopy.length - start);
	    			builder.replace(offsetCopied, copyFrom, start, fromLength);
	    			blockToCopy = blockToCopy.next;
	    			offsetCopied += fromLength;
	    		}
	    		
	    		if (offsetCopied < previousBufferCount)
	    		{
	    			if (Table.knownTable(tag))
	    				throw new InvalidFontException("Data could not be copied");
	    		}
	    		else 
    			{
	    			if (length > offsetCopied)
	    				builder.append(in, length - offsetCopied, offsetCopied);
	    		
	    			theArray = builder.toOTByteArray();
	    		}
    		}
    	}
    	
    	if (theArray != null) {
    		table = Table.factory (tag, theArray);}
      
    	if (next != null) {
    		next.load (in, firstTable); }
    }
    
    public boolean equals (Object o) {
      if (o == this) {
        return true; }
      else if (o == null) {
        return false; }
      else if (! ( o instanceof TableToLoad)) {
        return false; }
      else {
        TableToLoad ot = (TableToLoad) o;
        return (this.length == ot.length && this.tag == ot.tag && this.offset == ot.offset); }
    }
    
    public String toString () {
      return "" + length + "/" + tag + "/" + offset;
    }
    
    public int hashCode () {
      return toString ().hashCode ();
    }
  }

  /**
   * Given an input stream, creates OpenType FontData objects that represent it.
   * 
   * Note that fonts are parsed at this point.
   * 
   * @param in The input stream
   * @return An array of OpenType fonts
   * @throws IOException Thrown if the stream cannot be read
   * @throws InvalidFontException Thrown if the stream does not represent a valid OpenType font.
   * @throws UnsupportedFontException thrown if the stream contains information indicating that AFE
   * does not support this class of fonts.
   */
  public static OpenTypeFont[] load (FontInputStream in) 
  throws IOException, InvalidFontException, UnsupportedFontException {
    return load (in, null, null);
  }
  
  public static OpenTypeFont[] load (FontInputStream in, String base14CSSName, String base14PSName)
  throws IOException, InvalidFontException, UnsupportedFontException {
    
	FontByteArray array = new FontByteArray(in, 12);
    OTByteArray header = new OTByteArray (array);
    
    BlockToLoad thingsToLoad;
    if (header.getint32 (0) == Tag.misc_ttcf) {
      thingsToLoad = new TTCToLoad (header); }
    else { // must be a font header
      thingsToLoad = new FontToLoad (0, header); }
    thingsToLoad.load (in, thingsToLoad);
    
    byte[] digest = in.getDigest ();
    
    List v = new ArrayList ();
    thingsToLoad.buildFonts (v, digest, base14CSSName, base14PSName);
    OpenTypeFont[] res = new OpenTypeFont [v.size ()];
    v.toArray (res);
    return res;
  }
    
  /**
   * @return the number of starting bytes necessary to identify
   * that a stream appears to be an opentype font.
   */
  public static int getNumBytesNeededToIdentify()
  {
    return 4;
  }
  
  /**
   * Given the starting bytes of a font stream, returns true
   * if the bytes indicate that the font is an opentype font.
   * 
   * @param startingBytes This inital bytes of an input stream.
   * @return true iff the font appears to be an OpenType font. 
   */
  public static boolean isOpenType(byte[] startingBytes)  {
    int result = 0;
    for (int i = 0; i < 4; i++) {
      result = (result << 8) | startingBytes[i]; }

    if (   (result == Tag.misc_ttcf)
        || (result == Tag.misc_1_0)
        || (result == Tag.misc_OTTO)
        || (result == Tag.misc_true)) {
      return true; }

    return false; 
  }
}


