/*
 *
 *	File: AutoColor.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.cff.AutoColor;
import com.adobe.fontengine.font.cff.NonOverlappingOutlineConsumer;

/* CFF zone hint generator for TrueType to CFF conversion */
public class ZoneHint
{
	static final int kBotZoneBaseline = 0;		// all alphabets except kBotZoneDescender, J, Q
	static final int kBotZoneBaseline5 = 1;		// 3, 4, 5, 7, 9
	static final int kBotZoneBaseline6 = 2;		// 0, 1, 2, 6, 8
	static final int kBotZoneDescender = 3;		// g, j, p, q, y
	static final int kBotZoneColon = 4;		// :, ;
	static final int kTopZoneCapHeight = 5;		// all capital
	static final int kTopZoneAscender = 6;		// b, d, f, h, k, l
	static final int kTopZonexHeight = 7;		// a, c, e, g, m, n, o, p, q, r, s, u, v, w, x, y, z
	static final int kTopZoneijHeight = 8;		// i, j
	static final int kTopZoneHeight5 = 9;		// 0, 1, 2, 3, 4, 5, 7, 9
	static final int kTopZoneHeight6 = 10;		// 6, 8
	static final int kTopZoneColon = 11;		// :, ;
	static final int kZoneNone = 12;

	static final int kBotZoneFirst = kBotZoneBaseline;
	static final int kTopZoneFirst = kTopZoneCapHeight;
	static final int kBotZoneLast = kBotZoneColon;
	static final int kTopZoneLast = kTopZoneColon;
	static final int kBotZoneCount = kBotZoneLast - kBotZoneFirst + 1;
	static final int kTopZoneCount = kTopZoneLast - kTopZoneFirst + 1;

	/* Magic values from CFFHinter.cpp */
	static final double kMaxZoneHeight = 36.0;
	static final int kBlueFuzz = 0;
	static final double kDefaultBlueShift = 7.0;
	static final int kStrayCountDenom = 3;

	private Zone mVStems = new Zone(kZoneNone);
	private Zone mHStems = new Zone(kZoneNone);
	private Zone[] mTopZones = new Zone[kTopZoneCount];
	private Zone[] mBotZones = new Zone[kBotZoneCount];
	private ArrayList mTopZonesSorted;
	private ArrayList mBotZonesSorted;
	boolean mTopZoneWasSet;
	boolean mBotZoneWasSet;
	private int mCharTopExtreme;
	private int mCharBotExtreme;
	private int mScaledMaxZoneHeight;
	private int[] mTopZoneData;
	private int[] mBotZoneData;
	private int[] mBlueValues;
	private int[] mOtherBlues;
	private int mBlueFuzz;
	private double mBlueShift;
	private double mStdVStem;
	private double mStdHStem;
	private double mBlueScale;	

	public ZoneHint(OpenTypeFont font)
		throws InvalidFontException, UnsupportedFontException
	{
		double unitsPerEm = font.getUnitsPerEmX();
		mScaledMaxZoneHeight = (int)((kMaxZoneHeight * unitsPerEm) / 1000.0);
		AutoColor autoColor = new AutoColor(null, unitsPerEm,
						    AutoColor.AC_HINTSUB | AutoColor.AC_FIXWINDING | AutoColor.AC_GENERATEVSTEMS,
						    false, true, null, null);
		NonOverlappingOutlineConsumer noc = new NonOverlappingOutlineConsumer(autoColor, unitsPerEm);
		for (int i = 0; i < kTopZoneCount; i++) {
			mTopZones[i] = new Zone(kTopZoneFirst + i);
		}
		for (int i = 0; i < kBotZoneCount; i++) {
			mBotZones[i] = new Zone(kBotZoneFirst + i);
		}
		for (int i = 0; i < gZoneTestCharList.length; i++) {
			ZoneTestChar curChar = gZoneTestCharList[i];
			int curGID = font.getGlyphForChar(curChar.mCode);
			if (curGID > 0) {
				try {
					mCharTopExtreme = mCharBotExtreme = Integer.MAX_VALUE;
					mTopZoneWasSet = mBotZoneWasSet = false;
					double width = font.getHorizontalAdvance(i);
					autoColor.newGlyph(curGID, width, curChar.mCode);
					font.getGlyphOutline(curGID, noc, TTParser.APPROX_PATH);
					int[] rawData = autoColor.reportZones();
					int left, right, top, bottom;
					for (int j = 0; j < rawData.length;) {
						int type = rawData[j++];
						switch (type) {
						case AutoColor.ZN_VSTEM:
							right = rawData[j++];
							left = rawData[j++];
							addVStem(right, left, curChar);
							break;
						case AutoColor.ZN_HSTEM:
							top = rawData[j++];
							bottom= rawData[j++];
							addHStem(top, bottom, curChar);
							break;
						case AutoColor.ZN_CHAREXTREME:
							top = rawData[j++];
							bottom = rawData[j++];
							addCharExtreme(top, bottom);
							break;
						case AutoColor.ZN_CHARZONE:
							top = rawData[j++];
							bottom = rawData[j++];
							addCharZone(top, bottom, curChar);
							break;
						case AutoColor.ZN_ZONE:
							j += 2;	/* top, bottom */
							break;
						}
					}
					if (curChar.mTopZone < kTopZoneLast && !mTopZoneWasSet && mCharTopExtreme != Integer.MAX_VALUE)
						mTopZones[curChar.mTopZone - kTopZoneFirst].addValue(mCharTopExtreme);
					if (curChar.mBotZone < kBotZoneLast && !mBotZoneWasSet && mCharBotExtreme != Integer.MAX_VALUE)
						mBotZones[curChar.mBotZone - kBotZoneFirst].addValue(mCharBotExtreme);

				} catch (InvalidGlyphException e) {
				}
			}
		}
		mTopZonesSorted = sortZones(mTopZones);
		mBotZonesSorted = sortZones(mBotZones);
		Zone baseZone = findZone(mBotZonesSorted, kBotZoneBaseline);
		if (baseZone != null) {
			int maxZoneHeight = 0;
			mTopZoneData = new int[mTopZonesSorted.size() * 2];
			for (int i = 0; i < mTopZonesSorted.size(); i++) {
				Zone zone = (Zone)mTopZonesSorted.get(i);
				if (zone.getHeight() > maxZoneHeight) {
					maxZoneHeight = zone.getHeight();
				}
				mTopZoneData[i * 2] = zone.mMinValue;
				mTopZoneData[(i * 2) + 1] = zone.mMaxValue;
			}
			mBotZoneData = new int[mBotZonesSorted.size() * 2];
			for (int i = 0; i < mBotZonesSorted.size(); i++) {
				Zone zone = (Zone)mBotZonesSorted.get(i);
				if (zone.getHeight() > maxZoneHeight) {
					maxZoneHeight = zone.getHeight();
				}
				mBotZoneData[i * 2] = zone.mMinValue;
				mBotZoneData[(i * 2) + 1] = zone.mMaxValue;
			}
			mBlueValues = new int[mTopZoneData.length + 2];
			mBlueValues[0] = baseZone.mMinValue;
			mBlueValues[1] = baseZone.mMaxValue;
			for (int i = 0; i < mTopZoneData.length; i++) {
				mBlueValues[i + 2] = mTopZoneData[i];
			}
			mOtherBlues = new int[mBotZoneData.length - 2];
			for (int i = 0, j = 0; i < mBotZonesSorted.size(); i++) {
				Zone zone = (Zone)mBotZonesSorted.get(i);
				if (zone != baseZone) {
					mOtherBlues[j++] = zone.mMinValue;
					mOtherBlues[j++] = zone.mMaxValue;
				}
			}
			if (maxZoneHeight > 0) {
				mBlueScale = (1.0 / maxZoneHeight) * 0.8;
			}
			if (unitsPerEm != 1000) {
				mBlueShift = kDefaultBlueShift*unitsPerEm/1000.0;
			}
			mBlueFuzz = kBlueFuzz;
		}
		int stemMode = mVStems.getMode();
		if (stemMode == Integer.MAX_VALUE)
			stemMode = (int)(mVStems.getAverage() + 0.5);
		mStdVStem = stemMode;

		stemMode = mHStems.getMode();
		if (stemMode == Integer.MAX_VALUE)
			stemMode = (int)(mHStems.getAverage() + 0.5);
		mStdHStem = stemMode;
	}

	ArrayList sortZones(Zone[] zoneList)
	{
		ArrayList result = new ArrayList();
		for (int i = 0; i < zoneList.length; i++) {
			Zone zone = zoneList[i];
			if (zoneList[i].mNumValues != 0) {
				for (int j = 0; j <= result.size(); j++) {
					if (j == result.size() || zone.mMinValue <= ((Zone)result.get(j)).mMinValue) {
						result.add(j, zone);
						break;
					}
				}
			}
		}
		for (int i = 0; i < result.size();) {
			boolean incZone = true;
			Zone zone1 = (Zone)result.get(i);
			if (i + 1 < result.size()) {
				Zone zone2 = (Zone)result.get(i + 1);
				if (zone1.mMaxValue + (2 * kBlueFuzz) + 1 >= zone2.mMinValue ||
				    zone1.mMinValue + mScaledMaxZoneHeight >= zone2.mMaxValue) {
					Iterator iter = zone2.mHistValueList.keySet().iterator();
					while (iter.hasNext()) {
						Integer iValue = (Integer)iter.next();
						Integer iCount = (Integer)zone2.mHistValueList.get(iValue);
						zone1.addValue(iValue.intValue(), iCount.intValue());
					}
					if (zone1.mZoneType > zone2.mZoneType) {
						zone1.mZoneType = zone2.mZoneType;
					}
					result.remove(i + 1);
					incZone = false;
				}
			}
			int strayCount = zone1.mNumValues / kStrayCountDenom;
			while (zone1.getHeight() > mScaledMaxZoneHeight && strayCount > 0) {
				double average = zone1.getAverage();
				double midPoint = (zone1.mMinValue + zone1.mMaxValue) / 2.0;
				if (average >= midPoint) {
					zone1.removeValue(zone1.mMinValue);
				} else {
					zone1.removeValue(zone1.mMaxValue);
				}
				strayCount--;
			}
			if (zone1.getHeight() > mScaledMaxZoneHeight) {
				result.remove(i);
				incZone = false;
			}
			if (incZone) {
				i++;
			}
		}
		return result;
	}

	Zone findZone(ArrayList zoneArray, int zoneType)
	{
		for (int i = 0; i < zoneArray.size(); i++) {
			Zone zone = (Zone)zoneArray.get(i);
			if (zone.mZoneType == zoneType) {
				return zone;
			}
		}
		return null;
	}

	public int[] getTopZones()
	{
		return (mTopZoneData != null) ? mTopZoneData : new int[0];
	}

	public int[] getBottomZones()
	{
		return (mBotZoneData != null) ? mBotZoneData : new int[0];
	}

	public double[] getBlueValues()
	{
		int len = (mBlueValues != null) ? mBlueValues.length : 0;
		if (len != 0) {
			double[] retVal = new double[len];
			for (int i = 0; i < len; i++) {
				retVal[i] = mBlueValues[i];
			}
			return retVal;
		}
		return null;
	}

	public double[] getOtherBlues()
	{
		int len = (mOtherBlues != null) ? mOtherBlues.length : 0;
		if (len != 0) {
			double[] retVal = new double[len];
			for (int i = 0; i < len; i++) {
				retVal[i] = mOtherBlues[i];
			}
			return retVal;
		}
		return null;
	}

	public int getBlueFuzz()
	{
		return mBlueFuzz;
	}

	public double getBlueShift()
	{
		return mBlueShift;
	}

	public double getStdVStem()
	{
		return mStdVStem;
	}

	public double getStdHStem()
	{
		return mStdHStem;
	}

	public double getBlueScale()
	{
		return mBlueScale;
	}

	private void addVStem(int right, int left, ZoneTestChar testChar)
	{
		if (testChar.mCode >= 'a' && testChar.mCode <= 'z')
			mVStems.addValue(right - left);
	}

	private void addHStem(int top, int bottom, ZoneTestChar testChar)
	{
		int diff = top-bottom;
		mHStems.addValue(top - bottom);
	}

	private void addCharExtreme(int top, int bottom)
	{
		mCharTopExtreme = top;
		mCharBotExtreme = bottom;
	}

	private void addCharZone(int top, int bottom, ZoneTestChar testChar)
	{
		if (testChar.mTopZone != kZoneNone) {
			if (mCharTopExtreme == Integer.MAX_VALUE || top + mScaledMaxZoneHeight >= mCharTopExtreme) {
				mTopZones[testChar.mTopZone - kTopZoneFirst].addValue(top);
				mTopZoneWasSet = true;
			}
		}
		if (testChar.mBotZone != kZoneNone) {
			if (mCharBotExtreme == Integer.MAX_VALUE || bottom - mScaledMaxZoneHeight <= mCharBotExtreme) {
				mBotZones[testChar.mBotZone - kBotZoneFirst].addValue(bottom);
				mBotZoneWasSet = true;
			}
		}
	}

	private final class Zone
	{
		int mZoneType;
		int mNumValues;
		int mMaxValue;
		int mMinValue;
		HashMap mHistValueList = new HashMap();

		Zone(int zoneType)
		{
			mZoneType = zoneType;
		}

		void addValue(int value)
		{
			addValue(value, 1);
		}

		void addValue(int value, int increment)
		{
			Integer iValue = new Integer(value);
			Integer iCount = (Integer)mHistValueList.get(iValue);
			int count = (iCount != null) ? iCount.intValue() : 0;
			mHistValueList.put(iValue, new Integer(count + increment));
			setMinMaxWithValue(value);
			mNumValues += increment;
		}

		void removeValue(int value)
		{
			Integer iValue = new Integer(value);
			Integer iCount = (Integer)mHistValueList.get(iValue);
			if (iCount != null) {
				int count = iCount.intValue();
				if (--count == 0) {
					mHistValueList.remove(iValue);
					if (value == mMinValue || value == mMaxValue) {
						mMinValue = mMaxValue = 0;
						Iterator iter = mHistValueList.keySet().iterator();
						boolean first = true;
						while (iter.hasNext()) {
							iValue = (Integer)iter.next();
							if (first) {
								mMinValue = mMaxValue = iValue.intValue();
								first = false;
							}
							setMinMaxWithValue(iValue.intValue());
						}
					}
				} else {
					mHistValueList.put(iValue, new Integer(count));
				}
				mNumValues--;
			}
		}

		int getMode()
		{
			int value = 0;
			int valueCount = 0;
			int count = 0;
			Iterator iter = mHistValueList.keySet().iterator();
			while (iter.hasNext()) {
				Integer iValue = (Integer)iter.next();
				Integer iCount = (Integer)mHistValueList.get(iValue);

				if (iCount.intValue() > count) {
					value = iValue.intValue();
					count = iCount.intValue();
					valueCount = 1;
				} else if (iCount.intValue() == count) {
					valueCount++;
				}
			}

			if (valueCount <=1) {
				return (count == 1) ? Integer.MAX_VALUE : value;
			}
			
			int[] valueList = new int[valueCount];
			int n = 0;
			iter = mHistValueList.keySet().iterator();
			while (iter.hasNext()) {
				Integer iValue = (Integer)iter.next();
				Integer iCount = (Integer)mHistValueList.get(iValue);
				if (iCount.intValue() == count) {
					int i = 0;
					for (; i<n ; i++)
					{
						if (iValue.intValue() < valueList[i])
						   break;	
					}
					System.arraycopy(valueList, i, valueList, i+1, n-i);					
					valueList[i] = iValue.intValue();
					n++;
				}
			}

			return valueList[valueCount/2];
		}

		double getAverage()
		{
			double total = 0;
			Iterator iter = mHistValueList.keySet().iterator();
			while (iter.hasNext()) {
				Integer iValue = (Integer)iter.next();
				Integer iCount = (Integer)mHistValueList.get(iValue);
				total += (iValue.intValue() * iCount.intValue());
			}
			return total / mNumValues;
		}

		void setMinMaxWithValue(int value)
		{
			if (mNumValues == 0 || value > mMaxValue) {
				mMaxValue = value;
			}
			if (mNumValues == 0 || value < mMinValue) {
				mMinValue = value;
			}
		}

		int getHeight()
		{
			return mMaxValue - mMinValue;
		}
	}

	private static final class ZoneTestChar
	{
		final char mCode;
		final int mTopZone;
		final int mBotZone;

		private ZoneTestChar(char code, int topZone, int botZone)
		{
			mCode = code;
			mTopZone = topZone;
			mBotZone = botZone;
		}
	}

	private static final ZoneTestChar[] gZoneTestCharList = {
		new ZoneTestChar('.', kZoneNone, kBotZoneBaseline),
		new ZoneTestChar(':', kTopZoneColon, kBotZoneColon),
		new ZoneTestChar(';', kTopZoneColon, kBotZoneColon),
		new ZoneTestChar('0', kTopZoneHeight5, kBotZoneBaseline6),
		new ZoneTestChar('1', kTopZoneHeight6, kBotZoneBaseline6),
		new ZoneTestChar('2', kTopZoneHeight6, kBotZoneBaseline6),
		new ZoneTestChar('3', kTopZoneHeight5, kBotZoneBaseline5),
		new ZoneTestChar('4', kTopZoneHeight5, kBotZoneBaseline5),
		new ZoneTestChar('5', kTopZoneHeight5, kBotZoneBaseline5),
		new ZoneTestChar('6', kTopZoneHeight6, kBotZoneBaseline6),
		new ZoneTestChar('7', kTopZoneHeight5, kBotZoneBaseline5),
		new ZoneTestChar('8', kTopZoneHeight6, kBotZoneBaseline6),
		new ZoneTestChar('9', kTopZoneHeight5, kBotZoneBaseline5),
		new ZoneTestChar('A', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('B', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('C', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('D', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('E', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('F', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('G', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('H', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('I', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('J', kTopZoneCapHeight, kZoneNone),
		new ZoneTestChar('K', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('L', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('M', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('N', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('O', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('P', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('Q', kTopZoneCapHeight, kZoneNone),
		new ZoneTestChar('R', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('S', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('T', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('U', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('V', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('W', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('X', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('Y', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('Z', kTopZoneCapHeight, kBotZoneBaseline),
		new ZoneTestChar('a', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('b', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('c', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('d', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('e', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('f', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('g', kTopZonexHeight, kBotZoneDescender),
		new ZoneTestChar('h', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('i', kTopZoneijHeight, kBotZoneBaseline),
		new ZoneTestChar('j', kTopZoneijHeight, kBotZoneDescender),
		new ZoneTestChar('k', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('l', kTopZoneAscender, kBotZoneBaseline),
		new ZoneTestChar('m', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('n', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('o', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('p', kTopZonexHeight, kBotZoneDescender),
		new ZoneTestChar('q', kTopZonexHeight, kBotZoneDescender),
		new ZoneTestChar('r', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('s', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('t', kZoneNone, kBotZoneBaseline),
		new ZoneTestChar('u', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('v', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('w', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('x', kTopZonexHeight, kBotZoneBaseline),
		new ZoneTestChar('y', kTopZonexHeight, kBotZoneDescender),
		new ZoneTestChar('z', kTopZonexHeight, kBotZoneBaseline)
	};
}
