/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2009 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.dom;

import java.util.HashMap;
import java.util.Map;

import org.w3c.dom.DOMException;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.adobe.xfa.ut.StringUtils;

/**
 * Implements the W3C DOM NamedNodeMap interface.  This rather brute
 * force implementation stores two internal maps, one for name and the
 * other for namespace.  It also extends the NodeListImpl class to
 * support iterating through the node list contents.
 * @exclude from published api.
 */
class NamedNodeMapImpl extends NodeListImpl implements NamedNodeMap {
	private static final class NodeKey {
		NodeKey (String first, String second) {
			mFirst = first;
			mSecond = second;
		}

		NodeKey (String first) {
			mFirst = first;
			mSecond = null;
		}

		public boolean equals (Object other) {
			if (other == this) {
				return true;
			}
			
			if (other == null) {
				return false;
			}
			
			if (other.getClass() != NodeKey.class) {
				return false;
			}
			
			NodeKey otherKey = (NodeKey) other;
			return StringUtils.equalsWithNull (mFirst, otherKey.mFirst)
				&& StringUtils.equalsWithNull (mSecond, otherKey.mSecond);
		}

		public int hashCode () {
			int result = mFirst.hashCode();
			if (mSecond != null) {
				result ^= mSecond.hashCode();
			}
			return result;
		}

		final String mFirst;
		final String mSecond;
	}

	private final Map<NodeKey, NodeImpl> mNameMap = new HashMap<NodeKey, NodeImpl>();
	private final Map<NodeKey, NodeImpl> mNSMap = new HashMap<NodeKey, NodeImpl>();

	NamedNodeMapImpl (int initialCapacity) {
		super (initialCapacity);
	}

	public Node getNamedItem(String name) {
		return mNameMap.get (new NodeKey (name));
	}

	public Node getNamedItemNS(String namespaceURI, String localName) throws DOMException {
		return mNSMap.get (new NodeKey (namespaceURI, localName));
	}

	public Node removeNamedItem(String name) throws DOMException {
		throw new DOMException (DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
	}

	public Node removeNamedItemNS(String namespaceURI, String localName) throws DOMException {
		throw new DOMException (DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
	}

	public Node setNamedItem(Node arg) throws DOMException {
		throw new DOMException (DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
	}

	public Node setNamedItemNS(Node arg) throws DOMException {
		throw new DOMException (DOMException.NO_MODIFICATION_ALLOWED_ERR, "");
	}

	void addNode (NodeImpl node) {
		super.addNode (node);
		String name = node.getNodeName();
		if (name != null) {
			mNameMap.put (new NodeKey (name), node);
		}
		String ns = node.getNamespaceURI();
		if (ns != null) {
			mNSMap.put (new NodeKey (ns, node.getLocalName()), node);
		}
	}
}
