/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import com.day.cq.wcm.designimporter.ErrorCodes;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.ccil.cowan.tagsoup.AttributesImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.Attributes;

import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;
import com.day.cq.wcm.designimporter.util.TagUtils;

/**
 * The text component tag handler is used to translate the text component div(or an equivalent markup)
 * into a CQ text component.
 *
 * <p>
 * This tag handler is registered against the markup of type < div|span|p data-cq-component="text" >
 * (Read as a div, span, p tag with the attribute data-cq-component equal to "text"). That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.TextComponentTagHandlerFactory
 */
public class TextComponentTagHandler extends AbstractTagHandler {

    private String suffix;

    private Logger logger = LoggerFactory.getLogger(TextComponentTagHandler.class);

    private String resourceType;

    @Override
    public void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        super.beginHandling(uri, localName, qName, atts);

        if ("p".equalsIgnoreCase(localName)) {
            AttributesImpl modAttibutes = new AttributesImpl(atts);
            int index = atts.getIndex("data-cq-component");
            if (index > -1) modAttibutes.removeAttribute(index);

            startElement(uri, localName, qName, modAttibutes);
        }
    }

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        super.startElement(uri, localName, qName, atts);

        if (!(delegate instanceof DefaultTagHandler))
            throw new UnsupportedTagContentException(ErrorCodes.NESTED_COMPONENT_DIV_TAG_ENCOUNTERED_INSIDE_TEXT_COMPONENT_DIV_TAG);
    }

    private String getDesignPath() {
        try {
            String designPath = designImporterContext.designNode.getPath();
            return designPath;
        } catch (RepositoryException e) {
            logger.error("Error occurred while trying to access the path of the design node", e);
        }
        return "";
    }

    @Override
    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        super.endHandling(uri, localName, qName);

        if ("p".equalsIgnoreCase(localName)) {
            endElement(uri, localName, qName);
        }

        Map<String, Object> base = new HashMap<String, Object>();
        base.put("text", processInnerHtml(htmlBuffer.toString()));
        base.put("textIsRich", true);
        ValueMap properties = new ValueMapDecorator(base);
        suffix = designImporterContext.componentSuffixGenerator.getSuffix("text");
        PageComponent textComponent = pageBuilder.createComponent(resourceType, properties, getNameHint());
        getPageComponents().add(textComponent);
    }

    private String processInnerHtml(String string) {
        return string.replace("<%= currentDesign.getPath() %>", getDesignPath());
    }

    @Override
    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP)
            return true;
        else
            return false;
    }

    private String getNameHint() {
        return "text" + suffix;
    }

    @Override
    public Object getContent(HTMLContentType htmlContentType) {

        if (htmlContentType == HTMLContentType.MARKUP) {
            String cqIncludeJspTag = "<sling:include path=" + "\"" + getNameHint() + "\"" + "/>";

            return componentDivStartTag + cqIncludeJspTag + TagUtils.getMatchingEndTag(componentDivStartTag);

        }

        return null;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }

}
