/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.query;

import com.day.crx.statistics.Report;
import com.day.crx.statistics.PathBuilder;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import java.util.Calendar;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

import org.apache.jackrabbit.util.Text;

/**
 * <code>MostPopularQueriesReport</code> implements a report that returns the
 * most popular queries in a given period.
 *
 * @author mreutegg
 */
public class MostPopularQueriesReport extends Report {

    /**
     * Number of queries to return at most. Default value: top 10 queries.
     */
    private int size = 10;

    /**
     * The end date of the period. Default value: current date.
     */
    private long endDate = System.currentTimeMillis();

    /**
     * Per default cover last 30 days
     */
    private int period = 30;

    /**
     * Creates a new {@link MostPopularQueriesReport}.
     *
     * @param dataPath the path where the query statistics are stored.
     */
    public MostPopularQueriesReport(String dataPath) {
        super(dataPath);
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Query <code>String</code></li>
     * <li><code>Long</code> count (how may times the query was executed)</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        long startDate = getStartDate();
        Map queries = new HashMap();
        PathBuilder builder = new QueryPathBuilder();
        Query dummy = new Query(getDataPath(), "dummy", 0, 0);
        StringBuffer path = new StringBuffer();
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(startDate);
        for (; cal.getTimeInMillis() <= endDate; cal.add(Calendar.DAY_OF_MONTH, 1)) {
            path.setLength(0);
            dummy.setTimestamp(cal.getTimeInMillis());
            builder.formatPath(dummy, path);
            path.setLength(path.lastIndexOf("/"));
            if (session.itemExists(path.toString())) {
                for (NodeIterator nodes = ((Node) session.getItem(
                        path.toString())).getNodes(); nodes.hasNext(); ) {
                    Node qn = nodes.nextNode();
                    String query = Text.unescapeIllegalJcrChars(qn.getName());
                    // exclude queries that did not return results
                    if (qn.getProperty("numResults").getLong() > 0) {
                        long count = qn.getProperty("count").getLong();
                        Long c = (Long) queries.get(query);
                        if (c != null) {
                            count += c.longValue();
                        }
                        queries.put(query, new Long(count));
                    }
                }
            }
        }
        List result = new ArrayList();
        for (Iterator entries = queries.entrySet().iterator(); entries.hasNext(); ) {
            Map.Entry entry = (Map.Entry) entries.next();
            result.add(new Object[]{entry.getKey(), entry.getValue()});
        }
        Collections.sort(result, new Comparator() {
            public int compare(Object o1, Object o2) {
                Object[] e1 = (Object[]) o1;
                Object[] e2 = (Object[]) o2;
                int comp = ((Comparable) e2[1]).compareTo(e1[1]);
                if (comp != 0) {
                    return comp;
                }
                return ((String) e1[0]).toLowerCase().compareTo(((String) e2[0]).toLowerCase());
            }
        });
        result = result.subList(0, Math.min(getSize(), result.size()));
        return result.iterator();
    }

    //-----------------------------< properties >-------------------------------

    /**
     * @return the maximum number of queries to return.
     */
    public int getSize() {
        return size;
    }

    /**
     * @param size the maximum number of queries to return.
     */
    public void setSize(int size) {
        this.size = size;
    }

    /**
     * @return the start date of the report period (inclusive).
     */
    public long getStartDate() {
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(getEndDate());
        cal.add(Calendar.DAY_OF_MONTH, - (getPeriod() - 1));
        return cal.getTimeInMillis();
    }

    /**
     * @param end the end date of the report period (inclusive).
     */
    public void setEndDate(long end) {
        this.endDate = end;
    }

    /**
     * @return the end date of the report period (inclusive).
     */
    public long getEndDate() {
        return endDate;
    }

    /**
     * @return the report period in number of days.
     */
    public int getPeriod() {
        return period;
    }

    /**
     * @param period the report period in number of days.
     */
    public void setPeriod(int period) {
        this.period = period;
    }
}
