/*
*
*	File: OTSWFFont3Description.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2008 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.opentype;

import java.io.UnsupportedEncodingException;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.UnsupportedFontException;

class OTFSWFFont3Description
  implements com.adobe.fontengine.font.SWFFontDescription
  {
	/**
	 * 
	 */
	private final OpenTypeFont theFont;
	private final boolean wasEmbedded;
	
	OTFSWFFont3Description(OpenTypeFont font, boolean wasEmbedded) throws InvalidFontException, UnsupportedFontException {
		this.wasEmbedded = wasEmbedded;
		this.theFont = font;
	}

	public boolean canDisplay(char c) throws UnsupportedFontException, InvalidFontException {
		try {
			return (theFont.cmap.unicodeChar2glyph(c) != 0);
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}
	}

	public double getAscent() throws InvalidFontException{
		if (theFont.os2 != null)
			return theFont.os2.getWinAscent();
		else if (theFont.hhea != null)
			return theFont.hhea.getAscender();
		else
			return 0;
    }

	
	public double getDescent() throws InvalidFontException {
		if (theFont.os2 != null)
			return theFont.os2.getWinDescent();
		else if (theFont.hhea != null)
			return -theFont.hhea.getDescender();   
		else
			return 0;
	}

	public double getLineGap() throws InvalidFontException {
		if (theFont.os2 != null)
			return theFont.os2.getTypoLineGap();
		else if (theFont.hhea != null)
			return theFont.hhea.getLineGap();
		else
			return 0;
		
	}

	public double getEmScale() throws InvalidFontException {
		if (theFont.head != null)
			return theFont.head.getUnitsPerEm();
		else
			throw new InvalidFontException("Cannot determine em");
	}

	/**
	 * Return the first well-formed family name found.
	 */
	public String getFamily() throws InvalidFontException, UnsupportedFontException {
		return theFont.name.getFirstName(Name.PredefinedNames.FONT_FAMILY);
	}

	public int getFirstChar() throws InvalidFontException, UnsupportedFontException {
		if (theFont.os2 != null) {
			int first = theFont.os2.getFirstChar();
			
			try {
				first = theFont.cmap.removeSymbolModifier(first);
			} catch (NullPointerException e) {
				throw new InvalidFontException("Missing required table", e);
			}
			
			return first; }
		return 0;
	}
	
	public int getLastChar() throws InvalidFontException, UnsupportedFontException {

		try {
			if (theFont.cmap.symbolSubtableOffset != -1)
				return theFont.cmap.getHighestMappedCode(theFont.cmap.symbolSubtableOffset, true);
			
			if (theFont.cmap.unicodeSubtableOffset != -1)
				return theFont.cmap.getHighestMappedCode(theFont.cmap.unicodeSubtableOffset, true);
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}
			
		// Fonts that have only legacy encoded cmaps often set their os2 lastchar
		// to be the last character in some legacy encoding. In these cases, the
		// only way to figure out the last char is to look up the unicode
		// chars until you find one that works, converting all those characters as
		// you go. Since the expected use of this api is to prevent running characters,
		// we aren't saving anything by doing that....so we will just return 0xffff.
		return 0xffff;
	}
	
	public double getHorizontalAdvance(char c) throws InvalidFontException, UnsupportedFontException {
		try {
			return theFont.hmtx.getHorizontalAdvance(theFont.cmap.unicodeChar2glyph(c));
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}
	}

	public int getNumGlyphs() throws InvalidFontException,UnsupportedFontException {
		return theFont.getNumGlyphs();
	}

	public String getPostscriptName() throws InvalidFontException, UnsupportedFontException {
		if (theFont.name == null)
			return null;
		return theFont.name.getFirstName(Name.PredefinedNames.POSTSCRIPT_NAME);
	}
	
	public String getCopyright() throws InvalidFontException, UnsupportedFontException {
		if (theFont.name == null)
			return null;
		return theFont.name.getFirstName(Name.PredefinedNames.COPYRIGHT_NOTICE);
	}

	public String getTrademark() throws InvalidFontException, UnsupportedFontException {
		if (theFont.name == null)
			return null;
		return theFont.name.getFirstName(Name.PredefinedNames.TRADEMARK);
	}

	public String getFullName() throws InvalidFontException, UnsupportedFontException {
		if (theFont.name == null)
			return null;
		try {
			return theFont.name.getName(Name.PlatformId.MACINTOSH, Name.MacintoshEncodingId.ROMAN, 0, Name.PredefinedNames.FULL_FONT_NAME); }
		catch (UnsupportedEncodingException e){}
		
		return null;
	}
	
	public boolean isBold() throws InvalidFontException, UnsupportedFontException {
		return theFont.os2 != null  
        	&& (theFont.os2.getSelection () & Os2.SelectionBits.bold) != 0;

	}

	public boolean isItalic() throws InvalidFontException, UnsupportedFontException {
		return theFont.os2 != null  
        	&& (theFont.os2.getSelection () & Os2.SelectionBits.italic) != 0;
	}
	
	public Permission getPermissions() throws InvalidFontException, UnsupportedFontException {
		Permission perm = theFont.getEmbeddingPermission(wasEmbedded);
		if (perm == Permission.ILLEGAL_VALUE)
			return Permission.EDITABLE;
		
		return perm;
	}

	public void getOutline(char ccode, OutlineConsumer consumer) throws UnsupportedFontException, InvalidFontException {
		try {
			theFont.getGlyphOutline(theFont.cmap.unicodeChar2glyph(ccode), consumer);
		} catch (NullPointerException e) {
			throw new InvalidFontException("Missing required table", e);
		}
	}

	public String getSubFamily() throws InvalidFontException, UnsupportedFontException {
		if (theFont.name == null)
			return null;
		
		return theFont.name.getFirstName(Name.PredefinedNames.FONT_SUBFAMILY);
	}
  }