/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.workflow.core.purge;

import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.exec.Workflow;

/**
 * Callback to be implemented by applications which wish to be notified whenever workflow instances are purged from the
 * repository using one of the available purge tools (scheduled purge job or JMX console). This is useful, for example,
 * in situations where some application-specific data associated with a workflow instance needs to be deleted along
 * with the workflow instance itself. Applications that wish to be notified upon workflow instance purge need to
 * provide an implementation of this interface and register it into the OSGI service registry. When the purge process
 * runs, it will notify every registered implementation of this interface whenever it purges a workflow instance. The
 * implementations can then perform whatever operations they need to perform for the given workflow instance (including
 * ignoring the instance if it is not of interest).
 */
public interface WorkflowPurgeCallback {

    /**
     * Callback method invoked by the purge process whenever it purges a workflow instance. Implementations can ignore
     * the given instance if it is not of interest to them, or perform any additional processing required for the
     * instance.<br>
     * To ensure that the purge process completes safely, efficiently and atomically, the following guidelines should
     * be followed by implementations of this callback:
     * <ul>
     *     <li>The implementation should try to determine as efficiently as possible whether the given workflow instance
     *     is of interest, preferably by relying only on the supplied inputs and not executing any extra queries.</li>
     *     <li>The implementation should try to re-use the supplied session as far as possible, to ensure that any
     *     repository updates it performs (e.g. deletion of extra data associated with the given workflow instance)
     *     are atomic with the deletion of the workflow instance itself.</li>
     *     <li>The implementation is allowed to throw exceptions (which will be caught by the purge process and logged,
     *     but otherwise ignored), but should be careful not to modify the given session in a manner which prevents
     *     it from being committed later. This is to ensure that the main workflow instance purge and other registered
     *     callbacks are not affected by errors in a specific callback implementation.</li>
     * </ul>
     * @param session The workflow session being used by the purge process for deleting workflow instances.
     * @param workflowInstanceId The ID of the workflow instance being purged.
     * @param payload The payload associated with the workflow instance being purged.
     * @param modelId The ID of the workflow model associated with the workflow instance being purged.
     * @param state The state of the workflow instance being purged.
     * @throws Exception If an error occurs within the callback. To ensure isolation, the purge process will catch this
     * error and log it, but otherwise ignore it. Callback implementations should take care to fail fast by throwing
     * errors early if they cannot complete their processing, instead of potentially dirtying the given session with
     * changes which cannot be committed later on.
     */
    void onDelete(WorkflowSession session,
                  String workflowInstanceId,
                  String payload,
                  String modelId,
                  Workflow.State state)
            throws Exception;

}
