// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

import java.util.Iterator;


/**
 * The basic node class. Every node in the XMP tree is of this type.
 * It is iterable, so the tree can be iterated without knowing the exact type
 * upfront.
 * In case of a struct it will iterate the child nodes,
 * in case of an array, it will iterate the array items and in the case of
 * a simple property (leaf node) it will return null or a qualifier, if it has any.
 */
public interface XMPNode extends Iterable<XMPNode>, XMPPathAddressable
{
	/**
	 * An unmodifiable Iterator that iterates over the child elements of this node.
	 * If the node has no child elements, an empty iterator is returned.
	 * An eventual existing Qualifier Struct will not be visited by this iterator.
	 */
	Iterator<XMPNode> iterator(); 
	
	/**
	 * Executes the provided visitor
	 * @param visitor the Visitor to execute
	 */
	void accept( XMPNodeVisitor visitor );
	
	/**
	 * Get a qualifier "facade" for this Node that allows management of qualifiers.
	 * Qualifiers are not part of the list of children of the node and can only be accessed 
	 * through this helper object.
	 * Qualifier can be added to any node in the tree except XMPMetadata and Qualifiers itself.
	 * 
	 * @return the XMPQualifier facade or null if no qualifiers can be added to this node.
	 */
	XMPQualifiers accessQualifiers();
	
	/**
	 * Returns the name of this node or null if it has no name
	 * @return the name of this node or null if it has no name
	 */
	String getName();
	
	/**
	 * gets the parent of this node. 
	 * @return returns the parent of this node or null if it has none (example: root element)
	 */
	XMPNode getParent();
	
	/**
	 * returns the namespace URI of this node
	 * @return the namespace URI or null, if this node is not qualified
	 */
	String getNamespace();
	
	/**
	 * Save Cast to the type that is passed. If the node is not of this type it will return null.
	 * @param type The class type this object should be casted to
	 * @return the object casted to the desired type or null if the cast fails
	 */
	<AdapterType> AdapterType adaptTo(Class<AdapterType> type);
	
	/**
	 * Overwrites the current node with a deep copy of node which is provided as input. 
	 * The node from which copying is done should not be null and it should be of same 
	 * type as this node. 
	 * @param copyFrom XMPNode which has to be copied 
	 * @throws IllegalArgumentException If node sent as parameter to this method is null.
	 * @throws IllegalStateException If this metadata tree is invalid.
	 */
	void copyReplace(XMPNode copyFrom);
	
	/**
	 * Checks if the node's parent is of type XMPArray.
	 * @return True if it is, false otherwise.
	 */
	boolean isArrayItem();
	
	/**
	 * Checks if the node has qualifiers attached.
	 * @return True if it has, false otherwise.
	 */
	boolean hasQualifiers();

	/**
	 * Determine the number of node children.
	 * For arrays, its the number of elements;
	 * for language alternatives the number of languages;
	 * for structs, the number of fields and
	 * for simple properties it is always zero.
	 * 
	 * @return Returns the number of node children.
	 */
	int size();
}
