// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

import java.util.Iterator;
import java.util.Set;


/**
 * This represents a structure in the XMP tree. It has a number of child nodes that
 * can be of any valid XMPNode type.
 *
 */
public interface XMPStruct extends XMPNode
{
	/**
	 * Create a new child node as a simple property
	 * @param namespace the namespace of the new child
	 * @param name the name of the new child
	 * @param value the value of the new child
	 * @return the newly created child
	 * 
	 * @throws IllegalArgumentException If name is not valid as per XML Spec or namespace
	 * equals http://www.w3.org/1999/02/22-rdf-syntax-ns# 
	 */
	XMPSimple setSimple( String namespace, String name, String value );
	
	/**
	 * Create a new child node as a nested empty struct
	 * @param namespace the namespace of the new child
	 * @param name the name of the new child
	 * @return the newly created child as an empty struct
	 * 
	 * @throws IllegalArgumentException If name is not valid as per XML Spec or namespace
	 * equals http://www.w3.org/1999/02/22-rdf-syntax-ns# 
	 */
	XMPStruct setStruct( String namespace, String name );
	
	/**
	 * Create a new child node as an empty array with a certain form (ordered, unordered, alternative)
	 * @param namespace the namespace of the new child
	 * @param name the name of the new child
	 * @param form the array form of the new child
	 * @return the newly created child as an empty XMPArray with a certain form
	 * 
	 * @throws IllegalArgumentException If name is not valid as per XML Spec or namespace
	 * equals http://www.w3.org/1999/02/22-rdf-syntax-ns# 
	 */
	XMPArray setArray( String namespace, String name, XMPArray.Form form );
	
	/**
	 * Create a new child node as an empty Language Alternative
	 * @param namespace the namespace of the new child
	 * @param name the name of the new child
	 * @return the newly created child as an empty XMPArray of a certain type
	 * 
	 * @throws IllegalArgumentException If name is not valid as per XML Spec or namespace
	 * equals http://www.w3.org/1999/02/22-rdf-syntax-ns# 
	 */
	XMPLanguageAlternative setLanguageAlternative( String namespace, String name );
	
	/**
	 * Return a child node as base node (without specifing the exact type)
	 * @param namespace the namespace of the child to return
	 * @param name the name of the child to return
	 * @return the desired child node or null if there in none with this name
	 */
	XMPNode get( String namespace, String name );
	
	/**
	 * Remove a child from the struct
	 * @param namespace the namespace of the child to remove 
	 * @param name the name of the child to remove
	 * @return the node that was returned of null if there was no child to remove
	 */
	XMPNode remove ( String namespace, String name );
	
	/**
	 * Returns a child node as a simple property
	 * @param namespace the namespace of child
	 * @param name the name of the child
	 * @return the child node or null if there in none with this name or type
	 */
	XMPSimple getSimple( String namespace, String name );
	
	/**
	 * Returns a child node as a nested struct
	 * @param namespace the namespace of child
	 * @param name the name of the child
	 * @return the child node or null if there in none with this name or type
	 */
	XMPStruct getStruct( String namespace, String name );
	
	/**
	 * Returns a child node as an array
	 * @param namespace the namespace of child
	 * @param name the name of the child
	 * @return the child node or null if there in none with this name or type
	 */
	XMPArray getArray( String namespace, String name );
	
	/**
	 * Returns a child node as a language Alternative
	 * @param namespace the namespace of child
	 * @param name the name of the child
	 * @return the child node or null if there in none with this name or type
	 */
	XMPLanguageAlternative getLanguageAlternative( String namespace, String name );
	
	/**
	 * Renames an existing child node.
	 * @param namespace the namespace of child which is to be renamed
	 * @param name the name of child which is to be renamed
	 * @param newNamespace new namespace of the child node
	 * @param newName new name of the child node
	 * 
	 * @throws IllegalArgumentException If name is not valid as per XML Spec or namespace
	 * equals http://www.w3.org/1999/02/22-rdf-syntax-ns# 
	 */
	public void renameField(String namespace, String name, String newNamespace, String newName);

	/**
	 * Returns an iterator which iterates over all properties in this struct that belong to the specified namespace
	 * @param namespace the namespace
	 * @return Iterator which iterate over list of {@link XMPNode}
	 */
	Iterator<XMPNode> getNamespaceProperties( String namespace );
	
	
	/**
	 * Returns an immutable set of namespaces of properties in this struct.
	 * @return Returns an immutable Set of namespace strings.
	 */
	Set<String> getUsedNamespaces();

	
	/**
	 * Creates a new XMPSimple node and copies the provided simple node with qualifiers into the new node 
	 * @param simple the simple node to copy
	 * @return the newly created simple node
	 */
	XMPSimple copy( XMPSimple simple );
	
	
	/**
	 * Creates a new XMPStruct node and copies the whole tree of the provided struct into the new node 
	 * @param struct the Struct to copy
	 * @return the newly created struct node
	 */
	XMPStruct copy( XMPStruct struct );
	
	
	/**
	 * Creates a new XMPArray node and copies the whole tree of the provided array into the new node 
	 * @param array the array to copy
	 * @return the newly created array node
	 */
	XMPArray copy( XMPArray array );

}
