/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.cq.dam.download.api;

import java.net.URI;
import java.util.Collection;
import java.util.Map;

import org.osgi.annotation.versioning.ProviderType;

/**
 * Represents a final artifact that is a result of the download creation
 * process. The artifact will consist of a URI that can be used to retrieve the
 * artifact, along with various peices of status information.
 */
@ProviderType
public interface DownloadArtifact {

    /**
     * Retrieves the identifier of the artifact.
     * @return A unique artifact identifier.
     */
    String getId();

    /**
     * Retrieves the name of the artifact.
     * @return An artifact name.
     */
    String getName();

    /**
     * Retrieves the mime type of the artifact. This will typically be the
     * HTTP content type of the binary.
     * @return A mime type.
     */
    String getMimeType();

    /**
     * <p>Retrieves the URI where the artifact can be retrieved. The URI might
     * be either a path or an HTTP URL. If a path, the artifact should be
     * requested from the same host where the download was initiated.</p>
     * <p>If the artifact failed to generate successfully, then the URI value
     * will be <i>null</i>. In this case, use {@link #getFailureReason()} to
     * get information about why the artifact failed.</p>
     * @return An HTTP URL, an absolute path, or null if there was a failure.
     */
    URI getBinaryURI();

    /**
     * Retrieves a description of why the artifact failed to be created
     * successfully.
     * @return A failure description, or <i>null</i> if the artifact was created
     *  successfully.
     */
    String getFailureReason();

    /**
     * Retrieves details about the items that processed correctly.
     * @return All of the successful items that were processed.
     */
    Collection<String> getSuccesses();

    /**
     * Get the number of successful files.
     * @return count of successfully processed files
     */
    int getSuccessCount();

    /**
     * Retrieves information about the items that failed to process. The
     * presense of failures does not indicate that the artifact itself failed
     * to process, only that one or more of the requested items could not
     * be included in the artifact.
     * @return A map whose keys are item paths and whose values are failure
     * messages.
     */
    Map<String, String> getFailures();
}
