/*
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2019 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */
package com.adobe.cq.inbox.ui.column;

import com.adobe.cq.inbox.ui.column.provider.ColumnProvider;

import javax.annotation.Nonnull;

/**
 * A column in the inbox.Each column should be backed by a {@link ColumnProvider}
 */
public final class Column {
    private String name;
    private String title;
    private String type;
    private String templateResourcePath;
    private String templateName;

    /**
     * Create a new {@link Column}
     * @param name name of the column
     * @param title title of the column
     */
    public Column(@Nonnull String name, String title) {
        this.name = name;
        this.title = title;
    }

    /**
     * Create a new {@link Column} which is based on sightly template
     * @param name name of the column
     * @param title title of the column
     * @param templateResourcePath path of the resource containing the sightly template to render the column
     * @param templateName name of the sightly template to render the column
     */
    public Column(@Nonnull String name, String title, @Nonnull String templateResourcePath, @Nonnull String templateName) {
        this.name = name;
        this.title = title;
        this.templateName = templateName;
        this.templateResourcePath = templateResourcePath;
    }

    /**
     * Create a new {@link Column} of the given type
     * @param name name of the column
     * @param title title of the column
     * @param type fully qualified class name of the class representing the type of the data associated with the column
     */
    public Column(@Nonnull String name, String title, @Nonnull String type) {
        this(name, title);
        this.type = type;
    }

    /**
     * Create a new {@link Column} based on sightly template and of given type
     * @param name name of the column
     * @param title title of the column
     * @param type fully qualified class name of the class representing the type of the data associated with the column
     * @param templateResourcePath path of the resource containing the sightly template to render the column
     * @param templateName name of the sightly template to render the column
     */
    public Column(@Nonnull String name, String title,  @Nonnull String type, @Nonnull String templateResourcePath, @Nonnull String templateName) {
        this(name, title, templateResourcePath, templateName);
        this.type = type;
    }

    /**
     * Name of the column. Name is used to uniquely identify a column
     * @return column name
     */
    public String getName() {
        return name;
    }

    /**
     * The display name of the column
     * @return column title
     */
    public String getTitle() {
        return title != null && title.length() !=0 ? title : name;
    }

    /**
     * Fully qualified name of the class representing the type of the data associated with the column
     * Used for sorting on the column
     * e.g. java.lang.String
     *
     * The column will be sortable for following types
     *
     * java.lang.String
     * java.lang.Boolean
     * java.lang.Long
     * java.lang.Double
     * java.util.Date
     *
     * @return column type
     */
    public String getType() {
        return type;
    }

    /**
     * Path of the resource containing the sightly template to render the column
     * @return
     */
    public String getTemplateResourcePath() {
        return templateResourcePath;
    }


    /**
     * Name of the sightly template to render the column
     * The resource at {@code templateResourcePath} may contain multiple sightly templates
     *
     * The template accepts a parameter named <code>item</code> of type {@link com.adobe.cq.inbox.ui.InboxItem}
     *
     * Sample template
     *
     * {@code
     * <template data-sly-template.wftitle="${@ item}>">
     *     <div>
     *         ${item.workflowTitle @ i18n, locale=request.locale}
     *     </div>
     * </template>
     * }
     * @return
     */
    public String getTemplateName() {
        return templateName;
    }
}
