/*
*
*	File: NameHeuristics.java
*
* ****************************************************************************
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
* 
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.postscript;

import java.util.regex.Pattern;

import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.*;
/**
 * A class containing static methods that do name manipulation for legacy type1 fonts.
 *
 * <h4>Synchronization</h4>
 * 
 * This class represents a namespace. It does not contain instance data
 * or mutable static data. It is therefore threadsafe.
 */
final public class NameHeuristics {
    private static class PatternToWeight
    {
        final Pattern pattern;
        final int weight;
        
        PatternToWeight(Pattern pat, int i)
        {
            pattern = pat;
            weight = i;
        }
    }
    private static class PatternToStretch
    {
        final Pattern pattern;
        final CSSStretchValue stretch;
        
        PatternToStretch(Pattern pat, CSSStretchValue s)
        {
            pattern = pat;
            stretch = s;
        }
    }
       
    private final static PatternToWeight[] weightPatterns;
    private final static PatternToStretch[] widthPatterns;

    private final static Pattern smallCapsPattern;
    private final static Pattern[] dingbatsPatterns;
    
    static {
        weightPatterns = new PatternToWeight[21];
        weightPatterns[0] = new PatternToWeight(Pattern.compile("[- ]extra\\s*light", Pattern.CASE_INSENSITIVE), 100);
        weightPatterns[1] = new PatternToWeight(Pattern.compile("[- ]ultra\\s*light", Pattern.CASE_INSENSITIVE), 100);
        weightPatterns[2] = new PatternToWeight(Pattern.compile("[- ]thin", Pattern.CASE_INSENSITIVE),200);
        weightPatterns[3] = new PatternToWeight(Pattern.compile("[- ]highlight", Pattern.CASE_INSENSITIVE), 400);
        weightPatterns[4] = new PatternToWeight(Pattern.compile("[- ]light", Pattern.CASE_INSENSITIVE), 300);
        weightPatterns[5] = new PatternToWeight(Pattern.compile("[- ]medium", Pattern.CASE_INSENSITIVE), 500);
        weightPatterns[6] = new PatternToWeight(Pattern.compile("[- ]semi\\s*bold", Pattern.CASE_INSENSITIVE), 600);
        weightPatterns[7] = new PatternToWeight(Pattern.compile("[- ]demi\\s*bold", Pattern.CASE_INSENSITIVE), 600);
        weightPatterns[8] = new PatternToWeight(Pattern.compile("[- ]demi", Pattern.CASE_INSENSITIVE), 600);
        weightPatterns[9] = new PatternToWeight(Pattern.compile("[- ]extra\\s*bold", Pattern.CASE_INSENSITIVE), 800);
        weightPatterns[10] = new PatternToWeight(Pattern.compile("[- ]ultra\\s*bold", Pattern.CASE_INSENSITIVE), 800);
        weightPatterns[11] = new PatternToWeight(Pattern.compile("[- ]extra\\s*black", Pattern.CASE_INSENSITIVE), 950);
        weightPatterns[12] = new PatternToWeight(Pattern.compile("[- ]ext\\s*black", Pattern.CASE_INSENSITIVE), 950);
        weightPatterns[13] = new PatternToWeight(Pattern.compile("[- ]ultra\\s*black", Pattern.CASE_INSENSITIVE), 950);
        weightPatterns[14] = new PatternToWeight(Pattern.compile("[- ]ultra", Pattern.CASE_INSENSITIVE), 900);
        weightPatterns[15] = new PatternToWeight(Pattern.compile("[- ]super", Pattern.CASE_INSENSITIVE), 800);
        weightPatterns[16] = new PatternToWeight(Pattern.compile("[- ]bold", Pattern.CASE_INSENSITIVE), 700);
        weightPatterns[17] = new PatternToWeight(Pattern.compile("[- ]black", Pattern.CASE_INSENSITIVE), 900);
        weightPatterns[18] = new PatternToWeight(Pattern.compile("[- ]poster", Pattern.CASE_INSENSITIVE), 900);
        weightPatterns[19] = new PatternToWeight(Pattern.compile("[- ]heavy", Pattern.CASE_INSENSITIVE), 850);
        weightPatterns[20] = new PatternToWeight(Pattern.compile("[- ]nord", Pattern.CASE_INSENSITIVE), 900);
        
        widthPatterns = new PatternToStretch[17];
        widthPatterns[0] = new PatternToStretch(Pattern.compile("[- ]ultra\\s*condensed", Pattern.CASE_INSENSITIVE), CSSStretchValue.ULTRACONDENSED);
        widthPatterns[1] = new PatternToStretch(Pattern.compile("[- ]extra\\s*compressed", Pattern.CASE_INSENSITIVE), CSSStretchValue.ULTRACONDENSED);
        widthPatterns[2] = new PatternToStretch(Pattern.compile("[- ]ultra\\s*compressed", Pattern.CASE_INSENSITIVE), CSSStretchValue.ULTRACONDENSED);
        widthPatterns[3] = new PatternToStretch(Pattern.compile("[- ]extra\\s*condensed", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXTRACONDENSED);
        widthPatterns[4] = new PatternToStretch(Pattern.compile("[- ]compressed", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXTRACONDENSED);
        widthPatterns[5] = new PatternToStretch(Pattern.compile("[- ]semi\\s*condensed", Pattern.CASE_INSENSITIVE), CSSStretchValue.SEMICONDENSED);
        widthPatterns[6] = new PatternToStretch(Pattern.compile("[- ]compact", Pattern.CASE_INSENSITIVE), CSSStretchValue.SEMICONDENSED);
        widthPatterns[7] = new PatternToStretch(Pattern.compile("[- ]narrow", Pattern.CASE_INSENSITIVE), CSSStretchValue.SEMICONDENSED);
        widthPatterns[8] = new PatternToStretch(Pattern.compile("[- ]condensed", Pattern.CASE_INSENSITIVE), CSSStretchValue.CONDENSED);
        widthPatterns[9] = new PatternToStretch(Pattern.compile("[- ]semi\\s*expanded", Pattern.CASE_INSENSITIVE), CSSStretchValue.SEMIEXPANDED);
        widthPatterns[10] = new PatternToStretch(Pattern.compile("[- ]semi\\s*extended", Pattern.CASE_INSENSITIVE), CSSStretchValue.SEMIEXPANDED);
        widthPatterns[11] = new PatternToStretch(Pattern.compile("[- ]extra\\s*expanded", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXTRAEXPANDED);
        widthPatterns[12] = new PatternToStretch(Pattern.compile("[- ]extra\\s*extended", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXTRAEXPANDED);
        widthPatterns[13] = new PatternToStretch(Pattern.compile("[- ]ultra\\s*expanded", Pattern.CASE_INSENSITIVE), CSSStretchValue.ULTRAEXPANDED);
        widthPatterns[14] = new PatternToStretch(Pattern.compile("[- ]ultra\\s*extended", Pattern.CASE_INSENSITIVE), CSSStretchValue.ULTRAEXPANDED);
        widthPatterns[15] = new PatternToStretch(Pattern.compile("[- ]expanded", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXPANDED);
        widthPatterns[16] = new PatternToStretch(Pattern.compile("[- ]extended", Pattern.CASE_INSENSITIVE), CSSStretchValue.EXPANDED);
        
        smallCapsPattern = Pattern.compile("[- ]small\\s*caps", Pattern.CASE_INSENSITIVE);
        
        dingbatsPatterns = new Pattern[] {
            Pattern.compile ("Zapf\\s*Dingbats", Pattern.CASE_INSENSITIVE),
            Pattern.compile ("Adobe\\s*Pi", Pattern.CASE_INSENSITIVE)};
        
    }

    public static int fullNameToWeight(String fullName)
    {
        for (int i = 0; i < weightPatterns.length; ++i)
        {
            if (weightPatterns[i].pattern.matcher(fullName).find())
                return weightPatterns[i].weight;
        }

        return 400;
    }
    
    public static int weightNameToWeight (String weightName) {
      String s = "-" + weightName;
      for (int i = 0; i < weightPatterns.length; i++) {
        if (weightPatterns [i].pattern.matcher (s).lookingAt()) {
          return weightPatterns[i].weight; }}
      return 400;
    }
    
    public static CSSStretchValue fullNameToWidth(String fullName, String familyName)
    {
        if (familyName != null)
        {
            // familyName + ultracondensed refers to a weight of ultra and
            // a width of condensed, not to ultracondensed (eg, ITC Century Ultra Condensed
            // has a weight of 850 and a width of condensed -- and so does ITC Century Ultra Condensed Italic)
            Pattern pattern = Pattern.compile(Pattern.quote(familyName) + "[-\\s]*ultra\\s*condensed\\s*(italic|oblique)?", Pattern.CASE_INSENSITIVE);
            if (pattern.matcher(fullName).matches())
                return CSSStretchValue.CONDENSED;
            
        }

        for (int i = 0; i < widthPatterns.length; ++i)
        {
            if (widthPatterns[i].pattern.matcher(fullName).find())
                return widthPatterns[i].stretch;
        }

        return CSSStretchValue.NORMAL;
    }
    
    public static boolean fullNameIndicatesSmallCaps(String fullName)
    {
        return smallCapsPattern.matcher(fullName).find();
    }
    
    public static boolean fullNameIndicatesDingbats (String name)
    {
        if (name == null)
            return false;
        
        for (int i = 0; i < dingbatsPatterns.length; i++) {
          if (dingbatsPatterns [i].matcher (name).find ()) {
            return true; }}
        return false;
    }              
}
