/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.comments;

import java.util.Dictionary;
import java.util.Hashtable;

import org.osgi.service.event.Event;

/**
 * The <code>CommentingEvent</code> represents events occurred during commenting
 */
public final class CommentingEvent {

    /**
     * Base Event Topic for Commenting Events. EVENT_TOPIC will be formed by appending types to it.
     */
    public static final String EVENT_TOPIC_BASE = "com/adobe/granite/comments";

    private static final String PROPERTY_COMMENT_PATH = "commentPath";

    /**
     * full event.topic of the event (e.g. EVENT_TOPIC_BASE + "/" + COMMENTED)
     */
    private final String eventTopic;

    /**
     * The path of the comment
     */
    private final String commentPath;

    private CommentingEvent(Event event) {
        eventTopic = event.getTopic();
        if(event.getProperty(PROPERTY_COMMENT_PATH) instanceof String ){
            commentPath = (String) event.getProperty(PROPERTY_COMMENT_PATH);
        }else{
            commentPath = null;
        }
    }

    public CommentingEvent(final Type eventTopic, String commentPath) {
        this.eventTopic = EVENT_TOPIC_BASE + "/" + eventTopic.name().toLowerCase();
        this.commentPath = commentPath;
    }

    /**
     * convert OSGI event to {@link CommentingEvent}
     * @param event OSGI event
     * @return CommentingEvent. null if it can't be converted to CommentingEvent.
     */
    public static CommentingEvent fromEvent(final Event event) {
        String topic = event.getTopic();
        if(topic.substring(0, EVENT_TOPIC_BASE.length()).equalsIgnoreCase(EVENT_TOPIC_BASE)){
            String eventType = topic.substring(EVENT_TOPIC_BASE.length()+1);
            if(Type.valueOf(eventType.toUpperCase()) == Type.COMMENTED){
                return new CommentingEvent(event);
            }
        }
        return null;
    }

    /**
     * @return the path of the comment created.
     */
    public String getCommentPath() {
        return commentPath;
    }

    /**
     * @return event topic of the event
     */
    public String getEventTopic() {
        return eventTopic.toLowerCase();
    }

    /**
     * types of commenting events
     */
    public static enum Type {

        /**
         * raised when a resource is commented
         */
        COMMENTED
    }

    /**
     * creates an event for Type.COMMENTED
     * 
     * @param commentPath path of the comment node
     * @return CommentingEvent Object
     */
    public static CommentingEvent commented(final String commentPath) {
        return new CommentingEvent(Type.COMMENTED, commentPath);
    }

    /**
     * Create a distributable event.
     * 
     * @return An event.
     */
    public Event toEvent() {
        return new Event(eventTopic, this.getEventProperties());
    }

    /**
     * Generate a dictionary for the OSGi event.
     * 
     * @return a Dictionary of properties
     */
    private Dictionary<String, Object> getEventProperties() {
        Hashtable<String, Object> properties = new Hashtable<String, Object>();
        
        if(commentPath!=null){
            properties.put(PROPERTY_COMMENT_PATH, commentPath);
        }
        return properties;
    }
}
