/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.security.user;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.jcr.Node;
import javax.jcr.RepositoryException;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;

import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

/**
 * <p>
 * UserProperties is a general interface used to represent user preferences
 * and profiles. It provides convenience methods to retrieve properties
 * in various formats.
 * </p>
 *
 * <p>
 * The properties are either read from a single node or are composed from an
 * list of nodes. Such a composite can be retrieved via
 * {@link UserPropertiesManager#getUserProperties(String, String, Comparator)}.
 * The comparator will define the order in which a given property is read from
 * the composite in case it is present in more than one of the enclosed {@link UserProperties}.
 * </p>
 *
 * <p>
 * In addition this interface allows to access the underlying JCR Node used
 * to write or modify this {@code UserProperties}, version or lock it
 * as well as to register observation event listeners.
 * In case of a singular {@link UserProperties} object this is the node the information
 * is read from. In case of composite the it's the node representing the relative
 * root path specified in {@link UserPropertiesManager#getUserProperties(String, String, Comparator)}.
 * </p>
 */
@ProviderType
public interface UserProperties {

    /**
     * Constant for the property containing a description of an authorizable.
     */
    String ABOUT_ME = "aboutMe";

    /**
     * Constant for the email property of an authorizable.
     */
    String EMAIL = "email";

    /**
     * Constant for the family name of a given user.
     */
    String FAMILY_NAME = "familyName";

    /**
     * Constant for the given name of a given user or group.
     */
    String GIVEN_NAME = "givenName";

    /**
     * Constant for the middle name property of a user.
     */
    String MIDDLE_NAME = "middleName";

    /**
     * Constant for the display name (or nick name) property of a given user
     * or group. See also {@link #getDisplayName()} for a implementation
     * specific calculation of a display name that may rely on or fallback to
     * various name related properties.
     */
    String DISPLAY_NAME = "displayName";

    /**
     * Constant for the user property storing images of a user.
     */
    String PHOTOS = "photos";


    /**
     * Returns the ID of the authorizable this user properties belong to.
     *
     * @return the ID of the authorizable this user properties belong to.
     */
    @Nonnull
    String getAuthorizableID();

    /**
     * Returns the authorizables path
     *
     * @return the path of the authorizable.
     */
    @Nonnull
    String getAuthorizablePath();

    /**
     * Returns the principal name of the authorizable.
     *
     * @return The principal name.
     */
    @Nonnull
    String getPrincipalName();

    /**
     * Returns the JCR {@code node} associated with this user properties
     * instance. The node may then be used to write properties or execute other
     * JCR specific operations on this user properties that are not covered by
     * convenience methods defined by this interface.
     *
     * If this instance is composed of multiple enclosed {@link UserProperties}, this
     * method will return the {@link Node} representing the relative root path as
     * specified with {@link UserPropertiesManager#getUserProperties(String, String, Comparator)}.
     * If that node is not accessible to the editing session, this method will return {@code null}.
     *
     * @return the {@code node} associated with this user properties instance or {@code null} if the node is not accessible.
     */
    @CheckForNull
    Node getNode();

    /**
     * Retrieves all first-level property names found on the underlying {@link Node}
     * or on all enclosed user properties in case this instance is the result of aggregation.
     *
     * This is the same as {@link #getPropertyNames(String)} where the relative path is {@code null}.
     *
     * @return An array of property names.
     * @throws RepositoryException If an error occurs.
     */
    @Nonnull
    String[] getPropertyNames() throws RepositoryException;

    /**
     * Retrieves all property names available at the specified relative path on
     * the underlying {@link Node} or on the enclosed {@link UserProperties} in
     * case this instance is the result of aggregation. If {@code relPath} is {@code null},
     * the names are obtained from the {@code node} directly (and not from a subtree).
     *
     * @param relPath A relative path to the sub tree where to retrieve the
     * property names.
     * @return An array of property names.
     * @throws RepositoryException If an error occurs.
     */
    @Nonnull
    String[] getPropertyNames(@Nullable String relPath) throws RepositoryException;

    /**
     * <p>
     * Returns the string value of the property identified by the specified
     * {@code relativePath}. This may be a simple property name in case
     * of {@link #getPropertyNames()} or the relative path to the property
     * in case of {@link #getPropertyNames(String)}.
     * </p>
     * <p>
     * Note that this method is a simplified variant of
     * {@link #getProperty(String, Object, Class)} where the default value is
     * {@code null} and the desired type {@code String}.
     * </p>
     *
     * In case of aggregation the referenced property may be found on multiple
     * enclosed {@link UserProperties}. In this case the order of the underlying
     * {@code UserProperties} will determine, which property value will be exposed.
     * The order of the user properties is governed by explicitly specifying the
     * order in {@link UserPropertiesManager#getUserProperties(String, String, Comparator)}.
     *
     * @param relativePath Relative path referring to the property to
     * be retrieved.
     * @return The string value of the specified property or {@code null}.
     * @throws RepositoryException If an error occurs.
     */
    @CheckForNull
    String getProperty(@Nonnull String relativePath) throws RepositoryException;

    /**
     * Same as {@link #getProperty(String)}. Additionally the expected
     * format of the property value is specified by the type parameter. If no
     * property exists at {@code relativePath} the specified
     * {@code defaultValue} is returned.
     *
     * @param relativePath Relative path referring to the property to
     * be retrieved.
     * @param defaultValue The default value to be used if the property at
     * relativePath does not exist.
     * @param type The expected value type. The property value may need to
     * be converted.
     * @param <T> specifies the expected format of the property
     * @return The value of the specified property converted to the specified
     * {@code type} or the {@code defaultValue} if the property does
     * not exist or cannot be converted to the specified type.
     * @throws RepositoryException If an error occurs.
     */
    @CheckForNull
    <T> T getProperty(@Nonnull String relativePath, @Nullable T defaultValue, @Nonnull Class<T> type) throws RepositoryException;

    /**
     * Retrieve the resource at the specified path relative to this instance.
     *
     * @param relativePath Relative path to the resource to be retrieved.
     * @return The resource at the specified relative path or {@code null}
     * if the the path cannot be resolved to a resource.
     * @throws RepositoryException If an error occurs.
     */
    @CheckForNull
    Resource getResource(@Nonnull String relativePath) throws RepositoryException;

    /**
     * Retrieve the resources at the specified path relative to this instance.
     *
     * @param relativePath Relative path to the resources to be retrieved.
     * @return The resources at the specified relative path or {@code null}
     * if the the path cannot be resolved to resources.
     * @throws RepositoryException If an error occurs.
     */
    @CheckForNull
    Iterator<Resource> getResources(@Nonnull String relativePath) throws RepositoryException;

    /**
     * Calculates the resource path for the property located the specified
     * {@code relativePath} using the given {@code suffix}. If the
     * property does not exist the specified {@code defaultPath} is returned.
     *
     * @param relativePath Relative path to the property for which the resource
     * path will be calculated.
     * @param suffix An optional suffix to be appended to the resource path.
     * @param defaultPath An optional default path in case the requested
     * property does not exist (or is not accessible).
     * @return The resource path for the requested property or the default path.
     * @throws RepositoryException If an error occurs.
     */
    @Nonnull
    String getResourcePath(@Nonnull String relativePath, @Nullable String suffix, @Nonnull String defaultPath) throws RepositoryException;

    /**
     * Calculate a display name from this {@code UserProperties} instance.
     *
     * @return Implementation specific display name as stored in this user
     * properties instance or some implementation specific default.
     * @throws RepositoryException If an error occurs.
     */
    @Nonnull
    String getDisplayName() throws RepositoryException;

    /**
     * Calculate a display name from this {@code UserProperties} instance.
     *
     * @param nameDisplayOrder Order of given, middle and family names.
     * Western name order should be "givenName middleName familyName",
     * Eastern name order should be "familyName givenName middleName".
     * @return Implementation specific display name as stored in this user
     * properties instance or some implementation specific default.
     * @throws RepositoryException If an error occurs.
     */
    @Nonnull
    String getDisplayName(@Nullable String nameDisplayOrder) throws RepositoryException;

    /**
     * Checks if the authorizable of this properties is a group.
     * @return {@code true} if this properties belongs to a group.
     */
    boolean isGroupProperties();

    /**
     * Returns a list of {@link UserProperties} enclosed within this instance
     * or an empty list if this instance doesn't represent an aggregation of {@link UserProperties}.
     *
     * @return The collection of aggregated user properties or an empty collection if
     * there are no aggregated properties.
     */
    @Nonnull
    List<UserProperties> getAggregatedUserProperties();
}
