/*
 * File: ASCII85InputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.InputStream;

/**
 * ASCII85InputStream
 *  Reads data coded in ASCII85
 *
 *	Important: This filter assumes the input stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
public class ASCII85InputStream extends DecodeInputStream
{
	// Variables
	private static final int WS = 85;	/* white space */
	private static final int ZZ = 86;	/* z, all 0's */
	private static final int ED = 87;	/* ~, first char of EOD */
	private static final int IL = 88;	/* illegal character */

	private static byte	asc85DecodeTable[];
	private static long M085[];
	private static int M185[], M285[];
	private static short M385[];

	private int digit[] = new int[5];

	// Static initializer
	static
	{
		asc85DecodeTable = new byte[128];
		for (int i = 0; i < 128; i++) {
			switch ((char)i) {
			case ' ':
			case '\t':
			case '\n':
			case '\r':
			case '\f':
			case '\b':
				asc85DecodeTable[i] = (byte) WS; // white space
				break;
			case 'z':
				asc85DecodeTable[i] = (byte) ZZ; // four zeros
				break;
			case '~':
				asc85DecodeTable[i] = (byte) ED; // start of EOD
				break;
			default:
				asc85DecodeTable[i] =
					(('!' <= i) && (i <= 'u')) ?
					(byte)(i - '!') :
					IL; // illegal
				break;
			}
		}

		M085 = new long[85];
		M185 = new int[85];
		M285 = new int[85];
		M385 = new short[85];

		short m3 = 0, i3 = 85;
		int   m2 = 0, i2 = i3 * 85;
		int   m1 = 0, i1 = i2 * 85;
		long  m0 = 0, i0 = i1 * 85L;

		for (int i = 0; i < 85; i++) {
			M085[i] = m0;   m0 += i0;
			M185[i] = m1;   m1 += i1;
			M285[i] = m2;   m2 += i2;
			M385[i] = m3;   m3 += i3;
		}
	}

	// Constructors
	/**
	 * Creates an input stream filter.
	 * @param in	the input stream
	 */
	public ASCII85InputStream(InputStream in, int inSize, int outSize, FilterParams diparams)
	{
		super(in, inSize, outSize, 4, diparams);
	}

	public ASCII85InputStream(InputStream  in, FilterParams p)
	{
		super(in, 4, p);
	}

	public ASCII85InputStream(InputStream  in)
	{
		this(in, null);
	}

	@Override
	public void fill()
	{
		long value;
		int newChar, digitVal, dp;

		fillLoop:
		while (outCount <= (outBuf.length - 4) && !pendingEOF && pendingException == null) {
			digit[0] = digit[1] = digit[2] = digit[3] = digit[4] = 0;

			accumulate:
			for (dp = 0; dp < 5;) {
				if ( inCount <= inPos ) {
					if (!fillInputBuffer()) {
						pendingException =
							new ASCII85FilterException("Asc85 missing EOF");
						break fillLoop;
					}
				}
				newChar = inBuf[inPos++] & 0x7F;
				digitVal = asc85DecodeTable[newChar];

				if ( digitVal < 85 )
					digit[dp++] = digitVal;
				else switch (digitVal) {
				case WS: /* whitespace */
					break;

				case ZZ: /* all 0's */
					if (dp != 0) {
						pendingException =
							new ASCII85FilterException("Asc85 misaligned 'z'");
					}
					else
						dp = 5;
					break accumulate;

				case ED: /* '~', first character of EOD */
					if ( inCount <= inPos ) {
						if (!fillInputBuffer()) {
							pendingException =
								new ASCII85FilterException("Asc85 missing final '>'");
							break accumulate;
						}
					}
					newChar = inBuf[inPos++] & 0xFF;
					if (newChar != '>') {
						pendingException =
							new ASCII85FilterException("Asc85 missing final '>'");
					}
					pendingEOF = true;
					break accumulate;

				default: /* illegal character, or EOD */
					pendingException =
						new ASCII85FilterException("Asc85 illegal char");
					break accumulate;
				}
			} /* accumulate: */
			value = M085[digit[0]] + M185[digit[1]] + M285[digit[2]]
				+ M385[digit[3]] + digit[4];
			if (value > 0x0ffffffffL) {
//				pendingException =
//					new ASCII85FilterException("Asc85 pentet overflow");
				dp = 0;
			}

			switch (dp) {
			case 5:			/* 5 characters in, 4 bytes out - normal case */
				outBuf[outCount++] = (byte) (value >>> 24);
				outBuf[outCount++] = (byte) (value >>> 16);
				outBuf[outCount++] = (byte) (value >>>  8);
				outBuf[outCount++] = (byte)  value;
				break;

				/* Round up input value to get correct answer */
			case 4:	 /* 4 characters in, 3 bytes out */
				value += 0x0ff;
				outBuf[outCount++] = (byte) (value >>> 24);
				outBuf[outCount++] = (byte) (value >>> 16);
				outBuf[outCount++] = (byte) (value >>>  8);
				break;

			case 3:	 /* 3 characters in, 2 bytes out */
				value += 0x0ffff;
				outBuf[outCount++] = (byte) (value >>> 24);
				outBuf[outCount++] = (byte) (value >>> 16);
				break;

			case 1:	 /* 1 character  in, 1 byte out */
				pendingException =
					new ASCII85FilterException("Asc85 short pentet");
			case 2:	 /* 2 characters in, 1 byte out */
				value += 0x0ffffff;
				outBuf[outCount++] = (byte) (value >>> 24);
				break;

			default:
			case 0:			/* EOF with no characters */
				break;
			}
		} /* fillLoop: */
	}
}
