/*
 ******************************************************************************
 * Copyright (C) 2003-2008, International Business Machines Corporation and   *
 * others. All Rights Reserved.                                               *
 ******************************************************************************
 */

/*
 * File: ULocale.java
 * ************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.agl.util;

import java.io.Serializable;
import java.util.Locale;

import com.adobe.agl.impl.SimpleCache;

/**
 * A class analogous to {@link java.util.Locale} that provides additional
 * support for ICU protocol. In ICU 3.0 this class is enhanced to support RFC
 * 3066 language identifiers.
 * 
 * <p>
 * Many classes and services in ICU follow a factory idiom, in which a factory
 * method or object responds to a client request with an object. The request
 * includes a locale (the <i>requested</i> locale), and the returned object is
 * constructed using data for that locale. The system may lack data for the
 * requested locale, in which case the locale fallback mechanism will be invoked
 * until a populated locale is found (the <i>valid</i> locale). Furthermore,
 * even when a populated locale is found (the <i>valid</i> locale), further
 * fallback may be required to reach a locale containing the specific data
 * required by the service (the <i>actual</i> locale).
 * 
 * <p>
 * ULocale performs <b>'normalization'</b> and <b>'canonicalization'</b> of
 * locale ids. Normalization 'cleans up' ICU locale ids as follows:
 * <ul>
 * <li>language, script, country, variant, and keywords are properly cased<br>
 * (lower, title, upper, upper, and lower case respectively)</li>
 * <li>hyphens used as separators are converted to underscores</li>
 * <li>three-letter language and country ids are converted to two-letter
 * equivalents where available</li>
 * <li>surrounding spaces are removed from keywords and values</li>
 * <li>if there are multiple keywords, they are put in sorted order</li>
 * </ul>
 * Canonicalization additionally performs the following:
 * <ul>
 * <li>POSIX ids are converted to ICU format IDs</li>
 * <li>'grandfathered' 3066 ids are converted to ICU standard form</li>
 * <li>'PREEURO' and 'EURO' variants are converted to currency keyword form,
 * with the currency id appropriate to the country of the locale (for PREEURO)
 * or EUR (for EURO).
 * </ul>
 * All ULocale constructors automatically normalize the locale id. To handle
 * POSIX ids, <code>canonicalize</code> can be called to convert the id to
 * canonical form, or the <code>canonicalInstance</code> factory method can be
 * called.
 * </p>
 * 
 * <p>
 * This class provides selectors {@link #VALID_LOCALE} and
 * {@link #ACTUAL_LOCALE} intended for use in methods named <tt>getLocale()</tt>
 * . These methods exist in several ICU classes, including
 * {@link com.adobe.agl.util.Calendar}, {@link com.adobe.agl.util.Currency},
 * {@link com.adobe.agl.text.UFormat}, {@link com.adobe.agl.text.BreakIterator},
 * {@link com.adobe.agl.text.Collator},
 * {@link com.adobe.agl.text.DateFormatSymbols}, and
 * {@link com.adobe.agl.text.DecimalFormatSymbols} and their subclasses, if any.
 * Once an object of one of these classes has been created, <tt>getLocale()</tt>
 * may be called on it to determine the valid and actual locale arrived at
 * during the object's construction.
 * 
 * <p>
 * Note: The <tt>getLocale()</tt> method will be implemented in ICU 3.0; ICU 2.8
 * contains a partial preview implementation. The <i>actual</i> locale is
 * returned correctly, but the <i>valid</i> locale is not, in most cases.
 * 
 * @see java.util.Locale
 * @author weiv
 * @author Alan Liu
 * @author Ram Viswanadha
 * @stable ICU 2.8
 */
public final class ULocale implements Serializable {
	// using serialver from jdk1.4.2_05
	private static final long serialVersionUID = 3715177670352309217L;
	private static ULocale defaultULocale = ULocale.forLocale(Locale
			.getDefault());
	private Locale locale;

	private static SimpleCache CACHE = new SimpleCache();

	private static final Locale EMPTY_LOCALE = new Locale("", "");

	/**
	 * The root ULocale.
	 * 
	 * @stable ICU 2.8
	 */
	public static final ULocale ROOT = new ULocale("root", EMPTY_LOCALE);


	/**
	 * Return a ULocale object for a {@link java.util.Locale}. The ULocale is
	 * canonicalized.
	 * 
	 * @param loc
	 *            a JDK locale
	 * @stable ICU 3.2
	 */
	public static ULocale forLocale(Locale loc) {
		if (loc == null) {
			return null;
		}
		if(CACHE == null)
		{
			CACHE = new SimpleCache();
		}
		ULocale result = (ULocale) CACHE.get(loc);
		if (result == null) {
			result = new ULocale(loc);
		}
		CACHE.put(loc, result);
		return result;
	}

	/**
	 * Private constructor used by static initializers.
	 */
	private ULocale(String localeID, Locale locale) {
		this.locale = locale;
	}

	public ULocale(String localeID) {
		this(localeID, new Locale(localeID));

	}

	/**
	 * Construct a ULocale object from a {@link java.util.Locale}.
	 * 
	 * @param loc
	 *            a JDK locale
	 * @stable ICU 2.8
	 * @internal
	 */
	private ULocale(Locale loc) {
		this.locale = loc;
	}
	
    public String getLanguage() {
        return locale.getLanguage();
    }

    
    public String getCountry() {
        return locale.getCountry();
    }
    
    public String getScript() {
        return "";
    }

	public static ULocale getDefault() {
		return defaultULocale;
	}

	public Locale toLocale() {
		return locale;
	}
	
	public ULocale getFallback() {
		// Return ROOT as fallback ULocale in AGL_Proxy
		if(this == ROOT)
		{
			// If already at ROOT return null to end chain.
			return null;
		}
		return ROOT;
	}
	
	/**
	 * Return hashcode based on locale
	 */
    public int hashCode() {
        return locale.hashCode();
    }
    
    
    /**
     * Equals based on locale.
     */
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        
        if (obj instanceof ULocale) {
            return locale.equals(((ULocale)obj).locale);
        }
        return false;
    }
}
