/*
*
*	File: Type2OutlineParser.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.cff;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.Matrix;

/** Extract the unhinted bezier outlines from a Type2 charstring.
 * 
 * A <code>Type2OutlineParser</code> offers a single method, <code>parse</code> which is 
 * given a charstring and an <code>OutlineConsumer</code>. As the 
 * lines and curves of the charstring are parsed, 
 * the <code>OutlineConsumer</code> is called.
 * 
 * <p>The same <code>Type2OutlineParser</code> object can be used repeatedly
 * to extract outlines. The successive invocations do not need to be for
 * the same font (not even in the same fontset), nor do they need to use
 * the same consumer. 
 * 
 * <p>The consumer must not call <code>parse</code> on the Type2OutlineParser 
 * that invoked it.
 * 
 * <h4>Synchronization</h4>
 * 
 * This class is not synchronized. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 * 
 * The <code>parse</code> method invokes
 * the <code>OutlineConsumer</code> on the thread on which it is called.
 */
final public class Type2OutlineParser {
  /** The Type2Parser we build on top of. */
  private final Type2Parser type2Parser = new Type2Parser ();
  
  /** The Type2UnhintedRedirector that turns Type2 operators into raw outline. */
  private final Type2UnhintedRedirector type2Consumer;
 
  /**
   * Create a new Type2OutlineParser.
   * 
   * @param hintedOutlines True if the outlineConsumers passed to parse will be implementations
   * of HintedOutlineConsumer. False if they will be implementations of OutlineConsumer.
   */
  public Type2OutlineParser(boolean hintedOutlines)  {
    if (hintedOutlines) {
      type2Consumer = new Type2HintedRedirector (); }
    else {
      type2Consumer = new Type2UnhintedRedirector (); }
  }
  
  /** Parse the outlines of a Type2 charstring.
   * @param charStrings the <code>CharStrings</code> containing the  charstring
   * @param index the index of the charctstring in <code>charStrings</code>
   * @param localSubrs the local subroutines
   * @param globalSubrs the global subroutines
   * @param outlineConsumer the OutlineConsumer to alert of the parsing events
   * @param m the Matrix to be applied to the outlines in order to get to a 1 ppem outline
   * @param font the font to use to resolve seac operators
   * @throws InvalidFontException the charstring is malformed
   */
  public void parse (CharStrings charStrings, int index,
                       CharStrings localSubrs, CharStrings globalSubrs, 
                       OutlineConsumer outlineConsumer, Matrix m, NameKeyedFont font)
  throws InvalidFontException, UnsupportedFontException {
    type2Consumer.reset(outlineConsumer);
    
    outlineConsumer.setMatrix(m);   
    type2Parser.parse (charStrings, index, localSubrs, globalSubrs, type2Consumer, font);
  }
}
