/*
*
*	File: Hmtx.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'hmtx' table.
 *
 * <h4>Version handling</h4>
 * 
 * <p>'hmtx' tables are not versionned.</p>
 *
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 **/

final public class Hmtx extends Table {
  
  // The 'hmtx' table does not stand on its own, as 
  // one needs the 'hhea.numberOfHMetrics' field to interpret it.
  
  // So this class collects all these pieces and gives the illusion
  // of a standalone 'hmtx' table.
  
  protected final int numberOfHMetrics;
  protected final HmtxRaw hmtx;
  
  protected Hmtx (HmtxRaw hmtx, int numberOfHMetrics) {
    this.hmtx = hmtx;
    this.numberOfHMetrics = numberOfHMetrics;
  }
  
  /** Get the left side bearing of the glyph <code>glyphID</code>. */
  public int getLeftSideBearing (int glyphID)
  throws InvalidFontException {
    return hmtx.getLeftSideBearing (glyphID, numberOfHMetrics);
  }

  /** Get the horizontal advance of the glyph <code>glyphID</code>. */
  public int getHorizontalAdvance (int glyphID)
  throws InvalidFontException {
    return hmtx.getHorizontalAdvance (glyphID, numberOfHMetrics);
  }
  
  public void subsetAndStream (Subset subset, Map tables)
  throws InvalidFontException {
	OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(subset.getNumGlyphs () * 4);
    int o = 0;
    for (int i = 0; i < subset.getNumGlyphs (); i++) {
      int fullId = subset.getFullGid (i);
      newData.setuint16 (o, getHorizontalAdvance (fullId));
      newData.setint16 (o + 2, getLeftSideBearing (fullId));
      o += 4; }
      
    tables.put (new Integer (Tag.table_hmtx), newData); 
  }  
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = hmtx.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_hmtx), newData);
  }
}

