/*
 *
 *	File: OTByteArray.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.io.OutputStream;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;

/** A sequence of bytes in an OpenType font file, with 
 methods to access the basic types.

 The return type of the various accessors for the integral
 types is <code>int</code> whenever the OpenType type fits 
 in [-2^31 .. 2^31-1], <code>long</code> otherwise. The 
 only exception is <code>getuint32asint</code> */

final public class OTByteArray extends FontByteArray {	
  private OTByteArray (int size) {
    super (size);
  }

  /**
   * Initialize 'this' to have the data in 'array'. This takes ownership of the data
   * in 'array'; it doesn't copy it.
   * @param array
   * @throws java.io.IOException
   * @throws InvalidFontException
   * @throws UnsupportedFontException
   */
  OTByteArray (FontByteArray array) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (array, false);
  }

  /**
   * Copy the data in original, creating a new, independent OTByteArray.
   * @param original
   */
  private OTByteArray (OTByteArray original) {
    super(original, true);
  }

  public static final OTByteArrayBuilder getOTByteArrayBuilderInstance () {
    return new OTByteArrayBuilder();
  }

  static final OTByteArrayBuilder getOTByteArrayBuilderInstance (int size) {
    return new OTByteArrayBuilder(size);
  }

  static final OTByteArrayBuilder getOTByteArrayBuilderInstance (OTByteArray original) {
    return new OTByteArrayBuilder(original);
  }

  // It is useful to keep in mind the guarantees offered by the
  // Java language when reading the code below. 

  // - a <code>byte</code> is a signed integer, in the range [-128, 127]
  // - an <code>int</code> is a signed integer, in the range [-2^31, 2^31-1]
  // - all integral types are represented using 2's complement
  // - undecorated literals are of type int;
  //   a literal with an 'L' or 'l' suffix is of type long
  // - most binary operators involve promotion of their arguments; for integral types,
  //   this boils down to: if either argument is a long then the 
  //   other is promoted to long, otherwise if either argument is
  //   an int then the other is promoted to int
  // - narrowing of integral types is just a bit masking operation

  // As a consequence, if b is a byte, then
  // 'b & 0xff' is an int (because 0xff is and int, and b is 
  // promoted to an int); and the value of that expression is 
  // the value of the byte as if interpreted as an unsigned byte.

  // Another consequence is that '(byte)0xaa' just sticks the bits
  // 10101010 in a byte; the value of that expression is -86; 
  // and the value of '((byte)0xaa) & 0xff' is 0xaa.

  /** return the uint8 at <code>offset</code> in the table */
  public final int getuint8 (int offset) throws InvalidFontException  {
    return getRawByte(offset) & 0xff;
  }

  /** return the int8 at <code>offset</code> in the table */
  protected final int getint8 (int offset) throws InvalidFontException  {
    return getSignedRawByte(offset);
  }

  /** return the uint16 at <code>offset</code> in the table */
  protected final int getuint16 (int offset) throws InvalidFontException {
    return (getuint8 (offset) << 8) | getuint8 (offset + 1);
  }

  /** return the int16 at <code>offset</code> in the table */
  public final int getint16 (int offset) throws InvalidFontException {
    return (getint8 (offset) << 8) | getuint8 (offset + 1);
  }

  /** return the int32 at <code>offset</code> in the table */
  protected final int getint32 (int offset) throws InvalidFontException {
    return ( getint8 (offset)    << 24
        | getuint8 (offset+1) << 16
        | getuint8 (offset+2) <<  8
        | getuint8 (offset+3));
  }

  /** return the uint32 at <code>offset</code> in the table */
  protected final long getuint32 (int offset) throws InvalidFontException  {
    return (  ((long) getuint8 (offset)) << 24
            | getuint8 (offset+1)        << 16
            | getuint8 (offset+2)        <<  8
            | getuint8 (offset+3));
  }
  
  /** return the uint24 at <code>offset</code> in the table */
  protected final int getuint24 (int offset) throws InvalidFontException  {
    return (  getuint8 (offset)  << 16
            | getuint8 (offset+1)        <<  8
            | getuint8 (offset+2));
  }
  /** return the uint32 at <code>offset</code> in the table
   * as an int.
   * @param offset the offset of the uint32
   * @param exceptionMsg the detail message for the exception, if thrown
   * @throws  UnsupportedFontException if the value
   * cannot be represented in an int.
   */
  protected final int getuint32asint (int offset, String exceptionMsg) 
  throws UnsupportedFontException, InvalidFontException {
    if (getuint8 (offset) > 0x7f) {
      throw new UnsupportedFontException (exceptionMsg); }
    return (int) getuint32 (offset);
  }

  /** return the Fixed at <code>offset</code> in the table */
  protected final int[] getFixed (int offset) throws InvalidFontException {
    int[] result = new int[2];
    result[0] = getuint16 (offset);
    result[1] = getuint16 (offset + 2);
    return result; 
  }

  /** return the LONGDATETIME at <code>offset</code> in the table */
  protected final long getLONGDATETIME (int offset) throws InvalidFontException {
    return (  getuint32 (offset) << 32
            | getuint32 (offset));
  }


  /** return the Offset at <code>base+offset</code> in the table. 
   * The offset is relative to <code>base</code> and the returned
   * value is relative to the beginning of the table.
   */
  protected final int getOffset (int base, int offset) throws InvalidFontException {
    int o = getuint16 (base + offset); 
    if (o != 0) {
      return base + o; }
    else {
      return 0; }
  }	

  /** compute the checksum of the table.
   * @throws InvalidFontException */
  protected long checksum () throws InvalidFontException {
    return checksum (0, this.getSize(), 0);
  }

  /** update the checksum for the bytes between from (included) and to (not included).
   * This method is really meant to support the computation of the checksum
   * of a whole table (see the method below), or to be used to compute the 
   * checksum of the head table.
   * 
   * @param from the index of the first byte to include
   * @param to the index of the byte following the last byte to include
   * @param checksum the current checksum
   * @return the updated checksum
   * @throws InvalidFontException 
   */	
  protected long checksum (int from, int to, long checksum) 
  throws InvalidFontException {
    while (from < to) {
      int b1 = this.getuint8(from++);

      int b2 = 0;
      if (from < to) {
        b2 = this.getuint8 (from++); }

      int b3 = 0;
      if (from < to) {
        b3 = this.getuint8 (from++); }

      int b4 = 0;
      if (from < to) {
        b4 = this.getuint8 (from++); }

      int word = b1 << 24 | b2 << 16 | b3 << 8 | b4;
      checksum = (checksum + word) & 0xffffffffL; }

    return checksum;
  }

  public boolean equals (Object other){
    if (!(other instanceof OTByteArray)) {
      return false; }

    return super.equals(other);
  }

  final static class OTByteArrayBuilder extends FontByteArrayBuilder {
	  
    static class OTByteArrayBuilderOutputStreamAdaptor extends OutputStream{
		 private OTByteArrayBuilder builder = new OTByteArrayBuilder();
		 private int position = 0;
		  
		 public void write(int b) throws IOException {
			builder.ensureCapacity(position+1);
			builder.setuint8(position, b);
			position++;
		 }
		
		public void write(byte[] b, int off, int len) throws IOException {
			builder.append(b, off, len);
			position += len;
		}

		OTByteArrayBuilder getBuilder() {
			return builder;
		}
		  
	  }

	protected OTByteArrayBuilder () {
      super(new OTByteArray (1024));
    }

    protected OTByteArrayBuilder (int size) {
      super(new OTByteArray(size));
    }

    protected OTByteArrayBuilder (OTByteArray original) {
      super(new OTByteArray(original));
    }

    final void setuint8 (int offset, int value) {
      this.setRawByte (offset, value & 0xff);
    }

    final void setuint16 (int offset, int value) {
      this.setRawByte(offset, (value >> 8) & 0xff);
      offset += 1;
      this.setRawByte(offset, value & 0xff);
    }
    
    final int getuint16(int offset)
    {
    	int i = this.getRawByte(offset) & 0xff;
    	i <<= 8;
    	i |= (this.getRawByte(offset+1) & 0xff);
    	
    	return i;
    }

    final void setint16 (int offset, int value) {
      setuint16 (offset, value);
    }
    
    final void setuint24(int offset, int value) {
    	this.setRawByte(offset, (value >> 16) & 0xff);
        offset += 1;
        this.setRawByte(offset, (value >>  8) & 0xff);
        offset += 1;
        this.setRawByte(offset, value & 0xff);
    }

    final void setuint32 (int offset, int value) {
      this.setRawByte(offset, (value >> 24) & 0xff);
      offset += 1;
      this.setRawByte(offset, (value >> 16) & 0xff);
      offset += 1;
      this.setRawByte(offset, (value >>  8) & 0xff);
      offset += 1;
      this.setRawByte(offset, value & 0xff);
    }

    final void setFixed (int offset, int major, int minor) 
    throws InvalidFontException {
      setuint16 (offset, major);
      setuint16 (offset + 2, minor);
    }

    final long checksum (int from, int to, long checksum) 
    throws InvalidFontException {
      return ((OTByteArray)this.byteArray).checksum();
    }

    OTByteArray toOTByteArray () {
      OTByteArray retvalue = (OTByteArray) this.byteArray;
      this.byteArray = null;	// kill ourselves
      return retvalue;
    }		
  }
}
