/*
 * File: Vorg.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.Arrays;
import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'VORG' table.
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

public final class Vorg extends Table implements VerticalOriginFetcher {
 
  protected Vorg (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
  }
  
  int getDefaultVOrigin() throws InvalidFontException {
	  return this.data.getint16 (4);
  }
  
  public int getVOrigin(int gid) throws InvalidFontException {
	  int max = this.data.getuint16(6);
	  int min = 0;
	  while (min < max)
	  {
		  int mid = (min + max) / 2;
		  int midGid = this.data.getuint16(8 + 4*mid);
		  if (gid < midGid) {
			  max = mid;
		  } else if (midGid < gid) {
			  min = mid+1;
		  }
		  else {
			  return this.data.getint16(8 + 4*mid + 2);
		  }
	  }
	  
	  return getDefaultVOrigin();
  }
  
  
  
  static void subsetAndStream(Subset subset, Map tables, VerticalOriginFetcher fetcher) throws InvalidFontException, UnsupportedFontException
  {
	  int numGlyphs = subset.getNumGlyphs();
	  int[] vorgs = new int[numGlyphs];
	  int[] vorgsCopied = new int [numGlyphs];
	  for (int i = 0; i < numGlyphs; i++)
	  {
		  int fullGid = subset.getFullGid(i);
		  vorgs[i] = fetcher.getVOrigin(fullGid);
	  }
	  System.arraycopy(vorgs, 0, vorgsCopied, 0, vorgs.length);
	  Arrays.sort(vorgs);
	  int newDefaultOrig = vorgs[0];
	  int mostCommonVorgCount = 1;
	  int thisVorgCount = 1;
	  for (int i = 1; i < numGlyphs; i++)
	  {
		  if (vorgs[i] == vorgs[i-1])
		  {
			  thisVorgCount++;
			  if (thisVorgCount > mostCommonVorgCount) {
				  mostCommonVorgCount = thisVorgCount;
				  newDefaultOrig = vorgs[i];
			  }
		  } else
		  {
			  thisVorgCount = 1;
		  }
	  }
	  
	  int nonDefaultCount = 0;
	  
	  for (int i = 0; i < numGlyphs; i++)
	  {
		  if (vorgs[i] != newDefaultOrig)
		  {
			  nonDefaultCount++;
		  }
	  }
	  
	  // allocate new buffer
	  OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(8 + 4*nonDefaultCount);
	  newData.setuint16(0, 1);
	  newData.setuint16(2, 0);
	  newData.setint16(4, newDefaultOrig);
	  newData.setuint16(6, nonDefaultCount);
	  int currentNonDefault = 0;
	  for (int i = 0; i < vorgsCopied.length && currentNonDefault < nonDefaultCount; i++)
	  {
		  if (vorgsCopied[i] != newDefaultOrig)
		  {
			  newData.setuint16(8 + currentNonDefault*4, i);
			  newData.setuint16(8 + currentNonDefault*4 + 2, vorgsCopied[i]);
			  currentNonDefault++;
		  }
	  }

	  tables.put (new Integer (Tag.table_VORG), newData);
	  
  }
  
  void stream (Map tables)
  throws InvalidFontException {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_VORG), newData);
  
  }  
}
