/*
 *
 *	File: FontFactory.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.type1;

import java.io.IOException;
import java.net.URL;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.postscript.SubArrays;
import com.adobe.fontengine.font.type1.Type1Font;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.type1.AFM;
import com.adobe.fontengine.font.type1.PFM;

/** Creates font objects for legacy type1 fonts. 
 *
 * <h4>Synchronization</h4>
 * 
 * This class represents a namespace
 * and does not contain instance data or mutable static data. It is therefore threadsafe.
 */
final public class FontFactory {
 
	// things that are found at the start of PFA fonts
	private static final byte[] kType1PFAStart1 = {'%','!','P','S','-','A','d','o','b','e','F','o','n','t'};
	private static final byte[] kType1PFAStart2 = {'%','!','F','o','n','t','T','y','p','e','1'};
	private static final byte[] kType1PFAStart3 = {'%','!','F','o','n','t','T','y','p','e',' ','1'};
    private static final byte[] kAFMStart = {'S','t','a','r','t','F','o','n','t','M','e','t','r','i','c','s'};
    
	
	// found at the start of PFB fonts
	private static final byte[] kType1PFBStart = {(byte)0x80, 0x01};
	private static final int maxT1BytesNeeded = 6 + kType1PFAStart1.length;
	
	/** Returns the number of initial bytes necessary to identify a stream as a type1 font */
	public static int getNumBytesNeededToIdentify()
	{
	    return maxT1BytesNeeded;
	}
	
	private static boolean startsWithType1(byte[] startingBytes, int offset) 
	{
	    if (SubArrays.arrayCompare(kType1PFAStart1, 0, startingBytes, offset, kType1PFAStart1.length)
	            || SubArrays.arrayCompare(kType1PFAStart2, 0, startingBytes, offset, kType1PFAStart2.length)
	            || SubArrays.arrayCompare(kType1PFAStart3, 0, startingBytes, offset, kType1PFAStart3.length))
	        return true;
	    return false;
	}
	
	private static boolean isPFA(byte[] startingBytes)
	{
	    if (startsWithType1(startingBytes, 0))
	        return true;
	    
	    return false;
	}
	
	private static boolean isPFB(byte[] startingBytes)
	{
	    if (SubArrays.arrayCompare(kType1PFBStart, 0, startingBytes, 0, kType1PFBStart.length)) {
	        if (startsWithType1(startingBytes, 6))
	            return true;
	        else
	            return false;
	    }

	    return false;
	}
	
    /**
     * Determines if startingBytes represents the start of a type1 font
     * @param startingBytes The initial file bytes
     * @return true iff the bytes appear to be the start of a type1 font
     */
	public static boolean isType1(byte[] startingBytes)
	{
	    if (isPFB(startingBytes) || isPFA(startingBytes))
	        return true;
	    
	    return false;
	}
	
	/** Returns the number of initial bytes necessary to identify a stream as an AFM */
    public static int getNumBytesNeededToIdentifyAFM()
    {
        return kAFMStart.length;
    }
    
    /**
     * Determines if startingBytes represents the start of an AFM
     * @param startingBytes The initial file bytes
     * @return true iff the bytes appear to be the start of a afm
     */
    public static boolean isAFM(byte[] startingBytes)
    {
        if (startingBytes.length < getNumBytesNeededToIdentifyAFM())
            return false;
        
        return SubArrays.arrayCompare(startingBytes, 0, kAFMStart, 0, kAFMStart.length);
    }
    
    /** Returns the number of initial bytes necessary to identify a stream as a PFM */
    public static int getNumBytesNeededToIdentifyPFM()
    {
        return 68;
    }
   
    /**
     * Determines if startingBytes represents the start of a PFM
     * @param startingBytes The initial file bytes
     * @return true iff the bytes appear to be the start of a pfm
     */
    public static boolean isPFM(byte[] startingBytes)
    {
        if (startingBytes.length < getNumBytesNeededToIdentifyPFM())
            return false;

        // look at the version, size and type to see if these look like valid pfm fields
        return startingBytes[0] == 0x00 && startingBytes[1] == 0x01 
        	&& (startingBytes[66] & 0xff) == 0x81 && startingBytes[67] == 0x0
        	&& (((startingBytes[3] & 0xff) << 8 | startingBytes[2] & 0xff) >= getNumBytesNeededToIdentify());
        
    }
    
    /**
     * Given an input stream, creates Type1Font objects that represent it.
     * 
     * Note that fonts are parsed at this point. Also note that multiple master
     * fonts are not supported by AFE. Calling this function with a multiple master
     * font will cause an exception to be thrown.
     * 
     * @param in The input stream
     * @return An array of type1 fonts
     * @throws IOException Thrown if the stream cannot be read
     * @throws InvalidFontException Thrown if the stream does not represent a valid type1 font.
     */    
	public static Type1Font[] load (FontInputStream in, URL url)
	    throws IOException, InvalidFontException, UnsupportedFontException
	{	    
	    Type1Font font = new Type1Font(in, url); 
	    Type1Font[] fontArr = new Type1Font[1];
	    fontArr[0] = font;
	    return fontArr;
	}
    
	   
    /**
     * If the stream contains an AFM, it creates an AFM and returns it.
     * 
     * Note that AFMs are parsed at this point
     * 
     * @param str The stream to be probed.
     * @return null if str is not an AFM. A valid AFM object otherwise.
     * @throws IOException Thrown if the stream cannot be read
     * @throws InvalidFontException Thrown if the stream does not represent a valid AFM
     */
    public static MetricFile loadAFM(FontInputStream str, URL url)
    	throws IOException, InvalidFontException
    {
        return AFM.createAFM(str, url);
    }
  
    /**
     * If the stream contains an PFM, it creates an PFM object and returns it.
     * 
     * Note that PFMs are parsed at this point
     * 
     * @param str The stream to be probed.
     * @return null if str is not an PFM. A valid PFM object otherwise.
     * @throws IOException Thrown if the stream cannot be read
     */
    public static MetricFile loadPFM(FontInputStream str, URL url)
    	throws IOException, InvalidFontException
    {
        return PFM.createPFM(str, url);
    }
}
