/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.rest.utils;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLEncoder;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.request.RequestPathInfo;

import com.adobe.granite.rest.Constants;

/**
 * The {@code URIUtils} provides helper methods around working with URIs.
 */
public final class URIUtils {

    private URIUtils() {
        
    }

    /**
     * URL encodes a path, ensuring that any forward slashes remain unencoded.
     * @param toEncode The path to be safely URL encoded.
     * @return A URL-encoded version of the path.
     */
    public static String urlEncodePath(String toEncode) {
        try {
            return URLEncoder.encode(toEncode, Constants.DEFAULT_CHARSET).replace("%2F", "/");
        } catch (UnsupportedEncodingException e) {
            throw new RuntimeException("Unable to encode URL: default charset is unexpected", e);
        }
    }
    
    /**
     * Relativizes the specified {@code resourcePath} against
     * {@code requestPathInfo}.
     * <p>
     * 
     * @param requestPathInfo RequestPathInfo to relativize against
     * @param resourcePath Resource path to relativize
     * @return The resulting path
     */
    public static String relativize(RequestPathInfo requestPathInfo, String resourcePath) {
        if (requestPathInfo == null) {
            throw new IllegalArgumentException("RequestPathInfo cannot be null");
        }
        if (resourcePath == null) {
            throw new IllegalArgumentException("Resource URI cannot be null");
        }
        
        String requestPath = requestPathInfo.getResourcePath();
        String uri = resourcePath;
        
        // is same - return last segment
        if ((requestPath).equals(resourcePath)) {
            String lastSegment = resourcePath.substring(resourcePath.lastIndexOf("/")+1);
            if (requestPathInfo.getSelectorString() != null) {
                lastSegment += "." + requestPathInfo.getSelectorString();
            }
            return lastSegment;
        }
        
        // is parent - return parent segment with traversal operators
        if (isParent(requestPath,resourcePath)) {
            String pathTraversal = "";
            String diff = requestPath.replaceFirst(resourcePath, "");
            String lastSegment = resourcePath.substring(resourcePath.lastIndexOf("/")+1);
            for (int i = 0; i < StringUtils.countMatches(diff, "/"); i++) {
                pathTraversal += "../";
            }
            if (!pathTraversal.isEmpty()) {
                return pathTraversal + lastSegment;
            }
        }
        
        // is child - return last path segment + child path
        URI base = URI.create(requestPath);
        URI relative = base.relativize(URI.create(resourcePath));
        if (!relative.toString().isEmpty()) {
            if (relative.toString().equals(resourcePath)) {
                return resourcePath;
            }
            String lastSegment = requestPath.substring(requestPath.lastIndexOf("/")+1);
            uri = lastSegment + "/" + relative.toString();
        }
        
        return uri;
    }

    /**
     * Checks if {@code resourcePath} is part of the {@code requestPath}.
     * 
     * @param requestPath
     * @param resourcePath
     * @return
     */
    private static boolean isParent(String requestPath, String resourcePath) {
        int selectorIdx = requestPath.indexOf(".");
        if (selectorIdx > -1) {
            requestPath = requestPath.substring(0, selectorIdx);
        }
        
        boolean isIndexOf = requestPath.indexOf(resourcePath) > -1;
        boolean isShorter = requestPath.length() > resourcePath.length();
        
        return isIndexOf && isShorter;
    }

}
