/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.exec;

import com.adobe.granite.workflow.HasMetaData;
import com.adobe.granite.workflow.exec.filter.WorkItemFilter;
import com.adobe.granite.workflow.model.WorkflowModel;

import java.util.Date;
import java.util.List;

/**
 * A <code>Workflow</code> represents the current state of an instance of a
 * <code>{@link WorkflowModel}</code>. In terms of petri nets the state of a
 * workflow instance can be considered as the collections of tokens (or
 * <code>{@link WorkItem}</code>s) that flow through the workflow instance.
 */
public interface Workflow extends HasMetaData {

    /**
     * Enumerates the possible states of a Workflow
     */
    public enum State {
        /**
         * Workflow is running
         */
        RUNNING,
        /**
         * Workflow is completed
         */
        COMPLETED,

        /**
         * Workflow is suspended
         */
        SUSPENDED,

        /**
         * Workflow is aborted
         */
        ABORTED
    }

    /**
     * Returns the <code>Workflow</code> instance ID.
     *
     * @return The <code>Workflow</code> instance ID.
     */
    String getId();

    /**
     * Returns the list of <code>{@link WorkItem}</code>s assigned to the
     * <code>Workflow</code> instance.
     *
     * @return The list of <code>{@link WorkItem}</code>s assigned to the
     *         <code>Workflow</code>
     */
    List<WorkItem> getWorkItems();

    /**
     * Returns a filtered list of <code>{@link WorkItem}</code>s assigned to the
     * <code>Workflow</code> instance.
     *
     * @param filter One is able to filter a subset of the available
     *               <code>{@link WorkItem}</code>s by passing a <code>{@link WorkItemFilter}</code> impl.
     * @return The list of <code>{@link WorkItem}</code>s assigned to the
     *         <code>Workflow</code>
     */
    List<WorkItem> getWorkItems(WorkItemFilter filter);

    /**
     * Returns the <code>{@link WorkflowModel}</code> that defines the
     * underlying workflow model of the <code>Workflow</code> instance.
     *
     * @return The <code>{@link WorkflowModel}</code> of the
     *         <code>Workflow</code>.
     */
    WorkflowModel getWorkflowModel();

    /**
     * Indicates if the <code>Workflow</code> instance is still active.
     *
     * @return True if active, otherwise false.
     */
    boolean isActive();

    /**
     * Returns the workflows state. In the default implementation this could be
     * one of <code>RUNNING</code>, <code>SUSPENDED</code>,
     * <code>COMPLETED</code> or <code>ABORTED</code>.
     *
     * @return The current workflows state.
     */
    String getState();

    /**
     * Returns the initiator of the <code>Workflow</code> instance.
     *
     * @return The user who initiated the <code>Workflow</code>.
     */
    String getInitiator();

    /**
     * Getter for the point in time where the <code>Workflow</code> instance
     * was started.
     *
     * @return The start time of the <code>Workflow</code> instance.
     */
    Date getTimeStarted();

    /**
     * Getter for the point in time where the <code>Workflow</code> was
     * finished.
     *
     * @return The end time of the <code>Workflow</code> instance.
     */
    Date getTimeEnded();

    /**
     * Returns the {@link WorkflowData} object assigned to the
     * <code>Workflow</code> instance.
     *
     * @return The {@link WorkflowData} object assigned to the
     *         <code>Workflow</code> instance.
     */
    WorkflowData getWorkflowData();
}
