/**
 * 
 */
package com.adobe.internal.io;

import java.io.File;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;

/**
 * A utiltiy class for working with File objects.
 *
 */
class FileUtility
{

	// see http://tools.ietf.org/rfc/rfc3986.txt
	private static final boolean[] LEGAL_URL_PCHARS =
	{
		false, false, false, false, false, false, false, false, // 0x00 - 0x07
		false, false, false, false, false, false, false, false, // 0x08 - 0x0f
		
		false, false, false, false, false, false, false, false, // 0x10 - 0x17
		false, false, false, false, false, false, false, false, // 0x18 - 0x1f
		
		false, true, false, false, true, false, true, true, 	// 0x20 - 0x27
		true, true, true, true, true, true, true, true, 		// 0x28 - 0x2f
		
		true, true, true, true, true, true, true, true, 		// 0x30 - 0x37
		true, true, true, true, false, true, false, false, 		// 0x38 - 0x3f
		
		true, true, true, true, true, true, true, true, 		// 0x40 - 0x47
		true, true, true, true, true, true, true, true, 		// 0x48 - 0x4f
		
		true, true, true, true, true, true, true, true, 		// 0x50 - 0x57
		true, true, true, false, false, false, false, true, 	// 0x58 - 0x5f
		
		false, true, true, true, true, true, true, true, 		// 0x60 - 0x67
		true, true, true, true, true, true, true, true, 		// 0x68 - 0x6f
		
		true, true, true, true, true, true, true, true, 		// 0x70 - 0x77
		true, true, true, false, false, false, true, false, 	// 0x78 - 0x7f
		
		false, false, false, false, false, false, false, false, // 0x80 - 0x87
		false, false, false, false, false, false, false, false, // 0x88 - 0x8f
		
		false, false, false, false, false, false, false, false, // 0x90 - 0x97
		false, false, false, false, false, false, false, false, // 0x98 - 0x9f
		
		false, false, false, false, false, false, false, false, // 0xa0 - 0xa7
		false, false, false, false, false, false, false, false, // 0xa8 - 0xaf
		
		false, false, false, false, false, false, false, false, // 0xb0 - 0xb7
		false, false, false, false, false, false, false, false, // 0xb8 - 0xbf
		
		false, false, false, false, false, false, false, false, // 0xc0 - 0xc7
		false, false, false, false, false, false, false, false, // 0xc8 - 0xcf
		
		false, false, false, false, false, false, false, false, // 0xd0 - 0xd7
		false, false, false, false, false, false, false, false, // 0xd8 - 0xdf
		
		false, false, false, false, false, false, false, false, // 0xe0 - 0xe7
		false, false, false, false, false, false, false, false, // 0xe8 - 0xef
		
		false, false, false, false, false, false, false, false, // 0xf0 - 0xf7
		false, false, false, false, false, false, false, false, // 0xf8 - 0xff
	};
	
	/**
	 * 
	 */
	private FileUtility() 
	{
		// should never be instantiated
	}

	/**
	 * Turn a File into a URL following the rules of {@href http://tools.ietf.org/rfc/rfc3986.txt RFC3986}.
	 * @param file the file
	 * @return a properly constructed URL
	 * @throws MalformedURLException
	 */
	public static URL fileToURL(File file) 
	throws MalformedURLException
	{
		URI uri = file.toURI();
		return new URL("file", "", normalizeForURL(uri.getPath()));
	}

	/**
	 * RFC3986 basically says that only a subset of low ASCII can go into a URL and everything
	 * else needs to be escapped.  Non-ASCII is to be first turned into UTF-8 and then escaped.
	 * @param filePath
	 * @return normalized url
	 */
	private static String normalizeForURL(String filePath) 
	{
		StringBuilder sb = new StringBuilder();
		byte[] b;
		try {
			b = filePath.getBytes("UTF8");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("Unable to get UTF-8 encoding.", e);
		}
		for (int i = 0; i < b.length; i++)
		{
			if (LEGAL_URL_PCHARS[(b[i] & 0xff)])
			{
				sb.append((char) b[i]);
			} else {
				sb.append('%');
				sb.append(Integer.toHexString(b[i] & 0xff));
			}
		}
		return sb.toString();
	}
}
