/* ****************************************************************************
 *
 *	File: CosLinearization.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import com.adobe.internal.io.ByteWriterFactory.Fixed;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * CosLinearization
 *
 * Helper module for CosDocument.saveLinear
 *
 * @author peters
 */
class CosLinearization {
	// Constants
	private static final int DEFAULT_WIDTH = 612;		// Default page width (points)
	private static final int DEFAULT_HEIGHT = 792;		// Default page height (points)
	private static final int OBJSTM_MAXNUMOBJS = 100;	// Never put more than this many in any one object stream

	// Object processing depth
	private static final int PROCESS_SINGLE = 0;		// Just this object
	private static final int PROCESS_SHALLOW = 1;		// This object and immediate indirect descendants
	private static final int PROCESS_DEEP = 2;		// This object any any indirect descendants
	private static final int PROCESS_SECONDARY = 3;		// The rest of what was done with PROCESS_SINGLE
	private static final int PROCESS_TERTIARY = 4;		// The rest of what was done with PROCESS_SHALLOW

	// Basic stuff
	private CosDocument mCosDoc;
	private CosDictionary mCatalog;

	// Linearization control data
	private int mNumPages;
	private int mCurPageNum = -1;
	private boolean mCatalogEnable = false;
	private boolean mPageObjEnable = false;
	private boolean mStreamsEnable = false;
	private int mNumEmbFiles;
	private int mSpecialPageNum = -1;			// Could be target of OpenAction
	private CosDictionary mSpecialPageObj;			// But Acrobat always uses page zero
	private int mCurEnumTag;				// Prevent object reference cycles
	private boolean mUseBookmarks;				// Outlines with special page if true
	private CosList mPageContents = new CosList();		// For special case page hints
	private CosList mPageThumbObjs = new CosList();		// For special case thumb hints

	// The object collections
	private LinObjCol mCurPageCol;
	private LinObjCol mCurEmbFileCol;
	private CosList mMasterList = new CosList();		// List of all indirect objects processed
	private CosList mColArray = new CosList();

	// State stack and pointer
	private Object[] mStateStack = null;
	private int mStateStackPtr = 0;

	// Other data
	private int mXRefStyle;					// XREFTABLE, XREFHYBRID or XREFSTREAM
	private CosListInt mNewToOldObjNumMap;			// Map of newly assigned object numbers to old
	private CosListInt mNewToOldObjGenMap;			// Map of newly assigned object numbers to old generation
	private CosList mOldToOldObjStmMap;			// Map of old object streams relative to old object number
	private CosListInt mOldToNewObjNumMap;			// Map of old object numbers to newly assigned ones

	// The object collections
	private CosList mLinArray;				// COLTYPE_LINEARIZATION
	private CosList mPageArrays;				// COLTYPE_PAGE, array of arrays
	private CosList mPageArraysSh;				// COLTYPE_PAGE, shared, array of arrays
	private CosList mSplPageDeferredArray;			// COLTYPE_PAGE, deferred, special page only
	private CosList mEmbFileArrays;				// COLTYPE_EMBFILE, array of arrays
	private CosList mThumbArray;				// COLTYPE_THUMB
	private CosList mThumbArraySh;				// COLTYPE_THUMB, shared
	private CosList mAcroFormArray;				// COLTYPE_ACROFORM
	private CosList mAcroFormArraySh;			// COLTYPE_ACROFORM, shared
	private CosList mStructureArray;			// COLTYPE_STRUCTURE
	private CosList mStructureArraySh;			// COLTYPE_STRUCTURE, shared
	private CosList mRenditionArray;			// COLTYPE_RENDITION
	private CosList mRenditionArraySh;			// COLTYPE_RENDITION, shared
	private CosList mMediaClipArray;			// COLTYPE_MEDIACLIP
	private CosList mMediaClipArraySh;			// COLTYPE_MEDIACLIP, shared
	private CosList mOutlineArray;				// COLTYPE_OUTLINE
	private CosList mThreadArray;				// COLTYPE_THREAD
	private CosList mDestArray;				// COLTYPE_DEST
	private CosList mInfoDictArray;				// COLTYPE_INFODICT
	private CosList mPageLabelArray;			// COLTYPE_PAGELABEL
	private CosList mOtherArray;				// COLTYPE_OTHER
	private CosList mMainSection;				// Main section master list
	private CosList mMainSectionCompressed;			// Main section compressed list
	private CosList mUpdateSection;				// Update section master list
	private CosList mUpdateSectionCompressed;		// Update section compressed list
	private CosList mSharedObjects;				// Main section shared objects

	// Hint generation related data
	private CosList mObjectInfos;				// Passed in to buildHintStream
	private CosStream mHintStm;				// Passed in to buildHintStream
	private int mMainLastObjNum;				// Passed in to buildHintStream
	private int mUpdateLastObjNum;				// Passed in to buildHintStream
	private long mHintStmPos;				// Passed in to buildHintStream
	private long mObjEndPos;				// Passed in to buildHintStream
	private OutputByteStream mHintOutputStream;		// Where the hint stream is built
	private int mNumSplPageSharedObjs;			// Count of first page shared objects
	private int mNumSharedObjs;				// Total count of distinct shared objects
	private int[] mSharedObjLengths;			// Ordinary array of shared object lengths
	private CosListInt mSharedObjsNumMap;			// Map of object number to shared object number
	private CosList mSharedPageObjArrays;			// Array of shared objects per page
	private int[] mPageNumPrivateObjs;			// Number of non-shared objects on each page
	private int[] mPagePrivateLen;				// Length of individual page object areas
	private int[] mPageNumSharedObjs;			// Number of shared objects on each page
	private int mNumThumbPages;				// Number of pages with thumbnails
	private CosObject[] mThumbPageObject;			// First thumb object in page thumb group
	private int[] mThumbNumNoThumbs;			// Number of preceeding pages with no thumbnails
	private int[] mThumbNumObjects;				// Number of objects in this page's thumbnail
	private int[] mThumbTotalLen;				// Length of this page's private thumb object area
	private int[] mEmbFileFirstObjNum;			// Object number of first object in an embedded file
	private int[] mEmbFileNumObjs;				// Number of objects in an embedded file group
	private int[] mEmbFileLen;				// Length of individual embedded file object areas

	// Tag for LinObjInfo
	private int mLinObjInfoSN;				// CosLinObjInfo serial number assignment variable

	/**
	 * The constructor, requires a COS document to work with
	 */
	CosLinearization(CosDocument doc)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mCosDoc = doc;
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_LINEARIZATION));
		mColArray.add(new CosList()); 		// LinObjCol.COLTYPE_PAGE
		mColArray.add(new CosList()); 		// LinObjCol.COLTYPE_EMBFILE
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_THUMB));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_ACROFORM));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_STRUCTURE));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_RENDITION));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_MEDIACLIP));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_OUTLINE));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_THREAD));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_DEST));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_INFODICT));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_PAGELABEL));
		mColArray.add(new LinObjCol(LinObjCol.COLTYPE_OTHER));

		// Get the basics set up
		mCatalog = mCosDoc.getRoot();
		CosDictionary mInfoDict = mCosDoc.getInfo();
		CosDictionary mPageTree = mCatalog.getCosDictionary(ASName.k_Pages);
		mUseBookmarks = (safeGetName(mCatalog, ASName.k_PageMode) == ASName.k_UseOutlines);

		// First let's get the special objects out of the way
		mStreamsEnable = true;
		processLinObjs();

		// Fix up page tree pages objects. No values can be allowed
		// to be inherited, lest out of sequence objects be fetched.
		// Enumerate pages as we go, checking all referenced objects
		// and looking for shared ones.
		CosDictionary resObj = mCosDoc.createDirectCosDictionary();
		CosArray mbObj = mCosDoc.createCosArray();
		mbObj.addInt(0);
		mbObj.addInt(0);
		mbObj.addInt(DEFAULT_WIDTH);
		mbObj.addInt(DEFAULT_HEIGHT);
		CosArray cbObj = mbObj;
		CosNumeric rotObj = mCosDoc.createCosNumeric(0);
		mCurPageNum = 0;
		processPageTreeNode(mPageTree, resObj, mbObj, cbObj, rotObj);
		mNumPages = mCurPageNum;
		mCurPageNum = -1;

		// AcroForms
		mCurEnumTag++;
		processObj(safeGetDict(mCatalog, ASName.k_AcroForm), LinObjCol.COLTYPE_ACROFORM, PROCESS_SECONDARY, false, false);

		// Structure tree
		mCurEnumTag++;
		mStreamsEnable = false;
		processObj(safeGetDict(mCatalog, ASName.k_StructTreeRoot), LinObjCol.COLTYPE_STRUCTURE, PROCESS_DEEP, false, false);
		mStreamsEnable = true;

		// Renditions and media clips
		CosDictionary namesObj = safeGetDict(mCatalog, ASName.k_Names);
		if (namesObj != null) {
			mCurEnumTag++;
			processObj(safeGetDict(namesObj, ASName.k_Renditions), LinObjCol.COLTYPE_RENDITION, PROCESS_DEEP, false, false);
			mCurEnumTag++;
			processObj(safeGetDict(namesObj, ASName.k_MediaClips), LinObjCol.COLTYPE_MEDIACLIP, PROCESS_DEEP, false, false);
		}

		// Bookmarks
		if (!mUseBookmarks) {
			CosDictionary outlinesObj = safeGetDict(mCatalog, ASName.k_Outlines);
			processObj(outlinesObj, LinObjCol.COLTYPE_OUTLINE, PROCESS_DEEP, false, false);
		}

		// Dests
		CosDictionary destsObj = null;
		if (namesObj != null)
			destsObj = safeGetDict(namesObj, ASName.k_Dests);
		if (destsObj == null)
			destsObj = safeGetDict(mCatalog, ASName.k_Dests);
		mCurEnumTag++;
		processObj(destsObj, LinObjCol.COLTYPE_DEST, PROCESS_DEEP, false, false);

		// Other names stuff we didn't get yet
		mCurEnumTag++;
		processObj(namesObj, LinObjCol.COLTYPE_OTHER, PROCESS_SECONDARY, false, false);

		// The info dictionary
		mCurEnumTag++;
		processObj(mInfoDict, LinObjCol.COLTYPE_INFODICT, PROCESS_DEEP, false, false);

		// Page labels
		mCurEnumTag++;
		processObj(safeGetDict(mCatalog, ASName.k_PageLabels), LinObjCol.COLTYPE_PAGELABEL, PROCESS_DEEP, false, false);

		// Everything else
		mCurEnumTag++;
		mPageObjEnable = true;
		mStreamsEnable = true;
		processCatalog(LinObjCol.COLTYPE_OTHER);
		mPageObjEnable = false;
		mStreamsEnable = false;
	}

	/**
	 * These are the special objects that must come first in the
	 * file. There are (up to) four magic ones that aren't seen
	 * at this level: the linearization dictionary, the update
	 * section XRef stream (if present), the linearization hints
	 * stream, and the encryption dictionary (if present).
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processLinObjs()
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mCurEnumTag++;
		CosObject other;

		// The document catalog
		processCatalog(LinObjCol.COLTYPE_LINEARIZATION);

		// The PageMode, if any and indirect
		other = mCatalog.get(ASName.k_PageMode);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_DEEP, false, false);

		// The OpenAction, if any
		other = mCatalog.get(ASName.k_OpenAction);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_DEEP, false, false);

		// The AcroForm dict, if any
		other = mCatalog.get(ASName.k_AcroForm);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_SINGLE, false, false);

		// The Names dict, if any; Acrobat does this, but it's not documented as such
		other = mCatalog.get(ASName.k_Names);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_SINGLE, false, false);

		// The threads array, if any
		other = mCatalog.get(ASName.k_Threads);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_SHALLOW, false, false);

		// Optional content, if any
		other = mCatalog.get(ASName.k_OCProperties);
		processObj(other, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_DEEP, false, false);

		// Possibly do outlines now
		if (mUseBookmarks) {
			CosDictionary outlinesObj = safeGetDict(mCatalog, ASName.k_Outlines);
			processObj(outlinesObj, LinObjCol.COLTYPE_LINEARIZATION, PROCESS_DEEP, false, false);
		}
	}

	/**
	 * Recursively walk the page tree nodes and fix up any that are
	 * missing any of "Resources", "MediaBox", "CropBox" or "Rotate".
	 * These keys are inheritable so if they aren't present on any
	 * given page Acrobat may go reading out of sequence objects
	 * to find them in parent nodes. After we have fixed up a leaf
	 * node, we enumerate it's objects for proper ordering.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processPageTreeNode(CosDictionary node, CosDictionary res, CosArray mb, CosArray cb, CosNumeric rot)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mCurEnumTag++;
		if (node.containsKey(ASName.k_Kids)) {
			// Get any default keys from the Pages node
			// that need to be passed down to pages below
			CosDictionary dict;
			CosArray array;
			CosNumeric number;
			dict = safeGetDict(node, ASName.k_Resources);
			if (dict != null) {
				res = dict;
				node.remove(ASName.k_Resources);
			}
			array = safeGetArray(node, ASName.k_MediaBox);
			if (array != null) {
				mb = array;
				cb = array;
				node.remove(ASName.k_MediaBox);
			}
			array = safeGetArray(node, ASName.k_CropBox);
			if (array != null) {
				cb = array;
				node.remove(ASName.k_CropBox);
			}
			number = safeGetNumeric(node, ASName.k_Rotate);
			if (number != null) {
				rot = number;
				node.remove(ASName.k_Rotate);
			}

			// Now catalog the pages node and any
			// immediate indirect children and finally
			// enumerate the Kids array and recur
			processObj(node, LinObjCol.COLTYPE_OTHER, PROCESS_SHALLOW, false, false);
			CosArray kidsArray = node.getCosArray(ASName.k_Kids);
			Iterator iter = kidsArray.iterator();
			while (iter.hasNext()) {
				CosDictionary kid = (CosDictionary)iter.next();
				processPageTreeNode(kid, res, mb, cb, rot);
			}
		} else {
			if (!node.containsKey(ASName.k_Resources))
				node.put(ASName.k_Resources, res);
			if (node.containsKey(ASName.k_MediaBox))
				cb = safeGetArray(node, ASName.k_MediaBox);
			else
				node.put(ASName.k_MediaBox, mb);
			if (!node.containsKey(ASName.k_CropBox))
				node.put(ASName.k_CropBox, cb);
			if (!node.containsKey(ASName.k_Rotate))
				node.put(ASName.k_Rotate, rot);
			processPageObj(node);
		}
	}

	/**
	 * Do the page related processing for the objects on one page.
	 * Check to see if this is the "special" page object and, if so,
	 * set it's page number. This will invoke special treatment as
	 * well as letting CosLinearization know which was the special
	 * page. For some reason, Acrobat always wants this to be page
	 * zero, but we could set if from the OpenAction just as well.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processPageObj(CosDictionary pageObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		// Figure out if we're on the "special" page
		// Then set up a page collection for this page
		if (mCurPageNum == 0 && mSpecialPageObj == null)
			mSpecialPageObj = pageObj;
		if (pageObj == mSpecialPageObj)
			mSpecialPageNum = mCurPageNum;
		mCurPageCol = new LinObjCol(LinObjCol.COLTYPE_PAGE);
		((CosList)mColArray.get(LinObjCol.COLTYPE_PAGE)).add(mCurPageCol);
		boolean forceShared = (mSpecialPageNum == mCurPageNum);

		// Increment the enumeration tag and start processing
		// We use the same enumeration tag for the entire page
		processObj(pageObj, LinObjCol.COLTYPE_PAGE, PROCESS_SINGLE, forceShared, false);
		CosArray pageAnnots = safeGetArray(pageObj, ASName.k_Annots);
		processObj(pageAnnots, LinObjCol.COLTYPE_PAGE, PROCESS_SHALLOW, forceShared, false);
		CosArray pageBeads = safeGetArray(pageObj, ASName.k_B);
		processObj(pageBeads, LinObjCol.COLTYPE_PAGE, PROCESS_SHALLOW, forceShared, false);
		CosDictionary pageResObj = safeGetDict(pageObj, ASName.k_Resources);
		processPageRes(pageResObj);
		CosObject pageContObj;
		pageContObj = pageObj.get(ASName.k_Contents);
		mPageContents.add(pageContObj);
		processObj(pageContObj, LinObjCol.COLTYPE_PAGE, PROCESS_DEEP, forceShared, false);
		processObj(pageAnnots, LinObjCol.COLTYPE_PAGE, PROCESS_TERTIARY, forceShared, false);

		// Save thumb object for hints even if it's null
		CosDictionary thumbObj = safeGetDict(pageObj, ASName.k_Thumb);
		mPageThumbObjs.add(thumbObj);
		// If CS is complex, it's required to be indexed only
		if (thumbObj != null) {
			CosArray thumbCS = safeGetArray(thumbObj, ASName.k_ColorSpace);
			processObj(thumbCS, LinObjCol.COLTYPE_THUMB, PROCESS_DEEP, true, false);
			processObj(thumbObj, LinObjCol.COLTYPE_THUMB, PROCESS_DEEP, false, false);
		}

		// Anything else goes to the back of the line
		processObj(pageObj, LinObjCol.COLTYPE_OTHER, PROCESS_SECONDARY, false, false);
		mCurPageCol = null;
		mCurPageNum++;
	}

	/**
	 * Process a page's resource dictionary. Some resources are
	 * forced to be shared so they could be sandwiched between
	 * content slices. We don't slice contents, but we follow
	 * Acrobat's lead on the categories. Other resources, like
	 * fauxable fonts and images, can really be set to be usefully
	 * shared even without slicing. These can be aliased by the
	 * Viewer and then cleaned up later. We set these to "defer"
	 * which produces a numerator of "8" in the Acrobat slicing
	 * system.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processPageRes(CosDictionary pageResObj)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		// Go over the various resource categories
		if (pageResObj == null)
			return;
		boolean forceShared = (mSpecialPageNum == mCurPageNum);
		CosDictionary pageCSs = safeGetDict(pageResObj, ASName.k_ColorSpace);
		processObj(pageCSs, LinObjCol.COLTYPE_PAGE, PROCESS_DEEP, forceShared, false);
		CosDictionary pageFonts = safeGetDict(pageResObj, ASName.k_Font);
		if (pageFonts != null) {
			Iterator iter = pageFonts.keyIterator();
			while (iter.hasNext()) {
				ASName key = (ASName)iter.next();
				CosDictionary font = safeGetDict(pageFonts, key);
				if (font == null)
					continue;
				ASName baseFont = safeGetName(font, ASName.k_BaseFont);
				if (baseFont == ASName.k_Symbol || baseFont == ASName.k_ZapfDingbats)
					continue;
				CosDictionary fontDesc = safeGetDict(font, ASName.k_FontDescriptor);
				if (fontDesc != null) {
					int flags = safeGetInt(fontDesc, ASName.k_Flags);
					if ((flags & 0x20) == 0)	// PD_STD_ENCODING
						continue;
				}
				// Share and defer fauxable font
				processObj(font, LinObjCol.COLTYPE_PAGE, PROCESS_DEEP, forceShared, true);
			}
		}
		CosDictionary pageXObjects = safeGetDict(pageResObj, ASName.k_XObject);
		if (pageXObjects != null) {
			Iterator iter = pageXObjects.keyIterator();
			while (iter.hasNext()) {
				ASName key = (ASName)iter.next();
				CosDictionary xObject = safeGetDict(pageXObjects, key);
				if (xObject == null)
					continue;
				ASName subType = safeGetName(xObject, ASName.k_Subtype);
				if (subType != ASName.k_Image)
					continue;
				// Defer image
				processObj(xObject, LinObjCol.COLTYPE_PAGE, PROCESS_DEEP, forceShared, true);
			}
		}
		// Process the rest of the resources as normal page objects which may or may not be shared
		processObj(pageResObj, LinObjCol.COLTYPE_PAGE, PROCESS_DEEP, forceShared, false);
	}

	/**
	 * Process the document catalog to get any left over objects.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processCatalog(int colType)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mCatalogEnable = true;
		processObj(mCatalog, colType, PROCESS_SINGLE, false, false);
		if (colType != LinObjCol.COLTYPE_LINEARIZATION) {
			Iterator iter = mCatalog.keyIterator();
			while (iter.hasNext()) {
				ASName key = (ASName)iter.next();
				CosObject value;
				value = mCatalog.get(key);
				if (key == ASName.k_Threads)
					processThreadsArray(value);
				else
					processObj(value, colType, PROCESS_DEEP, false, false);
			}
		}
		mCatalogEnable = false;
	}

	/**
	 * Process threads array. Put thread pointers in beads.
	 * Process any beads that were left over from deleted pages
	 * so as not to break the chain. Follow the chains explicitly
	 * to avoid stack overflow.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processThreadsArray(CosObject threadsArray)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (threadsArray instanceof CosArray) {
			Iterator arrayIter = ((CosArray)threadsArray).iterator();
			while (arrayIter.hasNext()) {
				CosObject threadDict = (CosObject)arrayIter.next();
				if (threadDict instanceof CosDictionary) {
					CosDictionary firstBead = safeGetDict((CosDictionary)threadDict, ASName.k_F);
					if (firstBead != null) {
						addObjToCol(firstBead, LinObjCol.COLTYPE_THREAD, false, false);
						CosDictionary nextBead = safeGetDict(firstBead, ASName.k_N);
						while (nextBead != firstBead && nextBead != null) {
							if (!(nextBead.containsKey(ASName.k_T))){
								nextBead.put(ASName.k_T, threadDict);
							}
							addObjToCol(nextBead, LinObjCol.COLTYPE_THREAD, false, false);
							nextBead = safeGetDict(nextBead, ASName.k_N);
						}
					}
					CosDictionary threadInfo = safeGetDict((CosDictionary)threadDict, ASName.k_I);
					processObj(threadInfo, LinObjCol.COLTYPE_THREAD, PROCESS_DEEP, false, false);
				}
			}
		}
	}

	/**
	 * Process any single COS object. Depending on "level" we
	 * iterate to process it's first level children or the whole
	 * tree. First level processing is only meaning for indirect
	 * objects. Indirect objects are checked against the master
	 * list to see if we've ever seen them, and then tested for
	 * possible sharing. Any object we already know about we
	 * first tested to see if it's ever be processed on THIS pass
	 * of iteration, and if so we skip it. This avoids deadly
	 * object reference loops and also improves efficiency.
	 */
	private void processObj(CosObject obj, int colType, int level, boolean forceShared, boolean defer)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (mStateStack == null)
			mStateStack = new Object[65536];
		CosContainer container = null;
		Iterator containerIter = null;

		while (true) {
			while (true) {
				if (obj == null || obj instanceof CosNull)
					break;
				if (obj instanceof CosDictionary) {
					// Special case hacks. Should be ordered to be fast.
					if (obj instanceof CosStream && !mStreamsEnable)
						break;
					CosObject cosType = ((CosDictionary)obj).get(ASName.k_Type);
					if (cosType instanceof CosName) {
						ASName type = cosType.nameValue();
						if (type == ASName.k_Catalog && !mCatalogEnable) {
							break;
						} else if (type == ASName.k_Page && !(mPageObjEnable || level == PROCESS_SINGLE)) {
							break;
						} else if (type == ASName.k_Filespec || type == ASName.k_F) {
							CosDictionary efDict = safeGetDict((CosDictionary)obj, ASName.k_EF);
							if (efDict != null) {
								Object[] saveStack = mStateStack;
								int saveStackPtr = mStateStackPtr;
								mStateStack = null;
								mStateStackPtr = 0;
								processEmbFile(efDict);
								mStateStack = saveStack;
								mStateStackPtr = saveStackPtr;
							}
						}
					}
					// Get rid of ugly indirect length values
					CosObject length = ((CosDictionary)obj).get(ASName.k_Length);
					if (length instanceof CosNumeric && length.isIndirect()) {
						Number value = ((CosNumeric)length).numberValue();
						length = mCosDoc.createCosNumeric(value);
						((CosDictionary)obj).put(ASName.k_Length, length);
					}
				}
				int objNum = obj.getObjNum();
				if (objNum != 0 && level != PROCESS_SECONDARY && level != PROCESS_TERTIARY) {
					// It's an indirect object
					if (addObjToCol(obj, colType, forceShared, defer) == null)
						break;
				}
				if (level == PROCESS_SINGLE || !(obj instanceof CosContainer))
					break;
				if (containerIter != null) {
					mStateStack[mStateStackPtr++] = container;
					mStateStack[mStateStackPtr++] = containerIter;
					mStateStack[mStateStackPtr++] = Integer.valueOf(level);
				}
				container = (CosContainer)obj;
				if (container instanceof CosArray) {
					containerIter = ((CosArray)container).iterator();
				} else {
					List<ASName> keys = ((CosDictionary)container).getKeys();
					containerIter = keys.iterator();
				}
				if (level == PROCESS_SHALLOW)
					level = PROCESS_SINGLE;
				else if (level == PROCESS_TERTIARY)
					level = PROCESS_SECONDARY;
				else if (level == PROCESS_SECONDARY)
					level = PROCESS_DEEP;
				break;
			}
			if (containerIter == null)
				return;
			while (!containerIter.hasNext()) {
				if (mStateStackPtr == 0)
					return;
				level = ((Integer)mStateStack[--mStateStackPtr]).intValue();
				containerIter = (Iterator)mStateStack[--mStateStackPtr];
				container = (CosContainer)mStateStack[--mStateStackPtr];
			}
			if (container instanceof CosArray) {
				obj = (CosObject)containerIter.next();
			} else {
				ASName key = (ASName)containerIter.next();
				obj = null;
				if (key == ASName.k_Parent || key == ASName.k_SE)
					continue;
				obj = ((CosDictionary)container).get(key);
			}
		}
	}

	/**
	 * Process a potential embedded file set. Create a collection if
	 * there's really anything here. The passed dictionary is the value
	 * of the "EF" key in a Filespec dictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private void processEmbFile(CosDictionary efDict)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		mCurEmbFileCol = new LinObjCol(LinObjCol.COLTYPE_EMBFILE);
		processObj(safeGetDict(efDict, ASName.k_F), LinObjCol.COLTYPE_EMBFILE, PROCESS_DEEP, false, false);
		processObj(safeGetDict(efDict, ASName.k_DOS), LinObjCol.COLTYPE_EMBFILE, PROCESS_DEEP, false, false);
		processObj(safeGetDict(efDict, ASName.k_Mac), LinObjCol.COLTYPE_EMBFILE, PROCESS_DEEP, false, false);
		processObj(safeGetDict(efDict, ASName.k_Unix), LinObjCol.COLTYPE_EMBFILE, PROCESS_DEEP, false, false);
		if (mCurEmbFileCol.curSize() != 0) {
			((CosList)mColArray.get(LinObjCol.COLTYPE_EMBFILE)).add(mCurEmbFileCol);
			mNumEmbFiles++;
		}
		mCurEmbFileCol = null;
	}

	/**
	 * Here we add a object to a collection and fix up the share state
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 */
	private LinObjInfo addObjToCol(CosObject obj, int colType, boolean forceShared, boolean defer)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		LinObjCol col;
		LinObjInfo info;
		int objNum = obj.getObjNum();
		if (objNum == 0)
			return null;
		info = (LinObjInfo)mMasterList.get(objNum);
		// If we've seen this object on this enumeration pass, do nothing.
		if (info != null && info.getEnumTag() == mCurEnumTag)
			return null;
		if (colType == LinObjCol.COLTYPE_PAGE)
			col = mCurPageCol;
		else if (colType == LinObjCol.COLTYPE_EMBFILE)
			col = mCurEmbFileCol;
		else
			col = (LinObjCol)mColArray.get(colType);
		boolean canShare = col.canShare();
		boolean canCollect = false;

		// Make a new LinObjInfo the first time we see a CosObject
		if (info == null) {
			// New, never seen this object before
			if (!canShare)
				forceShared = false;
			info = new LinObjInfo(colType, canShare, obj.getInfo());
			// These only apply to the special page
			if (forceShared)
				info.makeShared();
			else
				defer = false;
			if (defer)
				info.makeDeferred();
			mMasterList.add(objNum, info);
			canCollect = true;
		} else {
			// Old, we can only collect it if it's sharable
			if (canShare && info.canShare()) {
				if (colType == LinObjCol.COLTYPE_THUMB) {
					// Thumbs can only share with other thumbs
					if (info.getType() == LinObjCol.COLTYPE_THUMB)
						canCollect = true;
				} else {
					// We can assign this a standard shared object reference
					canCollect = true;
				}
				if (canCollect)
					info.makeShared();
			}
		}
		if (canCollect && !col.hasInfo(objNum))
			col.putInfo(objNum, info);
		info.setEnumTag(mCurEnumTag);
		return info;
	}

	// Utilities for safely retrieving dictionary values
	private static final int safeGetInt(CosDictionary dict, ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
		obj = dict.get(key);
		if (obj instanceof CosNumeric)
			return obj.intValue();
		return 0;
	}

	private static final ASName safeGetName(CosDictionary dict, ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
		obj = dict.get(key);
		if (obj instanceof CosName)
			return obj.nameValue();
		return null;
	}

	private static final CosNumeric safeGetNumeric(CosDictionary dict, ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
		obj = dict.get(key);
		if (obj instanceof CosNumeric)
			return (CosNumeric)obj;
		return null;
	}

	private static final CosDictionary safeGetDict(CosDictionary dict, ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
		obj = dict.get(key);
		if (obj instanceof CosDictionary)
			return (CosDictionary)obj;
		return null;
	}

	private static final CosArray safeGetArray(CosDictionary dict, ASName key)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj;
		obj = dict.get(key);
		if (obj instanceof CosArray)
			return (CosArray)obj;
		return null;
	}

	/**
	 * 
	 *
	 * Return linearized update section lists
	 */
	CosList getMainSectionList() {return extractObjInfos(mMainSection);}

	/**
	 * 
	 *
	 * Return linearized update section lists
	 */
	CosList getMainSectionCompressedList() {return extractObjInfos(mMainSectionCompressed);}

	/**
	 * 
	 */
	CosList getUpdateSectionList() {return extractObjInfos(mUpdateSection);}

	/**
	 * 
	 */
	CosList getUpdateSectionCompressedList() {return extractObjInfos(mUpdateSectionCompressed);}

	// Other accessors

	/**
	 * 
	 */
	int getNumPages() {return mNumPages;}

	/**
	 * 
	 */
	CosDictionary getSpecialPageObj() {return mSpecialPageObj;}

	/**
	 * 
	 */
	void setNewToOldObjNumMap(CosListInt map) {mNewToOldObjNumMap = map;}

	/**
	 * 
	 */
	void setNewToOldObjGenMap(CosListInt map) {mNewToOldObjGenMap = map;}

	/**
	 * 
	 */
	void setOldToOldObjStmMap(CosList map) {mOldToOldObjStmMap = map;}

	/**
	 * 
	 */
	CosList getOldToOldObjStmMap() {return mOldToOldObjStmMap;}

	/**
	 * 
	 */
	void setOldToNewObjNumMap(CosListInt map) {mOldToNewObjNumMap = map;}

	/**
	 * 
	 *
	 * This is the first part of the COS level linearization work.
	 * We sort all the object collections and come up with two lists,
	 * one for the "update" section and one for the "main" section.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	void buildLinearizationData(int xrefStyle)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// We need to know the style right away. If it's HYBRID, the XRefStream
		// object set will have a different series of object numbers. If it's
		// HYBRID, we only put ObjStream objects in the XRefStream.
		mXRefStyle = xrefStyle;
		extractAndSortCollectionArrays();
		extractSharedObjects();
		buildLinearSections();
		mColArray = null;
	}

	/**
	 * Here we extract all the collections and sort
	 * them by serial number.
	 */
	private void extractAndSortCollectionArrays()
	{
		// Create and sort all the arrays by serial number
		// COLTYPE_LINEARIZATION
		mLinArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_LINEARIZATION)).getObjList();

		// COLTYPE_PAGE
		mPageArrays = new CosList();
		CosList pageCols = (CosList)mColArray.get(LinObjCol.COLTYPE_PAGE);
		for (int i = 0; i < mNumPages; i++) {
			CosList pageArray = ((LinObjCol)pageCols.get(i)).getObjList();
			mPageArrays.add(pageArray);
		}

		// COLTYPE_EMBFILE
		mEmbFileArrays = new CosList();
		CosList embFileCols = (CosList)mColArray.get(LinObjCol.COLTYPE_EMBFILE);
		for (int i = 0; i < mNumEmbFiles; i++) {
			CosList embFileArray = ((LinObjCol)embFileCols.get(i)).getObjList();
			mEmbFileArrays.add(embFileArray);
		}

		// COLTYPE_THUMB
		mThumbArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_THUMB)).getObjList();

		// COLTYPE_ACROFORM
		mAcroFormArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_ACROFORM)).getObjList();

		// COLTYPE_STRUCTURE
		mStructureArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_STRUCTURE)).getObjList();

		// COLTYPE_RENDITION
		mRenditionArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_RENDITION)).getObjList();

		// COLTYPE_MEDIACLIP
		mMediaClipArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_MEDIACLIP)).getObjList();

		// COLTYPE_OUTLINE
		mOutlineArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_OUTLINE)).getObjList();

		// COLTYPE_THREAD
		mThreadArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_THREAD)).getObjList();

		// COLTYPE_DEST
		mDestArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_DEST)).getObjList();

		// COLTYPE_INFODICT
		mInfoDictArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_INFODICT)).getObjList();

		// COLTYPE_PAGELABEL
		mPageLabelArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_PAGELABEL)).getObjList();

		// COLTYPE_OTHER
		mOtherArray = ((LinObjCol)mColArray.get(LinObjCol.COLTYPE_OTHER)).getObjList();
	}

	/**
	 * Here we process the collections that can share and separate
	 * the objects that are shared.
	 */
	private void extractSharedObjects()
	{
		Iterator iter;

		// COLTYPE_PAGE
		mPageArraysSh = new CosList();
		for (int i = 0; i < mNumPages; i++) {
			CosList pageArray = (CosList)mPageArrays.get(i);
			CosList pageArraySh = new CosList();
			iter = pageArray.iterator();
			while (iter.hasNext()) {
				LinObjInfo info = (LinObjInfo)iter.next();
				if (info.isShared()) {
					iter.remove();
					pageArraySh.add(info);
				}
			}
			mPageArraysSh.add(pageArraySh);
			if (i == mSpecialPageNum) {
				mSplPageDeferredArray = new CosList();
				iter = pageArraySh.iterator();
				while (iter.hasNext()) {
					LinObjInfo info = (LinObjInfo)iter.next();
					if (info.isDeferred()) {
						iter.remove();
						mSplPageDeferredArray.add(info);
					}
				}
			}
		}

		// COLTYPE_THUMB
		mThumbArraySh = new CosList();
		iter = mThumbArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (info.isShared()) {
				iter.remove();
				mThumbArraySh.add(info);
			}
		}

		// COLTYPE_ACROFORM
		mAcroFormArraySh = new CosList();
		iter = mAcroFormArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (info.isShared()) {
				iter.remove();
				mAcroFormArraySh.add(info);
			}
		}

		// COLTYPE_STRUCTURE
		mStructureArraySh = new CosList();
		iter = mStructureArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (info.isShared()) {
				iter.remove();
				mStructureArraySh.add(info);
			}
		}

		// COLTYPE_RENDITION
		mRenditionArraySh = new CosList();
		iter = mRenditionArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (info.isShared()) {
				iter.remove();
				mRenditionArraySh.add(info);
			}
		}

		// COLTYPE_MEDIACLIP
		mMediaClipArraySh = new CosList();
		iter = mMediaClipArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (info.isShared()) {
				iter.remove();
				mMediaClipArraySh.add(info);
			}
		}
	}

	/**
	 * Here we order the objects for the main and update sections.
	 * This is the last step before we emit the objects. The hint
	 * work has to be done after that.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	private void buildLinearSections()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		Iterator iter;
		Iterator pageIter;
		Iterator embFileIter;
		int curPage;

		// The "special" part first, leave a slot for the possible encryption dictionary
		mUpdateSection = new CosList();
		mUpdateSectionCompressed = new CosList();
		addToSection(mLinArray, LinObjCol.COLTYPE_LINEARIZATION, true);
		addToSection((CosList)mPageArrays.get(mSpecialPageNum), LinObjCol.COLTYPE_PAGE, true);
		addToSection((CosList)mPageArraysSh.get(mSpecialPageNum), LinObjCol.COLTYPE_PAGE, true);
		addToSection(mSplPageDeferredArray, LinObjCol.COLTYPE_PAGE, true);

		// Then figure out the shared objects, special ones first
		CosList mainShared = new CosList();
		CosList updateShared = new CosList();
		mSharedPageObjArrays = new CosList();
		CosList sharedSplPageObjs = new CosList();
		iter = ((CosList)mPageArraysSh.get(mSpecialPageNum)).iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			updateShared.add(info.getSN(), info);
			sharedSplPageObjs.add(info.getSN(), info);
		}
		iter = mSplPageDeferredArray.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			updateShared.add(info.getSN(), info);
			sharedSplPageObjs.add(info.getSN(), info);
		}

		// Now all the other shared objects. We skip ones that are in the
		// special section. No object gets emitted twice.
		curPage = 0;
		pageIter = mPageArraysSh.iterator();
		while (pageIter.hasNext()) {
			CosList pageArraySh = (CosList)pageIter.next();
			if (curPage++ == mSpecialPageNum) {
				mSharedPageObjArrays.add(sharedSplPageObjs);
			} else {
				CosList sharedCurPageObjs = new CosList();
				iter = pageArraySh.iterator();
				while (iter.hasNext()) {
					LinObjInfo info = (LinObjInfo)iter.next();
					if (updateShared.get(info.getSN()) == null)
						mainShared.add(info.getSN(), info);
					sharedCurPageObjs.add(info.getSN(), info);
				}
				mSharedPageObjArrays.add(sharedCurPageObjs);
			}
		}
		iter = mAcroFormArraySh.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (updateShared.get(info.getSN()) == null)
				mainShared.add(info.getSN(), info);
		}
		iter = mStructureArraySh.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (updateShared.get(info.getSN()) == null)
				mainShared.add(info.getSN(), info);
		}
		iter = mRenditionArraySh.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (updateShared.get(info.getSN()) == null)
				mainShared.add(info.getSN(), info);
		}
		iter = mMediaClipArraySh.iterator();
		while (iter.hasNext()) {
			LinObjInfo info = (LinObjInfo)iter.next();
			if (updateShared.get(info.getSN()) == null)
				mainShared.add(info.getSN(), info);
		}
		mSharedObjects = mainShared;

		// Now we can finally build the main section object list
		mMainSection = new CosList();
		mMainSectionCompressed = new CosList();
		curPage = 0;
		pageIter = mPageArrays.iterator();
		while (pageIter.hasNext()) {
			CosList pageArray = (CosList)pageIter.next();
			if (curPage++ != mSpecialPageNum)
				addToSection(pageArray, LinObjCol.COLTYPE_PAGE, false);
		}
		addToSection(mSharedObjects, LinObjCol.COLTYPE_OTHER, false);
		addToSection(mThumbArray, LinObjCol.COLTYPE_THUMB, false);
		addToSection(mThumbArraySh, LinObjCol.COLTYPE_THUMB, false);
		addToSection(mOutlineArray, LinObjCol.COLTYPE_OUTLINE, false);
		addToSection(mThreadArray, LinObjCol.COLTYPE_THREAD, false);
		addToSection(mDestArray, LinObjCol.COLTYPE_DEST, false);
		addToSection(mInfoDictArray, LinObjCol.COLTYPE_INFODICT, false);
		addToSection(mPageLabelArray, LinObjCol.COLTYPE_PAGELABEL, false);
		addToSection(mAcroFormArray, LinObjCol.COLTYPE_ACROFORM, false);
		addToSection(mStructureArray, LinObjCol.COLTYPE_STRUCTURE, false);
		addToSection(mRenditionArray, LinObjCol.COLTYPE_RENDITION, false);
		addToSection(mMediaClipArray, LinObjCol.COLTYPE_MEDIACLIP, false);
		embFileIter = mEmbFileArrays.iterator();
		while (embFileIter.hasNext()) {
			addToSection((CosList)embFileIter.next(), LinObjCol.COLTYPE_EMBFILE, false);
		}
		addToSection(mOtherArray, LinObjCol.COLTYPE_OTHER, false);
	}

	/*
	 * This is the point where we decide which objects will get compressed
	 * based on xref type, and object class.
	 */
	private void addToSection(CosList src, int colType, boolean toUpdateSection)
		throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		CosList destSection = (toUpdateSection) ? mUpdateSection : mMainSection;
		CosList destSectionCompressed = (toUpdateSection) ? mUpdateSectionCompressed : mMainSectionCompressed;
		if (mXRefStyle == CosSaveParams.XREFTABLE) {
			addAllInfos(destSection, src);
		} else if (mXRefStyle == CosSaveParams.XREFSTREAM) {
			if (colType == LinObjCol.COLTYPE_LINEARIZATION) {
				addAllInfos(destSection, src);
			} else {
				CosList compSrc = new CosList();
				Iterator iter = src.iterator();
				while (iter.hasNext()) {
					LinObjInfo linInfo = (LinObjInfo)iter.next();
					CosObject cosObj = linInfo.getObjInfo().getObject();
					if (cosObj instanceof CosStream) {
						destSection.add(linInfo);
						continue;
					}
					if (cosObj instanceof CosDictionary) {
						ASName type = safeGetName((CosDictionary)cosObj, ASName.k_Type);
						if (colType == LinObjCol.COLTYPE_PAGE && type == ASName.k_Page) {
							destSection.add(linInfo);
							continue;
						}
						if ((colType == LinObjCol.COLTYPE_PAGE || colType == LinObjCol.COLTYPE_OTHER) && (CosDocument.dictionariesNotToBeCompressed.contains(type))) {
							destSection.add(linInfo);
							continue;
						}
					}
					compSrc.add(linInfo);
				}
				addAllInfos(destSectionCompressed, compSrc);
				buildObjStms(destSection, compSrc, colType);
			}
		} else if (mXRefStyle == CosSaveParams.XREFHYBRID) {
			if (colType == LinObjCol.COLTYPE_STRUCTURE) {
				addAllInfos(destSectionCompressed, src);
				buildObjStms(destSection, src, colType);
			} else {
				addAllInfos(destSection, src);
			}
		}
	}

	/*
	 * Convert LinObjInfo lists to CosObjectInfo lists for export to CosDocument
	 */
	private CosList extractObjInfos(CosList linInfoList)
	{
		CosList objInfoList = new CosList();
		if (linInfoList == mMainSection || linInfoList == mUpdateSection)
			objInfoList.add(new CosObjectInfo());
		Iterator iter = linInfoList.iterator();
		while (iter.hasNext()) {
			LinObjInfo linInfo = (LinObjInfo)iter.next();
			objInfoList.add(linInfo.getObjInfo());
		}
		return objInfoList;
	}

	/**
	 * Here we gather up any objects in the object stream object
	 * section and build them into object streams. The resultant
	 * object streams are kept in their own section.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	private void buildObjStms(CosList dest, CosList src, int colType)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (src.count() == 0)
			return;
		Iterator objIter = src.iterator();
		while (objIter.hasNext()) {
			int count = 0;
			CosObjectStream objStm = mCosDoc.createCosObjectStream();
			while (objIter.hasNext()) {
				LinObjInfo objLinInfo = (LinObjInfo)objIter.next();
				CosObject obj = objLinInfo.getObjInfo().getObject();
				objStm.addObjectToStream(obj);
				if (++count == OBJSTM_MAXNUMOBJS)
					break;
			}
			LinObjInfo objStmLinInfo = new LinObjInfo(colType, false, objStm.getInfo());
			dest.add(objStmLinInfo);
		}
	}

	/**
	 * 
	 *
	 * Here's where we build the hint stream. We are passed the stream
	 * itself and an array of positions indexed by the NEW object number.
	 * Since there's only one CosObjectInfo per CosObject anywhere in the
	 * universe, these will agree with what we have in our lists. Note
	 * the positions we are passed EXCLUDE the length of the hint stream
	 * we are building, since we don't know how long it will be. The
	 * positions in the hints are DEFINED to exclude the length of the
	 * entire hint stream object.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	void buildHintStream(CosList objectInfos, CosStream hintStm, int mainLastObjNum, int updateLastObjNum, long hintStmPos, long objEndPos)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// Set up the buffer we will build into the stream
		mObjectInfos = objectInfos;
		mHintStm = hintStm;
		mMainLastObjNum = mainLastObjNum;
		mUpdateLastObjNum = updateLastObjNum;
		mHintStmPos = hintStmPos;
		mObjEndPos = objEndPos;
		mHintOutputStream = hintStm.getStreamManager().getOutputByteStreamClearDocument(Fixed.GROWABLE, -1);

		// Assign the shared object numbers we need
		convertLinInfos();
		CosList mSharedObjsArray = new CosList();
		addAllInfos(mSharedObjsArray, (CosList)mPageArraysSh.get(mSpecialPageNum));
		addAllInfos(mSharedObjsArray, mSplPageDeferredArray);
		mNumSplPageSharedObjs = mSharedObjsArray.count();
		addAllInfos(mSharedObjsArray, mSharedObjects);
		mNumSharedObjs = mSharedObjsArray.count();
		mSharedObjLengths = new int[mNumSharedObjs];
		int shObjNum = 0;
		mSharedObjsNumMap = new CosListInt();
		Iterator iter = mSharedObjsArray.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			int objNum = info.getObjNum();
			mSharedObjLengths[shObjNum] = (int)(getNextObjPos(objNum) - getObjPos(objNum));
			mSharedObjsNumMap.add(objNum, shObjNum++);
		}

		// Now process per-page and shared items, we always do this
		int pgNum;
		mPageNumPrivateObjs = new int[mNumPages];
		mPagePrivateLen = new int[mNumPages];
		mPageNumSharedObjs = new int[mNumPages];
		for (pgNum = 0; pgNum < mNumPages; pgNum++) {
			CosList pgPrivObjsArray = (CosList)mPageArrays.get(pgNum);
			mPageNumPrivateObjs[pgNum] = pgPrivObjsArray.count();
			if (pgNum == mSpecialPageNum) {
				mPageNumPrivateObjs[pgNum] += ((CosList)mPageArraysSh.get(pgNum)).count();
				mPageNumPrivateObjs[pgNum] += mSplPageDeferredArray.count();
				int firstObjN = mSpecialPageObj.getObjNum();
				mPagePrivateLen[pgNum] = (int)(getObjPos(1) - getObjPos(firstObjN));
			}
			else if (mPageNumPrivateObjs[pgNum] != 0) {
				CosObjectInfo info = (CosObjectInfo)pgPrivObjsArray.first();
				int firstObjN = info.getObjNum();
				int lastObjN = firstObjN + mPageNumPrivateObjs[pgNum] - 1;
				mPagePrivateLen[pgNum] = (int)(getNextObjPos(lastObjN) - getObjPos(firstObjN));
			}
			CosList pgShObjsArray = (CosList)mSharedPageObjArrays.get(pgNum);
			mPageNumSharedObjs[pgNum] = pgShObjsArray.count();
		}
		generatePageHints();
		generateSharedObjHints();

		// Now do the thumbs images, if any
		mNumThumbPages = 0;
		iter = mPageThumbObjs.iterator();
		while (iter.hasNext()) {
			if (iter.next() != null)
				mNumThumbPages++;
		}
		if (mNumThumbPages != 0 && !mThumbArray.isEmpty()) {
			mThumbPageObject = new CosObject[mNumThumbPages];
			mThumbNumNoThumbs = new int[mNumThumbPages];
			mThumbNumObjects = new int[mNumThumbPages];
			mThumbTotalLen = new int[mNumThumbPages];
			int objCtr = 0;
			int thumbCtr = 0;
			iter = mPageThumbObjs.iterator();
			while (iter.hasNext()) {
				CosObject thumbObj = (CosObject)iter.next();
				if (thumbObj != null) {
					mThumbPageObject[thumbCtr] = thumbObj;
					mThumbNumNoThumbs[thumbCtr++] = objCtr;
					objCtr = 0;
				} else {
					objCtr++;
				}
			}
			// First pass, get raw indexes for page thumbs
			objCtr = 0;
			thumbCtr = 0;
			iter = mThumbArray.iterator();
			while (iter.hasNext() && thumbCtr < mNumThumbPages) {
				CosObjectInfo info = (CosObjectInfo)iter.next();
				int thisObjNum = info.getObjNum();
				if (thisObjNum == mThumbPageObject[thumbCtr].getObjNum())
					mThumbNumObjects[thumbCtr++] = objCtr;
				objCtr++;
			}
			// Now convert indexes to counts
			long thisPos = getObjPos(mThumbPageObject[0].getObjNum());
			long nextPos;
			for (thumbCtr = 0; thumbCtr < mNumThumbPages - 1; thumbCtr++) {
				mThumbNumObjects[thumbCtr] = mThumbNumObjects[thumbCtr + 1] - mThumbNumObjects[thumbCtr];
				nextPos = getObjPos(mThumbPageObject[thumbCtr + 1].getObjNum());
				mThumbTotalLen[thumbCtr] = (int)(nextPos - thisPos);
				thisPos = nextPos;
			}
			mThumbNumObjects[thumbCtr] = mThumbArray.count() - mThumbNumObjects[thumbCtr];
			CosObjectInfo info = (CosObjectInfo)mThumbArray.last();
			nextPos = getNextObjPos(info.getObjNum());
			mThumbTotalLen[thumbCtr] = (int)(nextPos - thisPos);
			generateThumbHints();
		}

		// Bookmarks
		generateSimpleGenericHints(ASName.k_O, mOutlineArray);

		// Article threads
		generateSimpleGenericHints(ASName.k_A, mThreadArray);

		// AcroForms
		generateExtendedGenericHints(ASName.k_V, mAcroFormArray, mAcroFormArraySh);

		// Named destinations
		generateSimpleGenericHints(ASName.k_E, mDestArray);

		// Page labels
		generateSimpleGenericHints(ASName.k_L, mPageLabelArray);

		// Structure
		generateExtendedGenericHints(ASName.k_C, mStructureArray, mStructureArraySh);

		// Renditions
		generateExtendedGenericHints(ASName.k_R, mRenditionArray, mRenditionArraySh);

		// Media clips
		generateExtendedGenericHints(ASName.k_M, mMediaClipArray, mMediaClipArraySh);

		// Embedded files, array based, like pages
		if (mNumEmbFiles != 0) {
			int embFileNum;
			mEmbFileFirstObjNum = new int[mNumEmbFiles];
			mEmbFileNumObjs = new int[mNumEmbFiles];
			mEmbFileLen = new int[mNumEmbFiles];
			for (embFileNum = 0; embFileNum < mNumEmbFiles; embFileNum++) {
				CosList embFileObjInfos = (CosList)mEmbFileArrays.get(embFileNum);
				CosObjectInfo info = (CosObjectInfo)embFileObjInfos.first();
				int firstObjNum = info.getObjNum();
				mEmbFileFirstObjNum[embFileNum] = firstObjNum;
				int numEmbFileObjs = embFileObjInfos.count();
				mEmbFileNumObjs[embFileNum] = numEmbFileObjs;
				info = (CosObjectInfo)embFileObjInfos.last();
				int lastObjNum = info.getObjNum();
				mEmbFileLen[embFileNum] = (int)(getNextObjPos(lastObjNum) - getObjPos(firstObjNum));
			}
			generateEmbFileHints();
		}

		// Info dict
		generateSimpleGenericHints(ASName.k_I, mInfoDictArray);

		// Now put the hint buffer into the stream
		hintStm.put(ASName.k_Filter, ASName.k_FlateDecode);
		hintStm.newDataDecoded(mHintOutputStream.closeAndConvert());
		mHintOutputStream = null;
	}

	// Generate page hints data, see Appendix F, table F.3
	private void generatePageHints()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		int count;
		int minObjsPP = getMinIntVal(mPageNumPrivateObjs, mNumPages);
		putInt(minObjsPP);
		putInt((int)getObjPos(mSpecialPageObj.getObjNum()));
		int maxObjsPP = getMaxIntVal(mPageNumPrivateObjs, mNumPages);
		int fsObjsPP = getFieldSize(maxObjsPP - minObjsPP);
		putShort(fsObjsPP);
		int minPrivLen = getMinIntVal(mPagePrivateLen, mNumPages);
		putInt(minPrivLen);
		int maxPrivLen = getMaxIntVal(mPagePrivateLen, mNumPages);
		int fsPrivLen = getFieldSize(maxPrivLen - minPrivLen);
		putShort(fsPrivLen);
		// Acrobat always sets the following three fields to zero
		putInt(0);
		putShort(0);
		putInt(0);
		// Content length is a copy of page length, but with a bogus min of zero
		putShort(fsPrivLen);
		int maxShObjsPP = getMaxIntVal(mPageNumSharedObjs, mNumPages);
		int fsShObjsPP = getFieldSize(maxShObjsPP);
		putShort(fsShObjsPP);
		int fsShObjNum = getFieldSize(mNumSharedObjs);
		putShort(fsShObjNum);
		// These things are, effectively, constants
		putShort(4);
		putShort(8);
		putFields(mPageNumPrivateObjs, mNumPages, minObjsPP, fsObjsPP);
		putFields(mPagePrivateLen, mNumPages, minPrivLen, fsPrivLen);
		putFields(mPageNumSharedObjs, mNumPages, 0, fsShObjsPP);
		int totalShObjRefs = 0;
		for (count = 0; count < mNumPages; count++)
			totalShObjRefs += mPageNumSharedObjs[count];
		int[] shObjRefs = new int[totalShObjRefs];
		int[] shObjDivs = new int[totalShObjRefs];
		count = 0;
		Iterator iter = mSharedPageObjArrays.iterator();
		while (iter.hasNext()) {
			CosList pgShObjsArray = (CosList)iter.next();
			Iterator subIter = pgShObjsArray.iterator();
			while (subIter.hasNext()) {
				CosObjectInfo info = (CosObjectInfo)subIter.next();
				shObjRefs[count] = mSharedObjsNumMap.get(info.getObjNum());
				int divValue = 0;
				shObjDivs[count++] = divValue;
			}
		}
		putFields(shObjRefs, totalShObjRefs, 0, fsShObjNum);
		putFields(shObjDivs, totalShObjRefs, 0, 4);
		putFields(mPagePrivateLen, mNumPages, minPrivLen, fsPrivLen);
	}

	// Generate shared object hints data
	// There is always one shared object per "group"
	private void generateSharedObjHints()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (mNumSharedObjs == 0)
			return;
		int pos = (int) mHintOutputStream.getPosition();
		mHintStm.put(ASName.k_S, pos);
		// Shared section can be empty if all in first page area
		int firstShObjNum = 0;
		int firstShobjPos = 0;
		if (mSharedObjects.count() != 0) {
			firstShObjNum = ((CosObjectInfo)mSharedObjects.first()).getObjNum();
			firstShobjPos = (int)getObjPos(firstShObjNum);
		}
		putInt(firstShObjNum);
		putInt(firstShobjPos);
		putInt(mNumSplPageSharedObjs);
		putInt(mNumSharedObjs);
		putShort(0);
		int minShObjLen = getMinIntVal(mSharedObjLengths, mNumSharedObjs);
		putInt(minShObjLen);
		int maxShObjLen = getMaxIntVal(mSharedObjLengths, mNumSharedObjs);
		int fsShObjLen = getFieldSize(maxShObjLen - minShObjLen);
		putShort(fsShObjLen);
		putFields(mSharedObjLengths, mNumSharedObjs, minShObjLen, fsShObjLen);
		int[] shObjSigs = new int[mNumSharedObjs];
		putFields(shObjSigs, mNumSharedObjs, 0, 1);
	}

	// Generate page thumbnail hints data
	private void generateThumbHints()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (mNumThumbPages == 0)
			return;
		int pos = (int) mHintOutputStream.getPosition();
		mHintStm.put(ASName.k_T, pos);
		int firstThumbObjNum = ((CosObjectInfo)mThumbArray.first()).getObjNum();
		putInt(firstThumbObjNum);
		putInt((int)getObjPos(firstThumbObjNum));
		putInt(mNumThumbPages);
		int maxNumPagesNoThumbs = getMaxIntVal(mThumbNumNoThumbs, mNumThumbPages);
		int fsNumPagesNoThumbs = getFieldSize(maxNumPagesNoThumbs);
		putShort(fsNumPagesNoThumbs);
		int minThumbLen = getMinIntVal(mThumbTotalLen, mNumThumbPages);
		putInt(minThumbLen);
		int maxThumbLen = getMaxIntVal(mThumbTotalLen, mNumThumbPages);
		int fsThumbLen = getFieldSize(maxThumbLen - minThumbLen);
		putShort(fsThumbLen);
		int minNumObjsInThumb = getMinIntVal(mThumbNumObjects, mNumThumbPages);
		putInt(minNumObjsInThumb);
		int maxNumObjsInThumb = getMaxIntVal(mThumbNumObjects, mNumThumbPages);
		int fsNumObjsInThumb = getFieldSize(maxNumObjsInThumb - minNumObjsInThumb);
		putShort(fsNumObjsInThumb);
		int numSharedThumbObjs = mThumbArraySh.count();
		int numFirstSharedThumbObj = 0;
		int posFirstSharedThumbObj = 0;
		int lenSharedThumbObjs = 0;
		if (numSharedThumbObjs != 0) {
			numFirstSharedThumbObj = ((CosObjectInfo)mThumbArraySh.first()).getObjNum();
			int numLastSharedThumbObj = ((CosObjectInfo)mThumbArraySh.last()).getObjNum();
			posFirstSharedThumbObj = (int)getObjPos(numFirstSharedThumbObj);
			lenSharedThumbObjs = (int)(getNextObjPos(numLastSharedThumbObj) - getObjPos(numFirstSharedThumbObj));
		}
		putInt(numFirstSharedThumbObj);
		putInt(posFirstSharedThumbObj);
		putInt(numSharedThumbObjs);
		putInt(lenSharedThumbObjs);
		putFields(mThumbNumNoThumbs, mNumThumbPages, 0, fsNumPagesNoThumbs);
		putFields(mThumbNumObjects, mNumThumbPages, minNumObjsInThumb, fsNumObjsInThumb);
		putFields(mThumbTotalLen, mNumThumbPages, minThumbLen, fsThumbLen);
	}

	// Generate embedded file hints data
	private void generateEmbFileHints()
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (mNumEmbFiles == 0)
			return;
		int pos = (int) mHintOutputStream.getPosition();
		mHintStm.put(ASName.k_B, pos);
		putInt(mEmbFileFirstObjNum[0]);
		putInt((int)getObjPos(mEmbFileFirstObjNum[0]));
		putInt(mNumEmbFiles);
		int maxNumFirstObj = getMaxIntVal(mEmbFileFirstObjNum, mNumEmbFiles);
		int fsNumFirstObj = getFieldSize(maxNumFirstObj);
		putShort(fsNumFirstObj);
		int maxNumObjs = getMaxIntVal(mEmbFileNumObjs, mNumEmbFiles);
		int fsNumObjs = getFieldSize(maxNumObjs);
		putShort(fsNumObjs);
		int maxLen = getMaxIntVal(mEmbFileLen, mNumEmbFiles);
		int fsLen = getFieldSize(maxLen);
		putShort(fsLen);
		// These things don't really have sharing
		putShort(0);
		putFields(mEmbFileFirstObjNum, mNumEmbFiles, 0, fsNumFirstObj);
		putFields(mEmbFileNumObjs, mNumEmbFiles, 0, fsNumObjs);
		putFields(mEmbFileLen, mNumEmbFiles, 0, fsLen);
	}

	// Generate simple generic hint data
	// This is for generic collections that cannot share objects
	private void generateSimpleGenericHints(ASName key, CosList colObjInfos)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (colObjInfos.isEmpty())
			return;
		int pos = (int) mHintOutputStream.getPosition();
		mHintStm.put(key, pos);
		int firstObjNum = ((CosObjectInfo)colObjInfos.first()).getObjNum();
		putInt(firstObjNum);
		long firstObjPos = getObjPos(firstObjNum);
		putInt((int)firstObjPos);
		int objCount = colObjInfos.count();
		putInt(objCount);
		int lastObjNum = ((CosObjectInfo)colObjInfos.last()).getObjNum();
		long lastObjNextPos = getNextObjPos(lastObjNum);
		putInt((int)(lastObjNextPos - firstObjPos));
	}

	// Generate extended generic hint data
	// This is for generic collections that know about sharing objects
	private void generateExtendedGenericHints(ASName key, CosList colObjInfos, CosList colObjInfosShared)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		if (colObjInfos.isEmpty())
			return;
		generateSimpleGenericHints(key, colObjInfos);
		int numShObjs = colObjInfosShared.count();
		putInt(numShObjs);
		int fsShObjs = getFieldSize(mNumSharedObjs);
		putShort(fsShObjs);
		int[] shObjRefs = new int[numShObjs];
		int count = 0;
		Iterator iter = colObjInfosShared.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			shObjRefs[count++] = mSharedObjsNumMap.get(info.getObjNum());
		}
		putFields(shObjRefs, numShObjs, 0, fsShObjs);
	}

	// Extract CosObjInfos and fix up references to compressed objects
	// Hints refer to the enclosing object stream, not the object itself
	private void convertLinInfos()
	{
		mLinArray = convertOneArray(mLinArray);
		CosList newArray = new CosList();
		Iterator iter = mPageArrays.iterator();
		while (iter.hasNext())
			newArray.add(convertOneArray((CosList)iter.next()));
		mPageArrays = newArray;
		newArray = new CosList();
		iter = mPageArraysSh.iterator();
		while (iter.hasNext())
			newArray.add(convertOneArray((CosList)iter.next()));
		mPageArraysSh = newArray;
		mSplPageDeferredArray = convertOneArray(mSplPageDeferredArray);
		newArray = new CosList();
		iter = mEmbFileArrays.iterator();
		while (iter.hasNext())
			newArray.add(convertOneArray((CosList)iter.next()));
		mEmbFileArrays = newArray;
		mThumbArray = convertOneArray(mThumbArray);
		mThumbArraySh = convertOneArray(mThumbArraySh);
		mAcroFormArray = convertOneArray(mAcroFormArray);
		mAcroFormArraySh = convertOneArray(mAcroFormArraySh);
		mStructureArray = convertOneArray(mStructureArray);
		mStructureArraySh = convertOneArray(mStructureArraySh);
		mRenditionArray = convertOneArray(mRenditionArray);
		mRenditionArraySh = convertOneArray(mRenditionArraySh);
		mMediaClipArray = convertOneArray(mMediaClipArray);
		mMediaClipArraySh = convertOneArray(mMediaClipArraySh);
		mOutlineArray = convertOneArray(mOutlineArray);
		mThreadArray = convertOneArray(mThreadArray);
		mDestArray = convertOneArray(mDestArray);
		mInfoDictArray = convertOneArray(mInfoDictArray);
		mPageLabelArray = convertOneArray(mPageLabelArray);
		mOtherArray = convertOneArray(mOtherArray);
		mSharedObjects = convertOneArray(mSharedObjects);
		newArray = new CosList();
		iter = mSharedPageObjArrays.iterator();
		while (iter.hasNext())
			newArray.add(convertOneArray((CosList)iter.next()));
		mSharedPageObjArrays = newArray;
	}

	// Fix up one array of LinObjInfos
	private CosList convertOneArray(CosList array)
	{
		if (array.isEmpty())
			return array;
		CosList newArray = new CosList();
		Iterator iter = array.iterator();
		while (iter.hasNext()) {
			LinObjInfo linInfo = (LinObjInfo)iter.next();
			CosObjectInfo objInfo = linInfo.getObjInfo();
			if (objInfo.isCompressed())
				objInfo = objInfo.getStreamInfo();
			newArray.add(objInfo.getObjNum(), objInfo);
		}
		return newArray;
	}

	// These two little accessors deal with object streams
	// and corner cases
	private final long getObjPos(int objNum)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObjectInfo info = (CosObjectInfo)mObjectInfos.get(objNum);
		if (info.isCompressed())
			info = info.getStreamInfo();
		return info.getPos() + mHintStmPos;
	}

	private final long getNextObjPos(int objNum)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObjectInfo info = (CosObjectInfo)mObjectInfos.get(objNum);
		if (info.isCompressed())
			objNum = info.getStreamInfo().getObjNum();
		if (objNum == mMainLastObjNum)
			return mObjEndPos + mHintStmPos;
		if (objNum == mUpdateLastObjNum)
			objNum = 1;
		else
			objNum++;
		return getObjPos(objNum);
	}

	// Get minimum integer value from an array
	private final int getMinIntVal(int[] values, int arrayLen) {
		if (arrayLen == 0)
			return 0;
		int minVal = 0x7FFFFFFF;
		for (int i = 0; i < arrayLen; i++) {
			if (values[i] < minVal)
				minVal = values[i];
		}
		return minVal;
	}

	// Get maximum integer value from an array
	private final int getMaxIntVal(int[] values, int arrayLen) {
		int maxVal = 0;
		for (int i = 0; i < arrayLen; i++) {
			if (values[i] > maxVal)
				maxVal = values[i];
		}
		return maxVal;
	}

	// Figure out how many bits it takes to represent a value
	private final int getFieldSize(int value) {
		int size = 32;
		for (int i = 0; i < 32; i++) {
			if ((value & 0x80000000) != 0)
				return size;
			value <<= 1;
			size--;
		}
		return size;
	}

	// Load a 32-bit value into the hint buffer
	private final void putInt(int value)
		throws PDFIOException, IOException
	{
		mHintOutputStream.write((byte)(value >> 24));
		mHintOutputStream.write((byte)(value >> 16));
		mHintOutputStream.write((byte)(value >> 8));
		mHintOutputStream.write((byte)value);
	}

	// Load a 16-bit value into the hint buffer
	private final void putShort(int value)
		throws PDFIOException, IOException
	{
		mHintOutputStream.write((byte)(value >> 8));
		mHintOutputStream.write((byte)value);
	}

	private final void putFields(int[] values, int count, int minVal, int fieldSize)
		throws PDFIOException, IOException
	{
		int bitsFree = 8;
		byte workByte = 0;
		for (int i = 0; i < count; i++) {
			int bitsToDo = fieldSize;
			int curVal = values[i] - minVal;
			while (bitsToDo > 0) {
				if (bitsFree == 8) {
					if (bitsToDo >= 8) {
						bitsToDo -= 8;
						mHintOutputStream.write((byte)(curVal >> bitsToDo));
					} else {
						workByte = (byte)(curVal << (8 - bitsToDo));
						bitsFree -= bitsToDo;
						bitsToDo = 0;
					}
				} else {
					if (bitsFree > bitsToDo) {
						workByte |= (byte)((curVal << (bitsFree - bitsToDo)) & (0xFF >> (8 - bitsFree)));
						bitsFree -= bitsToDo;
						bitsToDo = 0;
					} else {
						workByte |= (byte)((curVal >> (bitsToDo - bitsFree)) & (0xFF >> (8 - bitsFree)));
						mHintOutputStream.write(workByte);
						bitsToDo -= bitsFree;
						bitsFree = 8;
						workByte = 0;
					}
				}
			}
		}
		if (bitsFree < 8)
			mHintOutputStream.write(workByte);
	}

	// Copy all the info items from one CosList to another
	private void addAllInfos(CosList dest, CosList src)
	{
		Iterator iter = src.iterator();
		while (iter.hasNext()) {
			dest.add(iter.next());
		}
	}

	/*
	 * Map newly assigned object numbers to old ones.
	 * Used when the object cache is active.
	 */
	int mapNewToOldObjNum(int objNum) {
		if (mNewToOldObjNumMap != null)
			objNum = mNewToOldObjNumMap.get(objNum);
		return objNum;
	}

	/*
	 * Map newly assigned object numbers to old ones.
	 * Used when the object cache is active.
	 */
	int mapNewToOldObjGen(int objNum) {
		int objGen = 0;
		if (mNewToOldObjGenMap != null)
			objGen = mNewToOldObjGenMap.get(objNum);
		return objGen;
	}

	/*
	 * Map old object numbers to newly assigned ones.
	 * Used when the object cache is active.
	 */
	int mapOldToNewObjNum(int objNum) {
		if (mOldToNewObjNumMap != null)
			objNum = mOldToNewObjNumMap.get(objNum);
		return objNum;
	}

	/*
	 * Get object stream for old object number.
	 * Used when the object cache is active.
	 */
	CosObject mapOldToOldObjStm(int objNum) {
		CosObject stmObj = null;
		if (mOldToOldObjStmMap != null)
			stmObj = (CosObject)mOldToOldObjStmMap.get(objNum);
		return stmObj;
	}

	/*
	 * Object collection types. See Appendix F of the PDF spec for handling details.
	 */
	static private final class LinObjCol
	{
		// Object collection type enum
		// Special case collections
		static final int COLTYPE_LINEARIZATION = 0;	// The linearization overhead stuff
		static final int COLTYPE_PAGE = 1;		// Page related objects (CanShare)
		static final int COLTYPE_EMBFILE = 2;		// Embedded file objects (CanShare)
		static final int COLTYPE_THUMB = 3;		// Thumbnail related, completely special

		// Extended hint format collections that permit sharing
		static final int COLTYPE_ACROFORM = 4;		// Acroform objects (CanShare)
		static final int COLTYPE_STRUCTURE = 5;		// Structure objects (CanShare)
		static final int COLTYPE_RENDITION = 6;		// Renditions objects (CanShare)
		static final int COLTYPE_MEDIACLIP = 7;		// Media clip objects (CanShare)

		// Simple hint format collections
		static final int COLTYPE_OUTLINE = 8;		// Bookmark related objects
		static final int COLTYPE_THREAD = 9;		// Thread related objects
		static final int COLTYPE_DEST = 10;		// Named destination related objects
		static final int COLTYPE_INFODICT = 11;		// Doc level InfoDict objects
		static final int COLTYPE_PAGELABEL = 12;	// Page label related objects

		// The collection where everything else goes
		static final int COLTYPE_OTHER = 13;

		final boolean[] colTypeCanShare = {
			false, true, false, true, true, true, true, true,
			false, false, false, false, false, false, false
		};

		// Data fields
		private final int mColType;			// Collection type from above enum
		private CosList mColObjs;			// The LinObjInfos in this collection

		// Constructor
		private LinObjCol(int colType)
		{
			mColType = colType;
			mColObjs = new CosList();
		}

		// Accessor methods
		boolean canShare() {return colTypeCanShare[mColType];}
		boolean hasInfo(int n) {return mColObjs.get(n) != null;}
		void putInfo(int n, LinObjInfo info) {mColObjs.add(n, info);}
		int curSize() {return mColObjs.size();}

		// Return the collection objects as a CosList ordered by serial number
		CosList getObjList()
		{
			CosList retVal = new CosList();
			Iterator iter = mColObjs.iterator();
			while (iter.hasNext()) {
				LinObjInfo info = (LinObjInfo)iter.next();
				retVal.add(info.getSN(), info);
			}
			mColObjs = null;
			return retVal;
		}
	}

	/*
	 * Linearization object info tag, one per indirect object.
	 */
	final class LinObjInfo implements Comparable
	{
		// Data fields
		final private int mFirstType;		// First collection type
		final private boolean mCanShare;	// True if can be shared, else false
		final private CosObjectInfo mObjInfo;	// The corresponding COS object info
		final private int mThisSN;		// The serial number
		private boolean mIsShared = false;	// True if shared, else false
		private boolean mDefer = false;		// True if defer object to end of page
		private int mEnumTag;			// Prevent object reference cycles
		private int mSharedObjNum;		// Shared object index

		// The constructor
		private LinObjInfo(int firstType, boolean canShare, CosObjectInfo info)
			throws PDFCosParseException, PDFIOException, PDFSecurityException
		{
			mFirstType = firstType;
			mCanShare = canShare;
			mObjInfo = info;
			mThisSN = ++mLinObjInfoSN;
		}

		// Accessor methods
		int getType() {return mFirstType;}
		boolean canShare() {return mCanShare;}
		CosObjectInfo getObjInfo() {return mObjInfo;}
		boolean isShared() {return mIsShared;}
		void makeShared() {if (mCanShare) mIsShared = true;}
		boolean isDeferred() {return mDefer;}
		void makeDeferred() {if (mIsShared) mDefer = true;}
		int getEnumTag() {return mEnumTag;}
		void setEnumTag(int enumTag) {mEnumTag = enumTag;}
		int getSN() {return mThisSN;}
		void setSharedObjNum(int sharedObjNum) {mSharedObjNum = sharedObjNum;}
		int getSharedObjNum() {return mSharedObjNum;}

		/**
		 * This little thing is important. We sort objects in the linearized
		 * file based upon their order of submission to the collections. If
		 * you don't like the order, you need to change the order in which
		 * the objects are processed in PDFLinearization. CosLinearization
		 * only sets the order of the various collections and parts thereof.
		 * The order of submission controls the order WITHIN those parts.
		 */
		public int compareTo(Object obj)
		{
			return mThisSN - ((LinObjInfo)obj).getSN();
		}
	}
}
