/* ****************************************************************************
 *
 *	File: ASNumber.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParsePosition;
import java.util.Locale;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.cos.DoubleToStringFormatter;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;

/**
 * 
 * ASNumber encapulates a number while maintaining the underlying textual
 * representation for the value. That is, it does not convert the string
 * to a value until asked to do so.
 *
 * @author dpond
 */
public class ASNumber extends ASObject
{
	/**
	 * Constructor which creates a number from string value passed in proper number format.
	 */
	public ASNumber(ASString value)
		throws PDFParseException
	{
		strVal = value;
		numVal = convertString(value);
//		compareTo(value);
	}

	/**
	 * Constructor which creates a number from string value passed in proper number format.
	 */
	public ASNumber(String value)
		throws PDFParseException
	{
		strVal = new ASString(value);
		numVal = convertString(strVal);
//		compareTo(strVal);
	}

	void compareTo(ASString value)
		throws PDFParseException
	{
		Number oldVal = null;
		try {
			NumberFormat parser = NumberFormat.getInstance(Locale.US);
			oldVal = parser.parse(value.asString());
		} catch (java.text.ParseException e) {
			throw new PDFParseException(e);
		}
		boolean bad = false;
		if (oldVal == null) {
			if (numVal != null)
				bad = true;
		} else if (oldVal instanceof Long) {
			if (!(numVal instanceof Long))
				bad = true;
			else if (((Long)oldVal).compareTo((Long)numVal) != 0)
				bad = true;
		} else if (oldVal instanceof Double) {
			if (!(numVal instanceof Double))
				bad = true;
			else if (((Double)oldVal).compareTo((Double)numVal) != 0)
				bad = true;
		} else {
			bad = true;
		}
		if (bad)
			System.out.println("ASNumber value compare error");
	}

	/**
	 * Constructor.
	 */
	public ASNumber(ASNumber value)
	{
		strVal = value.strValue();
		numVal = value.numberValue();
	}

	/**
	 * Constructor.
	 */
	public ASNumber(Number value)
	{
		strVal = new ASString(String.valueOf(value.longValue()));
		numVal = value;
	}

	/**
	 * Constructor.
	 */
	public ASNumber(Float value)
	{
		strVal = new ASString(doubleToString(value.doubleValue()));
		numVal = value;
	}

	/**
	 * Constructor.
	 */
	public ASNumber(Double value)
	{
		strVal = new ASString(doubleToString(value.doubleValue()));
		numVal = value;
	}

	/**
	 * Constructor.
	 */
	public ASNumber(int value)
	{
		strVal = new ASString(String.valueOf(value));
		numVal = Integer.valueOf(value);
	}

	/**
	 * Constructor.
	 */
	public ASNumber(long value)
	{
		strVal = new ASString(String.valueOf(value));
		numVal = Long.valueOf(value);
	}

	/**
	 * Constructor.
	 */
	public ASNumber(float value)
	{
		strVal = new ASString(doubleToString(value));
		numVal = Float.valueOf(value);
	}

	/**
	 * Constructor.
	 */
	public ASNumber(double value)
	{
		strVal = new ASString(doubleToString(value));
		numVal = Double.valueOf(value);
	}

	/**
	 * Returns string representation of this number object.
	 */
	public ASString strValue()
	{
		return strVal;
	}

	
	@Override
	public String toString()
	{
		return strVal.asString();
	}

	/**
	 * Returns integer value of this number.
	 */
	public int intValue()
	{
		return numVal.intValue();
	}

	/**
	 * Returns long value of this number.
	 */
	public long longValue()
	{
		return numVal.longValue();
	}

	/**
	 * Returns float value of this number.
	 */
	public float floatValue()
	{
		return numVal.floatValue();
	}

	/**
	 * Returns double value of this number.
	 */
	public double doubleValue()
	{
		return numVal.doubleValue();
	}

	/**
	 * Returns this number.
	 */
	public Number numberValue()
	{
		return numVal;
	}
	
	/**
	 * Converts the double value passed here to an string and returns it.
	 */
	public static String doubleToString(double value)
	{
		String str = customFormatter.appendFormatted(value);
		if(str != null)
			return str;
		else
			return mDecimalFormat.format(value);
	}
	
	private static Number convertString(ASString value)
		throws PDFParseException
	{
		byte[] bytes = value.getBytes();
		long numVal = 0;
		int strBeg = 0;
		int strEnd = bytes.length;
		boolean negative = false;
		int decimalAt = -1;
		if (bytes[0] == '-') {
			negative = true;
			strBeg++;
		}
		else if (bytes[0] == '+') {
			negative = false;
			strBeg++;
		}
		while (strBeg < strEnd) {
			if (bytes[strBeg] != '0')
				break;
			strBeg++;
		}
		for (int i = strBeg; i < strEnd; i++) {
			if (bytes[i] == '.') {
				decimalAt = i;
				break;
			}
		}
		if (decimalAt >= 0) {
			while (bytes[strEnd - 1] == '0')
				strEnd--;
			
			if (strEnd - strBeg > 16) {///if it's a big real number then let's java handle it.
				try {
					NumberFormat parser = NumberFormat.getInstance(Locale.US);
					return bytes[0] == '+' ? parser.parse(new String(bytes), new ParsePosition(1)) : parser.parse(new String(bytes)); 
				} catch (java.text.ParseException e) {
					throw new PDFParseException(e);
				}
			}
		}
		for (int i = strBeg; i < strEnd; i++) {
			if (i == decimalAt)
				continue;
			byte digit = bytes[i];
			if (digit < '0' || digit > '9')
				throw new PDFParseException("Not a decimal number");
			numVal *= 10;
			numVal += digit - '0';
		}
		if (numVal == 0 && negative)
			return new Double(-0.0);
		if (negative)
			numVal = 0 - numVal;
		if (decimalAt < 0){/// if it's not real number then we should return the number from here.
			if ((numVal >= Integer.MIN_VALUE) && (numVal <= Integer.MAX_VALUE)){
				return Integer.valueOf((int)numVal);
			}
			else {
				return Long.valueOf(numVal);
			}
		}
		
		double dblVal = numVal;
		long baseVal = 1;
		for (int i = decimalAt + 1; i < strEnd; i++)
			baseVal *= 10;
		dblVal /= baseVal;
		return new Double(dblVal);
	}
	
	
	/**
	 * Writes the ASHexString in to the given OutputStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
		throws PDFIOException
	{
		try {
			outputByteStream.write( this.strValue().getBytes());
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
	
	private final static DecimalFormat mDecimalFormat;
	static {
		DecimalFormatSymbols dfs = new DecimalFormatSymbols(Locale.US);
		dfs.setZeroDigit('0');
		dfs.setDecimalSeparator('.');
		dfs.setMinusSign('-');
		mDecimalFormat = new DecimalFormat("#0.0####", dfs);
		mDecimalFormat.setGroupingUsed(false);
	}

	private final static DoubleToStringFormatter customFormatter = new DoubleToStringFormatter(5, '.', '-', true);

	private ASString strVal;
	private Number numVal;
}
