/***************************************************************************/
/*                                                                         */
/*                      ADOBE CONFIDENTIAL                                 */
/*                      _ _ _ _ _ _ _ _ _ _                                */
/*                                                                         */
/*  Copyright 2006, Adobe Systems Incorporated                             */
/*  All Rights Reserved.                                                   */
/*                                                                         */
/*  NOTICE: All information contained herein is, and remains the property  */
/*  of Adobe Systems Incorporated and its suppliers, if any.  The          */
/*  intellectual and technical concepts contained herein are proprietary   */
/*  to Adobe Systems Incorporated and its suppliers and may be coveredby   */
/*  U.S. and Foreign Patents, patents in process, and are protected by     */
/*  trade secret or copyright law.  Dissemination of this information or   */
/*  reproduction of this material is strictly forbidden unless prior       */
/*  written permission is obtained from Adobe Systems Incorporated.        */
/*                                                                         */
/***************************************************************************/

package com.adobe.versioncue.nativecomm;

import java.io.File;
import java.util.Map;

import com.adobe.versioncue.nativecomm.hook.IProcessLauncherHook;

/**
 * @author <a href="mailto:tnaroska@adobe.com">tnaroska</a>
 * @version $Revision: #1 $
 * @since Mar 8, 2006
 */
public interface IServiceBuilder
{
	/**
	 * Creates and returns an INativeService object.
	 * 
	 * @return INativeService object
	 * @throws NativeCommException on failure
	 */
	INativeService create() throws NativeCommException;

	/**
	 * Returns the unique id of the NativeService.
	 * 
	 * @return unique id of the NativeService
	 */
	String id();

	/**
	 * Sets the unique id of the NativeService.
	 * 
	 * @param id unique id to set
	 * @return the IServiceBuilder
	 */
	IServiceBuilder id(String id);

	/**
	 * Returns the maximum connection pool size
	 * 
	 * @return the maximum connection pool size
	 */
	int maxConnections();

	/**
	 * Sets the maximum connection pool size
	 * 
	 * @param maxConnections maximum connection pool size
	 * @return the IServiceBuilder
	 */
	IServiceBuilder maxConnections(int maxConnections);

	/**
	 * Returns the maximum number of requests per process.
	 * 
	 * @return the maximum number of requests per process (0 == unlimited)
	 */
	int maxRequests();

	/**
	 * Sets the maximum number of requests per process.
	 * 
	 * @param maxRequests maximum number of requests per process (0 == unlimited)
	 * @return the IServiceBuilder
	 */
	IServiceBuilder maxRequests(int maxRequests);

	/**
	 * Returns the arguments to be passed to the native process.
	 * 
	 * @return arguments to be passed to the native process.
	 */
	String[] arguments();

	/**
	 * Sets the arguments to be passed to the native process.
	 * 
	 * @param args to be passed to the native process
	 * @return the IServiceBuilder
	 */
	IServiceBuilder arguments(String[] args);

	/**
	 * Returns a rereference to this ServiceBuilders environment map. Key/value
	 * pairs added to the returned map a passed to the native process via
	 * environment variables.
	 * 
	 * @return reference to this ServiceConfigs environment map
	 */
	Map<String, String> environment();

	/**
	 * Returns the path of the NativeService library to load.
	 * 
	 * @return path of the NativeService library to load.
	 */
	String library();

	/**
	 * Sets the path of the NativeService library to load.
	 * 
	 * @param library path of the NativeService library to load
	 * @return the IServiceBuilder
	 */
	IServiceBuilder library(String library);

	/**
	 * Sets the path of the NativeService library to load.
	 * 
	 * @param library path of the NativeService library to load
	 * @return the ServiceBuilder
	 */
	IServiceBuilder library(File library);

	/**
	 * Returns the working directory of the native process.
	 * 
	 * @return the working directory of the native process; null if default is
	 *         used)
	 */
	File directory();

	/**
	 * Sets the working directory of the native process.
	 * 
	 * @param directory
	 *            working directory of the native process; if null, the parent
	 *            directory of the library is used
	 * @return the IServiceBuilder
	 */
	IServiceBuilder directory(File directory);

	/**
	 * Returns the native executable to launch.
	 * 
	 * @return the native executable to launch; if null, the default NativeComm
	 *         host is used
	 */
	String command();

	/**
	 * Sets the native executable to launch.
	 * 
	 * @param command
	 *            native executable to launch; if null, the default NativeComm
	 *            host is used
	 * @return the IServiceBuilder
	 */
	IServiceBuilder command(String command);

	/**
	 * Sets the native executable to launch.
	 * 
	 * @param command
	 *            native executable to launch; if null, the default NativeComm
	 *            host is used
	 * @return the IServiceBuilder
	 */
	IServiceBuilder command(File command);

	/**
	 * Returns the number of connections established per process.
	 * 
	 * @return number of connections established per process
	 */
	int connectionsPerProcess();

	/**
	 * Sets the number of connections established per process.
	 * 
	 * @param connectionsPerProcess number of connections established per process
	 * @return the IServiceBuilder
	 */
	IServiceBuilder connectionsPerProcess(int connectionsPerProcess);

	/**
	 * Returns the service log directory.
	 * 
	 * @return the service log directory; if null the working directory is used
	 */
	File logDir();

	/**
	 * Sets the service log directory.
	 * 
	 * @param logdir the service log directory; if null the working directory is used
	 * @return the IServiceBuilder
	 */
	IServiceBuilder logDir(File logdir);

	/**
	 * Returns the service temp directory.
	 * 
	 * @return the service temp directory; if null the default temp path is
	 *         inherited
	 */
	File tmpDir();

	/**
	 * Sets the service temp directory.
	 * 
	 * @param tmpDir the service temp directory; if null the default temp path is inherited
	 * @return the IServiceBuilder
	 */
	IServiceBuilder tmpDir(File tmpDir);

	/**
	 * Maximum time to wait for a spawned child process' connection. If a
	 * spawned child process fails to connect to the parent within the specified
	 * maximum timespan, it is forcefully killed and the launch is considered
	 * failed.
	 * 
	 * @return maximum time to wait for a spawned child process' connection.
	 */
	int launchTimeout();

	/**
	 * Sets the maximum time to wait for a spawned child process' connection.
	 * 
	 * @param timeout maximum time to wait for a spawned child process' connection.
	 * @return the IServiceBuilder
	 * 
	 * @see IServiceBuilder#launchTimeout()
	 */
	IServiceBuilder launchTimeout(int timeout);

	/**
	 * Maximum time to wait for a spawned child process' graceful termination.
	 * If a spawned child process fails to quit within the specified maximum
	 * timespan after its' connections to the parent are closed, it is
	 * forcefully killed.
	 * 
	 * @return maximum time to wait for a spawned child process' graceful
	 *         termination.
	 */
	int terminationTimeout();

	/**
	 * Sets the maximum time to wait for a spawned child process' graceful
	 * termination.
	 * 
	 * @param timeout maximum time to wait for a spawned child process' graceful termination.
	 * @return the IServiceBuilder
	 * 
	 * @see IServiceBuilder#terminationTimeout()
	 */
	IServiceBuilder terminationTimeout(int timeout);

	/**
	 * Returns TCP port to listen on for debug connection.
	 * 
	 * @return TCP port to listen on for debug connection; 0 if not debugging
	 */
	int debugPort();

	/**
	 * Sets the TCP port to listen on for debug connection.
	 * 
	 * @param port TCP port to listen on for debug connection; 0 if not debugging
	 * @return the IServiceBuilder
	 */
	IServiceBuilder debugPort(int port);

	/**
	 * Logger name (i.e. class name of NComm service user).
	 * Set this to activate log streaming from native processes to java side (unified log). 
	 * 
	 * @param loggerName - logger name (i.e. class name of NComm service user).
	 * @return the IServiceBuilder
	 */
	IServiceBuilder loggerName(String loggerName);

	/**
	 * Returns the Logger name (i.e. class name of NComm service user).
	 * @return the Logger name
	 */
	String loggerName();

	/**
	 * Returns the {@link IProcessLauncherHook}.
	 * @return the {@link IProcessLauncherHook}.
	 */
	IProcessLauncherHook processLauncherHook();

	/**
	 * Set the {@link IProcessLauncherHook} instance or null.
	 * @param hook - {@link IProcessLauncherHook} instance or null.
	 * @return the IServiceBuilder
	 */
	IServiceBuilder processLauncherHook(IProcessLauncherHook hook);
}