/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.util.List;

import com.adobe.xfa.ut.StringUtils;


/**
 * The GenericNode class exists as a general-purpose class to represent many kinds
 * of XFA nodes that don't require any special processing.  Rather than create Java classes
 * to represent each of them, we implement them with this single class.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class GenericNode extends ProtoableNode {

	static int[][] goUpdateInfoAttributeTags = {
		{	XFA.BEFORETARGETTAG,
			XFA.AFTERTARGETTAG,
			XFA.BOOKENDLEADERTAG,
			XFA.BOOKENDTRAILERTAG,
			XFA.OVERFLOWTARGETTAG,
			XFA.OVERFLOWLEADERTAG,
			XFA.OVERFLOWTRAILERTAG,
			0},
		{	XFA.TARGETTAG,
			XFA.LEADERTAG,
			XFA.TRAILERTAG,
			0},
		{	XFA.TARGETTAG,
			XFA.LEADERTAG,
			XFA.TRAILERTAG,
			0},
		{	XFA.TARGETTAG,
			XFA.LEADERTAG,
			XFA.TRAILERTAG,
			0}	
		};

	/**
	 * 	 Build a list of class types and the attributes of those classes that may contain IDs.
	 * JavaPort: In C++ these relationships are housed in a struct.
	 * In Java it's painful to create the corresponding class, so just keep the relationships in two arrays.
	 */

	static int[] goUpdateInfoElementTags = {
		XFA.BREAKTAG,
		XFA.BREAKBEFORETAG,
		XFA.BREAKAFTERTAG,
		XFA.OVERFLOWTAG
	};

	public GenericNode(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, null, null, null, -1, null);
		
	}

	public Element clone(Element pParent, boolean bDeep) {
		GenericNode oClone = (GenericNode) super.clone(pParent, bDeep);
		return oClone;
	}

	public Attribute defaultAttribute(int eTag) {
		// If this generic node is an edge, and we're an unspecified property,
		// the default value of presence
		// is inherited from its border.
		if (eTag == XFA.PRESENCETAG && getElementClass() == XFA.EDGETAG
				&& isDefault(true)) {
			Element oParent = getXFAParent();
			if (oParent != null && oParent.isSameClass(XFA.BORDERTAG))
				return oParent.defaultAttribute(eTag);
		}

		// Not special case, so let the base class handle it
		return super.defaultAttribute(eTag);
	}

	public void updateIDValues(String sPrefix, List<String> oldReferenceList) {
		for (int i = 0; i < goUpdateInfoElementTags.length; i++) {
			if (isSameClass(goUpdateInfoElementTags[i])) {
				for (int nAttrTag = 0; goUpdateInfoAttributeTags[i][nAttrTag] != 0; nAttrTag++) {
					int eTag = goUpdateInfoAttributeTags[i][nAttrTag];
					// Call XFANodeImpl::getAttribute, bypassing XFAProtoableNodeImpl::getAttribute, because
					// we don't want to inherit attributes.  Also pass TRUE meaning peek.
					// Javaport: Can't (easily) bypass ProtoableNode.getAttribute, so look up the attribute by
					// going straight to the element attribute array
					// look into the dom for the attribute
					Attribute oAttr = null;
					String aPropertyName = getAtom(eTag);
					int attr = findSchemaAttr(aPropertyName);
					if (attr != -1) {
						oAttr = getAttr(attr);
					}
					if (oAttr != null) {
						StringBuilder sAttrValue
							= new StringBuilder(oAttr.toString());
						StringBuilder sNewAttrValue = new StringBuilder();
						String sToken = StringUtils.parseToken(sAttrValue);
						while (sToken != null) {
							if (sToken.charAt(0) == '#') {
								sToken = sToken.substring(1);	// Eat '#'
								
								if (oldReferenceList != null)
									oldReferenceList.add(sToken);
								
								sToken = "#" + sPrefix + ":" + sToken;
							}
							sNewAttrValue.append(sToken);
							if (sAttrValue.length() > 0)	// Not finished tokenizing?
								sNewAttrValue.append(' ');
							sToken = StringUtils.parseToken(sAttrValue);
						}

						setAttribute(new StringAttr(aPropertyName, sNewAttrValue.toString()), eTag);
					}
				}
				break;
			}
		}
	}

	/**
	 * @see ProtoableNode#isContextSensitiveAttribute(int)
	 * @exclude from published api.
	 */
	public boolean isContextSensitiveAttribute(int eTag) {
		return eTag == XFA.PRESENCETAG || 
			super.isContextSensitiveAttribute(eTag);
	}

}
