/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2006 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.configuration;

import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import com.adobe.xfa.Attribute;
import com.adobe.xfa.AttributeInfo;
import com.adobe.xfa.ChildReln;
import com.adobe.xfa.ChildRelnInfo;
import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EnumValue;
import com.adobe.xfa.Int;
import com.adobe.xfa.Node;
import com.adobe.xfa.STRS;
import com.adobe.xfa.Schema;
import com.adobe.xfa.SchemaPairs;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormat;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;


/**
 * This class defines the valid node relationships in XFA Configuration.
 * It also provides interfaces to determine the default values for attributes.
 * 
 * @author Ian Benedict Carreon
 * @author Mike Tardif (ported to Java)
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class ConfigurationSchema extends Schema {

	private boolean			mbInitPrintSchema;

	private Map<String, String>		moPrintAttributes;

	private Map<String, String>		moPrintComments;
	
	private final boolean[]		mbLeafTags = new boolean[mnElementMax - mnElementMin + 1];

	private static class ConfigValueChildReln extends ChildReln {

		ConfigValueChildReln(ChildReln oOther, Attribute oDefaultValue) {
			super(oOther.getOccurrence(), oOther.getMax());
			moDefaultValue = oDefaultValue;
		}

		Attribute getDefaultValue() {
			return moDefaultValue;
		}

		public boolean isBaseClass() {
			return false;
		}


		private final Attribute moDefaultValue;

	}
	
	private final static String	gsXMLPrefix
								= "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";

	private final static String	gsNewLine
								= System.getProperty("line.separator");

	/**
	 * Default constructor.
	 * @exclude from published api.
	 * @exclude
	 */
	public ConfigurationSchema() {
		super(STRS.XFACONFIGURATIONNS,
				XFA.XFA_ATTRIBUTE_MIN, 
				XFA.XFA_ATTRIBUTE_MAX, 
				XFA.XFA_ELEMENT_MIN, 
				XFA.XFA_ELEMENT_MAX);
		initSchema();
	}


	public Attribute defaultAttribute(int eAttrTag, int eClassTag) {
		Attribute oAttr = null;
		ChildReln oReln = null;
		ChildRelnInfo oRelnInfo
				= super.getNodeSchema(eClassTag).getChildRelnInfo(eAttrTag);
		if (oRelnInfo != null)
		    oReln = oRelnInfo.getRelationship();
		if (oReln instanceof ConfigValueChildReln)
			oAttr = ((ConfigValueChildReln) oReln).getDefaultValue();
		if (oAttr == null) {
		    AttributeInfo oAttrInfo
				= super.getNodeSchema(eClassTag).getAttributeInfo(eAttrTag);
		    if (oAttrInfo != null)
    			oAttr = oAttrInfo.getDefault();
		}
		return oAttr;
	}

	/**
	 * Create a new attribute value.
	 * 
	 * @param eAttrTag
	 *            The name of the attribute to create. For efficiency, this name
	 *            should come from XFANamespace
	 * @param value
	 *            the value to assign to the new attribute
	 * @param eClassTag
	 *            Identify which schema element this attribute belongs to
	 * @return a new Attribute.
	 */
	public Attribute newAttribute(int eAttrTag, String value, int eClassTag) {
		Attribute oAttr = null;
		ChildReln oReln = getNodeSchema(eClassTag).getChildRelnInfo(eAttrTag).getRelationship();
		if (oReln instanceof ConfigValueChildReln)
			oAttr = ((ConfigValueChildReln) oReln).getDefaultValue();
		if (oAttr == null)
			oAttr = super.getNodeSchema(eClassTag).getAttributeInfo(eAttrTag).getDefault();
		if (oAttr == null && (eClassTag == XFA.CONFIGURATIONVALUETAG
		|| eClassTag == XFA.CONFIGURATIONKEYTAG))
			return new StringAttr(XFA.getString(eAttrTag), value);

		if (oAttr == null) {
			throw new ExFull(ResId.InvalidAttributeException, " ("
					+ XFA.getString(eAttrTag) + "=" + value + STRS.RIGHTBRACE);
		}
		return oAttr.newAttribute(value);
	}

	/*
	 * Find an element tag in the XFA namespace by name given a parent element.
	 *
	 * @param parent the parent element context.
	 * @return the tag value.
	 */
	protected int getElementTag(Element element) {
		if (element.isValidChild(XFA.CONFIGURATIONKEYTAG, 0, false, false))
			return XFA.CONFIGURATIONKEYTAG;
		else if (element.isValidChild(XFA.CONFIGURATIONVALUETAG, 0, false, false))
			return XFA.CONFIGURATIONVALUETAG;
		return XFA.INVALID_ELEMENT;
	}


	protected void initSchema() {
		super.initSchema();
		// uri schema
		addUri(XFA.URITAG);

		// config values
		addValue(XFA.ADJUSTDATATAG);
		addValue(XFA.ATTRIBUTESTAG);
		addValue(XFA.BASETAG);
		addValue(XFA.BEHAVIOROVERRIDETAG);
		addValue(XFA.BINARYTAG);
		addValue(XFA.CSPACETAG);
		addValue(XFA.ENCRYPTTAG);
		addValue(XFA.USERPASSWORDTAG);
		addValue(XFA.MASTERPASSWORDTAG);
		addValue(XFA.ENCRYPTIONLEVELTAG);
		addValue(XFA.PLAINTEXTMETADATATAG);
		addValue(XFA.PRINTTAG);
		addValue(XFA.CHANGETAG);
		addValue(XFA.CONTENTCOPYTAG);
		addValue(XFA.FORMFIELDFILLINGTAG);
		addValue(XFA.ACCESSIBLECONTENTTAG);
		addValue(XFA.DOCUMENTASSEMBLYTAG);
		addValue(XFA.TAGGEDTAG);
		addValue(XFA.TAGGEDMODETAG);
		addValue(XFA.SUBMITFORMATTAG);
		addValue(XFA.CREATORTAG);
		addValue(XFA.PRODUCERTAG);
		addValue(XFA.SCRIPTMODELTAG);
		addValue(XFA.COPIESTAG);
		addValue(XFA.DESTINATIONTAG);
		addValue(XFA.ENCODINGTAG);
		addValue(XFA.EVENTTAG);
		addValue(XFA.EXCLUDENSTAG);
		addValue(XFA.FILTERTAG);
		addValue(XFA.FONTINDEXTAG);
		addValue(XFA.FONTRESOURCETAG);
		addValue(XFA.FONTSELECTTAG);
		addValue(XFA.GROUPPARENTTAG);
		addValue(XFA.IFEMPTYTAG);
		addValue(XFA.INCREMENTALLOADTAG);
		addValue(XFA.INCREMENTALMERGETAG);
		addValue(XFA.INTERACTIVETAG);
		addValue(XFA.LEVELTAG);
		addValue(XFA.LINEARIZEDTAG);
		addValue(XFA.MODETAG);
		addValue(XFA.MODIFYANNOTSTAG);
		addValue(XFA.MSGIDTAG);
		addValue(XFA.NAMEATTRTAG);
		addValue(XFA.OPERATIONTAG);
		addValue(XFA.PICTURETAG);
		addValue(XFA.PRESENCETAG);
		addValue(XFA.PRINTHIGHQUALITYTAG);
		addValue(XFA.RANGETAG);
		addValue(XFA.RECORDTAG);
		addValue(XFA.RECURSIVETAG);
		addValue(XFA.RENAMETAG);
		addValue(XFA.SEVERITYTAG);
		addValue(XFA.STARTNODETAG);
		addValue(XFA.SUBMITURLTAG);
		addValue(XFA.SUPPRESSBANNERTAG);
		addValue(XFA.THRESHOLDTAG);
		addValue(XFA.TOTAG);
		addValue(XFA.TYPETAG);
		addValue(XFA.VERSIONTAG);
		addValue(XFA.WHITESPACETAG);
		addValue(XFA.WINDOWTAG);
		addValue(XFA.RELEVANTTAG);
		addValue(XFA.EMBEDTAG);
		addValue(XFA.SUBSETBELOWTAG);
		addValue(XFA.DEFAULTTYPEFACETAG);
		addValue(XFA.SYSTEMTAG);
		addValue(XFA.ADOBETAG);
		addValue(XFA.NETWORKTAG);
		addValue(XFA.LOCALTAG);
		addValue(XFA.EMBEDRENDEREDOUTPUTTAG);
		addValue(XFA.PACKETSTAG);
		addValue(XFA.STARTPAGETAG);
		addValue(XFA.LOCALETAG);
		addValue(XFA.LOCALESETTAG);
		addValue(XFA.DYNAMICRENDERTAG);
		addValue(XFA.RENDERPOLICYTAG);
		addValue(XFA.CURRENTPAGETAG);
		addValue(XFA.LAYOUTTAG);
		addValue(XFA.RUNSCRIPTSTAG);
		addValue(XFA.EXCLUDETAG);
		addValue(XFA.COMPRESSLOGICALSTRUCTURETAG);
		addValue(XFA.COMPRESSOBJECTSTREAMTAG);
		addValue(XFA.OUTPUTBINTAG);
		addValue(XFA.VALIDATETAG);
		addValue(XFA.ADDVIEWERPREFERENCESTAG);
		addValue(XFA.NUMBEROFCOPIESTAG);
		addValue(XFA.DUPLEXOPTIONTAG);
		addValue(XFA.PAGERANGETAG);
		addValue(XFA.PRINTSCALINGTAG);
		addValue(XFA.ENFORCETAG);
		addValue(XFA.PICKTRAYBYPDFSIZETAG);
		addValue(XFA.ADDSILENTPRINTTAG);
		addValue(XFA.PRINTERNAMETAG);
		addValue(XFA.PAGINATIONTAG);
		addValue(XFA.PAGINATIONOVERRIDETAG);
		addValue(XFA.FLIPLABELTAG);
		addValue(XFA.JOGTAG);
		addValue(XFA.STAPLETAG);
		addValue(XFA.ALWAYSEMBEDTAG);
		addValue(XFA.NEVEREMBEDTAG);
		addValue(XFA.PARTTAG);
		addValue(XFA.AMDTAG);
		addValue(XFA.CONFORMANCETAG);
		addValue(XFA.INCLUDEXDPCONTENTTAG);
		addValue(XFA.OVERPRINTTAG); // Added with CL356195 Jan, 5 2007
		addValue(XFA.ADOBEEXTENSIONLEVELTAG); // Added with CL437681 Dec 12, 2007
		addValue(XFA.VALIDATEAPPROVALSIGNATURESTAG); // Added with CL459722 Mar 31, 2008
		addValue(XFA.VALIDATIONMESSAGINGTAG);

		// equate - does not have a text child
		putAttribute(XFA.EQUATETAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATETAG, EnumValue.getEnum(XFA.VALIDATETAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATETAG, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATETAG, XFA.FROMTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATETAG, XFA.TOTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATETAG, XFA.FORCETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(XFA.EQUATETAG);


		// equateRange - does not have a text child
		putAttribute(XFA.EQUATERANGETAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATERANGETAG, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATERANGETAG, XFA.FROMTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATERANGETAG, XFA.TOTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.EQUATERANGETAG, XFA.UNICODERANGETAG, null, Schema.XFAVERSION_27, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(XFA.EQUATERANGETAG);

		// version control - does not have a text child
		// the version control section must be in 1.0
		// if not we can't protect 1.0 docs.
		putAttribute(XFA.VERSIONCONTROLTAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.VERSIONCONTROLTAG, EnumValue.getEnum(XFA.SOURCEBELOWTAG, EnumAttr.getEnum(EnumAttr.SOURCEBELOW_UPDATE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL335230 Sep 26, 2006
		putAttribute(XFA.VERSIONCONTROLTAG, EnumValue.getEnum(XFA.SOURCEABOVETAG, EnumAttr.getEnum(EnumAttr.SOURCEABOVE_WARN)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL335230 Sep 26, 2006
		putAttribute(XFA.VERSIONCONTROLTAG, EnumValue.getEnum(XFA.OUTPUTBELOWTAG, EnumAttr.getEnum(EnumAttr.OUTPUTBELOW_WARN)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL335230 Sep 26, 2006
		setLeaf(XFA.VERSIONCONTROLTAG);

		// font - does not have a text child
		putAttribute(XFA.FONTTAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.FONTTAG, XFA.TYPEFACETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.FONTTAG, EnumValue.getEnum(XFA.WEIGHTTAG, EnumAttr.getEnum(EnumAttr.WEIGHT_NORMAL)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.FONTTAG, EnumValue.getEnum(XFA.POSTURETAG, EnumAttr.getEnum(EnumAttr.POSTURE_NORMAL)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.FONTTAG, XFA.PSNAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.FONTTAG, EnumValue.getEnum(XFA.GENERICFAMILYTAG, EnumAttr.getEnum(EnumAttr.GENERICFAMILY_SANSSERIF)), Schema.XFAVERSION_27, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(XFA.FONTTAG);

		// config value
		addValue(XFA.CONFIGURATIONVALUETAG);
		
		// config key
		putValue(XFA.CONFIGURATIONKEYTAG, XFA.CONFIGURATIONVALUETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGURATIONKEYTAG, XFA.CONFIGURATIONKEYTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.CONFIGURATIONKEYTAG);

		// add font
		putValue(XFA.ADDFONTTAG, XFA.ENCODINGTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ADDFONTTAG, XFA.FONTSELECTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ADDFONTTAG, XFA.FONTRESOURCETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.ADDFONTTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.ADDFONTTAG);

		// macro cache
		putAttribute(XFA.MACROCACHETAG, XFA.FORMATSTAG, null, Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL207193 Jun 9, 2005
		addKey(XFA.MACROCACHETAG);

		// render cache
		putAttribute(XFA.RENDERCACHETAG, EnumValue.getEnum(XFA.ENABLETAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL235185 Sep 14, 2005
		addKey(XFA.RENDERCACHETAG);

		// template cache
		putAttribute(XFA.TEMPLATECACHETAG, XFA.MAXENTRIESTAG, new Int(XFA.MAXENTRIES, 5), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL235185 Sep 14, 2005
		addKey(XFA.TEMPLATECACHETAG);

		// cache
		putElement(XFA.CACHETAG, XFA.MACROCACHETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL207193 Jun 9, 2005
		putElement(XFA.CACHETAG, XFA.RENDERCACHETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL235185 Sep 14, 2005
		putElement(XFA.CACHETAG, XFA.TEMPLATECACHETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL207193 Jun 9, 2005
		addKey(XFA.CACHETAG);

		// callout
		putAttribute(XFA.CALLOUTTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putValue(XFA.CALLOUTTAG, XFA.EVENTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL207193 Jun 9, 2005
		putValue(XFA.CALLOUTTAG, XFA.BINARYTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL207193 Jun 9, 2005
		addKey(XFA.CALLOUTTAG);

		// metadata
		addKey(XFA.METADATATAG);

		// common
		putValue(XFA.COMMONTAG, XFA.LOCALETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL266579 Jan 9, 2006		
		putElement(XFA.COMMONTAG, XFA.METADATATAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL333929 Sep 20, 2006
		putValue(XFA.COMMONTAG, XFA.LOCALESETTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);		
		putValue(XFA.COMMONTAG, XFA.SUPPRESSBANNERTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.SUPPRESSBANNERTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.DATATAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.TEMPLATETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.TEMPTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.LOGTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.CALLOUTTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.COMMONTAG, XFA.MESSAGINGTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.COMMONTAG, XFA.VERSIONCONTROLTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL335230 Sep 26, 2006
		putValue(XFA.COMMONTAG, XFA.VALIDATIONMESSAGINGTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.VALIDATIONMESSAGINGTAG, EnumAttr.getEnum(EnumAttr.VALIDATIONMESSAGING_ALLMESSAGESINDIVIDUALLY)), Schema.XFAVERSION_30, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.COMMONTAG);

		// compression
		putValue(XFA.COMPRESSIONTAG, XFA.TYPETAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.TYPE, STRS.NONE), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.COMPRESSIONTAG, XFA.LEVELTAG, ChildReln.getZeroOrOne(), new Int(XFA.COMPRESSION, 6), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.COMPRESSIONTAG, XFA.COMPRESSLOGICALSTRUCTURETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.COMPRESSLOGICALSTRUCTURETAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.COMPRESSIONTAG, XFA.COMPRESSOBJECTSTREAMTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.COMPRESSOBJECTSTREAMTAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_28,	Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);   //Added Nov 5, 2007 CL429454
		addKey(XFA.COMPRESSIONTAG);

		// encryption
		putValue(XFA.ENCRYPTIONTAG, XFA.ENCRYPTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ENCRYPTTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ENCRYPTIONTAG, XFA.USERPASSWORDTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ENCRYPTIONTAG, XFA.MASTERPASSWORDTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ENCRYPTIONTAG, XFA.ENCRYPTIONLEVELTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ENCRYPTIONLEVELTAG, EnumAttr.getEnum(EnumAttr.ENCRYPTLEVEL_40)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.ENCRYPTIONTAG, XFA.PERMISSIONSTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.ENCRYPTIONTAG);

		// viewer preferences
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.ADDVIEWERPREFERENCESTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ADDVIEWERPREFERENCESTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL289690 Apr 10, 2006
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.NUMBEROFCOPIESTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL289690 Apr 10, 2006
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.DUPLEXOPTIONTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.DUPLEXOPTIONTAG, EnumAttr.getEnum(EnumAttr.DUPLEXOPTION_SIMPLEX)), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL289690 Apr 10, 2006
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.PAGERANGETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL289690 Apr 10, 2006
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.PRINTSCALINGTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PRINTSCALINGTAG, EnumAttr.getEnum(EnumAttr.PRINTSCALING_APPDEFAULT)), Schema.XFAVERSION_30, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL493017 Sept 17, 2008
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.ENFORCETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_30, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL493017 Sept 17, 2008
		putValue(XFA.VIEWERPREFERENCESTAG, XFA.PICKTRAYBYPDFSIZETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PICKTRAYBYPDFSIZETAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_30, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL493017 Sept 17, 2008
		addKey(XFA.VIEWERPREFERENCESTAG);

		// silent Print
		putValue(XFA.SILENTPRINTTAG, XFA.ADDSILENTPRINTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ADDSILENTPRINTTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL290451 Apr 12, 2006
		putValue(XFA.SILENTPRINTTAG, XFA.PRINTERNAMETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL290451 Apr 12, 2006
		addKey(XFA.SILENTPRINTTAG);

		// defaultTypeface
		putAttribute(XFA.DEFAULTTYPEFACETAG, EnumValue.getEnum(XFA.WRITINGSCRIPTTAG, EnumAttr.getEnum(EnumAttr.WRITINGSCRIPT_STAR)), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL204496 May 31, 2005

		// font information
		putValue(XFA.FONTINFOTAG, XFA.EMBEDTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.EMBEDTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.FONTINFOTAG, XFA.SUBSETBELOWTAG, ChildReln.getZeroOrOne(), new Int(XFA.SUBSETBELOW,100), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.FONTINFOTAG, XFA.DEFAULTTYPEFACETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.FONTINFOTAG, XFA.ALWAYSEMBEDTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL332424 Sep 14, 2006
		putValue(XFA.FONTINFOTAG, XFA.NEVEREMBEDTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL332424 Sep 14, 2006
		putElement(XFA.FONTINFOTAG, XFA.MAPTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.FONTINFOTAG);

		// mediumInfo
		putElement(XFA.MEDIUMINFOTAG, XFA.MAPTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.MEDIUMINFOTAG);

		// PS Map
		putValue(XFA.PSMAPTAG, XFA.FONTTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PSMAPTAG);

		// PDFA
		putValue(XFA.PDFATAG, XFA.PARTTAG, ChildReln.getZeroOrOne(), new Int(XFA.PART, 1), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL343791 Oct 30, 2006
		putValue(XFA.PDFATAG, XFA.AMDTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL343791 Oct 30, 2006vv
		putValue(XFA.PDFATAG, XFA.CONFORMANCETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.CONFORMANCETAG, EnumAttr.getEnum(EnumAttr.PDFACONFORMANCE_A)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL343791 Oct 30, 2006
		putValue(XFA.PDFATAG, XFA.INCLUDEXDPCONTENTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL ...
		addKey(XFA.PDFATAG);

		// page offset
		putAttribute(XFA.PAGEOFFSETTAG, XFA.XTAG, new StringAttr(XFA.X, STRS.USEXDCSETTING), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		putAttribute(XFA.PAGEOFFSETTAG, XFA.YTAG, new StringAttr(XFA.Y, STRS.USEXDCSETTING), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		addKey(XFA.PAGEOFFSETTAG);

		// compress
		putAttribute(XFA.COMPRESSTAG, EnumValue.getEnum(XFA.SCOPETAG, EnumAttr.getEnum(EnumAttr.COMPRESSSCOPE_IMAGEONLY)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL340535 Oct 17, 2006
		addKey(XFA.COMPRESSTAG);

		// XDP
		putValue(XFA.XDPTAG, XFA.EMBEDRENDEREDOUTPUTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.EMBEDRENDEREDOUTPUTTAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.XDPTAG, XFA.PACKETSTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XDPTAG);

		// permissions
		putValue(XFA.PERMISSIONSTAG, XFA.MODIFYANNOTSTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.PRINTHIGHQUALITYTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.PRINTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PRINTTAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.CHANGETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.CHANGETAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.CONTENTCOPYTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.CONTENTCOPYTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.FORMFIELDFILLINGTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.FORMFIELDFILLINGTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.ACCESSIBLECONTENTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ACCESSIBLECONTENTTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.DOCUMENTASSEMBLYTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.DOCUMENTASSEMBLYTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PERMISSIONSTAG, XFA.PLAINTEXTMETADATATAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PLAINTEXTMETADATATAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PERMISSIONSTAG);

		// data
		putValue(XFA.DATATAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.RECORDTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.RANGETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.STARTNODETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.INCREMENTALLOADTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.INCREMENTALLOADTAG, EnumAttr.getEnum(EnumAttr.INCREMENTALLOAD_NONE)), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL324530 Aug 15, 2006
		putValue(XFA.DATATAG, XFA.ATTRIBUTESTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ATTRIBUTESTAG, EnumAttr.getEnum(EnumAttr.ATTRIBUTE_PRESERVE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.WINDOWTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.ADJUSTDATATAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.ADJUSTDATATAG, EnumAttr.getEnum(EnumAttr.BOOL_TRUE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DATATAG, XFA.EXCLUDENSTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.DATATAG, XFA.XSLTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.DATATAG, XFA.TRANSFORMTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.DATATAG, XFA.OUTPUTXSLTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.DATATAG);

		// debug
		putValue(XFA.DEBUGTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.DEBUGTAG);

		// map
		putValue(XFA.MAPTAG, XFA.EQUATETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.MAPTAG, XFA.EQUATERANGETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_27, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.MAPTAG);

		// image
		putAttribute(XFA.IMAGETAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putValue(XFA.IMAGETAG, XFA.CSPACETAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.CSPACE, STRS.SRGB), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.IMAGETAG, XFA.VERSIONTAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.VERSION, "1.3"), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.IMAGETAG, XFA.COMPRESSIONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.IMAGETAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.IMAGETAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.IMAGETAG);

		// list Font
		putElement(XFA.LISTFONTTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.LISTFONTTAG, XFA.XFTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.LISTFONTTAG, XFA.OUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.LISTFONTTAG);

		// log
		putValue(XFA.LOGTAG, XFA.THRESHOLDTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.THRESHOLDTAG, EnumAttr.getEnum(EnumAttr.MESSAGE_TRACE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.LOGTAG, XFA.TOTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.TOTAG, EnumAttr.getEnum(EnumAttr.LOG_NULL)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.LOGTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.LOGTAG, XFA.MODETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.MODETAG, EnumAttr.getEnum(EnumAttr.MODE_APPEND)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.LOGTAG);

		// message
		putValue(XFA.MESSAGETAG, XFA.MSGIDTAG, ChildReln.getZeroOrOne(), new Int(XFA.MSGID, 0), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.MESSAGETAG, XFA.SEVERITYTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.SEVERITYTAG, EnumAttr.getEnum(EnumAttr.SEV_IGNORE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.MESSAGETAG);

		// messaging
		putElement(XFA.MESSAGINGTAG, XFA.MESSAGETAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.MESSAGINGTAG);

		// open action
		putValue(XFA.OPENACTIONTAG, XFA.DESTINATIONTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.DESTINATIONTAG, EnumAttr.getEnum(EnumAttr.DESTINATION_NONE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.OPENACTIONTAG);

		// batch output
		putAttribute(XFA.BATCHOUTPUTTAG, EnumValue.getEnum(XFA.FORMATTAG, EnumAttr.getEnum(EnumAttr.BATCHFORMAT_NONE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		addKey(XFA.BATCHOUTPUTTAG);

		// output
		putValue(XFA.OUTPUTTAG, XFA.TOTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.TOTAG, EnumAttr.getEnum(EnumAttr.OUTPUT_NULL)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.OUTPUTTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.OUTPUTTAG, XFA.TYPETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.TYPETAG, EnumAttr.getEnum(EnumAttr.OUTPUT_NATIVE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.OUTPUTTAG);

		// present
		putValue(XFA.PRESENTTAG, XFA.INCREMENTALMERGETAG, ChildReln.getZeroOrOne(), new Int(XFA.INCREMENTALMERGE, 1), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL283886 Mar 20, 2006
		putValue(XFA.PRESENTTAG, XFA.DESTINATIONTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PRESENTTAG, XFA.COPIESTAG, ChildReln.getZeroOrOne(), new Int(XFA.COPIES, 1), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PRESENTTAG, XFA.LAYOUTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.LAYOUTTAG, EnumAttr.getEnum(EnumAttr.XDCLAYOUT_PAGINATE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PRESENTTAG, XFA.SUBMITURLTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Changed to 0..N with CL 435066
		putValue(XFA.PRESENTTAG, XFA.VALIDATETAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.VALIDATE, "preSubmit"), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL286317 Mar 29, 2006
		putValue(XFA.PRESENTTAG, XFA.PAGINATIONTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PAGINATIONTAG, EnumAttr.getEnum(EnumAttr.PAGINATION_SIMPLEX)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL296209 May 1, 2006
		putValue(XFA.PRESENTTAG, XFA.PAGINATIONOVERRIDETAG,	ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PAGINATIONOVERRIDETAG, EnumAttr.getEnum(EnumAttr.PAGINATIONOVERRIDE_NONE)), Schema.XFAVERSION_27, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PRESENTTAG, XFA.OVERPRINTTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.OVERPRINTTAG, EnumAttr.getEnum(EnumAttr.OVERPRINT_NONE)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL356195 Jan, 5 2007
		putValue(XFA.PRESENTTAG, XFA.BEHAVIOROVERRIDETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_30, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.COMMONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.OUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.CACHETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.SCRIPTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.PDFTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.PCLTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.PSTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.IMAGETAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.ZPLTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.XDPTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.WEBCLIENTTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PRESENTTAG, XFA.DRIVERTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PRESENTTAG);

		// path
		putValue(XFA.PATHTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PATHTAG);

		// staple
		putAttribute(XFA.STAPLETAG, XFA.MODETAG, EnumValue.getEnum(XFA.MODETAG, EnumAttr.getEnum(EnumAttr.STAPLEMODE_USEPRINTERSETTING)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL334070 Sep 21, 2006

		// PCL
		putAttribute(XFA.PCLTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.PCLTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PCLTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PCLTAG, XFA.MACROTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_22, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL196752 May 1, 2005
		putElement(XFA.PCLTAG, XFA.PAGEOFFSETTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		putValue(XFA.PCLTAG, XFA.JOGTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.JOGTAG, EnumAttr.getEnum(EnumAttr.JOG_USEPRINTERSETTING)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		putValue(XFA.PCLTAG, XFA.STAPLETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL334070 Sep 21, 2006
		putValue(XFA.PCLTAG, XFA.OUTPUTBINTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL285072 Mar 24, 2006
		putElement(XFA.PCLTAG, XFA.BATCHOUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL332628 Sep 15, 2006
		putElement(XFA.PCLTAG, XFA.MEDIUMINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PCLTAG);

		// ZPL
		putAttribute(XFA.ZPLTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.ZPLTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.ZPLTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.ZPLTAG, XFA.MACROTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_22, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL196752 May 1, 2005
		putValue(XFA.ZPLTAG, XFA.FLIPLABELTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.FLIPLABELTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		putElement(XFA.ZPLTAG, XFA.BATCHOUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL332628 Sep 15, 2006
		addKey(XFA.ZPLTAG);

		// PS
		putAttribute(XFA.PSTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.PSTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PSTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PSTAG, XFA.MACROTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_22, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL196752 May 1, 2005
		putElement(XFA.PSTAG, XFA.COMPRESSTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL340535 Oct 17, 2006
		putValue(XFA.PSTAG, XFA.JOGTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.JOGTAG, EnumAttr.getEnum(EnumAttr.JOG_USEPRINTERSETTING)), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL313372 Jun 28, 2006
		putValue(XFA.PSTAG, XFA.STAPLETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL334070 Sep 21, 2006
		putValue(XFA.PSTAG, XFA.OUTPUTBINTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL285735 Mar 27, 2006
		putElement(XFA.PSTAG, XFA.BATCHOUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL332628 Sep 15, 2006
		putElement(XFA.PSTAG, XFA.MEDIUMINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.PSTAG);

		// web client
		putAttribute(XFA.WEBCLIENTTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.WEBCLIENTTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.WEBCLIENTTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.WEBCLIENTTAG);

		// generic driver
		putAttribute(XFA.DRIVERTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.DRIVERTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.DRIVERTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.DRIVERTAG, XFA.CONFIGURATIONVALUETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.DRIVERTAG, XFA.CONFIGURATIONKEYTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.DRIVERTAG, XFA.LEVELTAG, null, Schema.XFAVERSION_22, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL196752 May 1, 2005
		addKey(XFA.DRIVERTAG);

		// macro
		putAttribute(XFA.DRIVERTAG, XFA.LEVELTAG, null, Schema.XFAVERSION_22, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL196752 May 1, 2005);
		addKey(XFA.MACROTAG);

		// pdf driver
		putAttribute(XFA.PDFTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putValue(XFA.PDFTAG, XFA.SCRIPTMODELTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.SCRIPTMODELTAG, EnumAttr.getEnum(EnumAttr.SCRIPTMODEL_XFA)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.SUBMITFORMATTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.SUBMITFORMATTAG, EnumAttr.getEnum(EnumAttr.SUBMITFORMAT_HTML)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.TAGGEDTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.TAGGEDTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.TAGGEDMODETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.TAGGEDMODETAG, EnumAttr.getEnum(EnumAttr.TAGGEDMODE_NORMAL)), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL214943 Jul 8, 2005
		putValue(XFA.PDFTAG, XFA.VERSIONTAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.VERSION, "1.7"), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.ADOBEEXTENSIONLEVELTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_28, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL437681 Dec 12, 2007
		putValue(XFA.PDFTAG, XFA.INTERACTIVETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.INTERACTIVETAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.CREATORTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.PRODUCERTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.LINEARIZEDTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.LINEARIZEDTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.PDFTAG, XFA.RENDERPOLICYTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.RENDERPOLICYTAG, EnumAttr.getEnum(EnumAttr.RENDERPOLICY_SERVER)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.COMPRESSIONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.OPENACTIONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.ENCRYPTIONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.FONTINFOTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.PDFTAG, XFA.BATCHOUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_24, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL221355 Aug 3, 2005
		putElement(XFA.PDFTAG, XFA.VIEWERPREFERENCESTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL289690 Apr 10, 2006
		putElement(XFA.PDFTAG, XFA.SILENTPRINTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_25, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL290451 Apr 12, 2006
		putElement(XFA.PDFTAG, XFA.PDFATAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_26, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL343791 Oct 30, 2006
		addKey(XFA.PDFTAG);

		// present.script
		putValue(XFA.SCRIPTTAG, XFA.CURRENTPAGETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.SCRIPTTAG, XFA.EXCLUDETAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.EXCLUDE, "nullTest formatTest"), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.SCRIPTTAG, XFA.RUNSCRIPTSTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.RUNSCRIPTSTAG, EnumAttr.getEnum(EnumAttr.RUNSCRIPTS_BOTH)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.SCRIPTTAG);

		// remove font
		putValue(XFA.REMOVEFONTTAG, XFA.FONTINDEXTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.REMOVEFONTTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.REMOVEFONTTAG);

		// resolve font
		putElement(XFA.RESOLVEFONTTAG, XFA.SEARCHFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.RESOLVEFONTTAG, XFA.OUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.RESOLVEFONTTAG, XFA.FONTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.RESOLVEFONTTAG);

		// search font
		putValue(XFA.SEARCHFONTTAG, XFA.FILTERTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.SEARCHFONTTAG, XFA.RECURSIVETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.SEARCHFONTTAG, XFA.PATHTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.SEARCHFONTTAG, XFA.OUTPUTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.SEARCHFONTTAG);

		// temp
		putValue(XFA.TEMPTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.TEMPTAG);

		// template
		putValue(XFA.TEMPLATETAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TEMPLATETAG, XFA.STARTPAGETAG, ChildReln.getZeroOrOne(), new Int(XFA.STARTPAGE, 0), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TEMPLATETAG, XFA.BASETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TEMPLATETAG, XFA.RELEVANTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.TEMPLATETAG, XFA.XSLTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.TEMPLATETAG);

		// transform
		putAttribute(XFA.TRANSFORMTAG, XFA.REFTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.IFEMPTYTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.IFEMPTYTAG, EnumAttr.getEnum(EnumAttr.IFEMPTY_DATAVALUE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.PRESENCETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.PRESENCETAG, EnumAttr.getEnum(EnumAttr.NODEPRESENCE_PRESERVE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.RENAMETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.WHITESPACETAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.WHITESPACETAG, EnumAttr.getEnum(EnumAttr.WHITESPACE_PRESERVE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.PICTURETAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.NAMEATTRTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.TRANSFORMTAG, XFA.GROUPPARENTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.TRANSFORMTAG);

		// update XDC
		putValue(XFA.UPDATEXDCTAG, XFA.ENCODINGTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.UPDATEXDCTAG, XFA.FONTSELECTTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.UPDATEXDCTAG, XFA.XDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.UPDATEXDCTAG, XFA.XCITAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.UPDATEXDCTAG, XFA.SEARCHFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.UPDATEXDCTAG);

		// XCI
		putValue(XFA.XCITAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XCITAG);

		// XDC
		putValue(XFA.XDCTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCTAG, XFA.XSLTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XDCTAG);

		// XDC font agent
		putValue(XFA.XDCFONTAGENTTAG, XFA.OPERATIONTAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.COMMONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.SEARCHFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.UPDATEXDCTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.ADDFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.REMOVEFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.LISTFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XDCFONTAGENTTAG, XFA.RESOLVEFONTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XDCFONTAGENTTAG);

		// generic agent
		putAttribute(XFA.AGENTTAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.AGENTTAG, XFA.COMMONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.AGENTTAG, XFA.CONFIGURATIONVALUETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.AGENTTAG, XFA.CONFIGURATIONKEYTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.AGENTTAG);

		// acrobat
		putElement(XFA.ACROBATTAG, XFA.ACROBAT7TAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ACROBATTAG, XFA.SUBMITURLTAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_28, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Changed to 0..N with CL 435066 
		putElement(XFA.ACROBATTAG, XFA.COMMONTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ACROBATTAG, XFA.CONFIGURATIONVALUETAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.ACROBATTAG, XFA.CONFIGURATIONKEYTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ACROBATTAG, XFA.VALIDATETAG, ChildReln.getZeroOrOne(),	new StringAttr(XFA.VALIDATE, "preSubmit"), Schema.XFAVERSION_28, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putValue(XFA.ACROBATTAG, XFA.VALIDATEAPPROVALSIGNATURESTAG, ChildReln.getZeroOrOne(), new StringAttr(XFA.VALIDATEAPPROVALSIGNATURES, "docReady postSign"), Schema.XFAVERSION_28, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); // Added with CL459722 Mar 31, 2008
		addKey(XFA.ACROBATTAG);

		// acrobat.acrobat7
		putValue(XFA.ACROBAT7TAG, XFA.DYNAMICRENDERTAG, ChildReln.getZeroOrOne(), EnumValue.getEnum(XFA.DYNAMICRENDERTAG, EnumAttr.getEnum(EnumAttr.DYNAMICRENDER_FORBIDDEN)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.ACROBAT7TAG);

		// XFT
		putValue(XFA.XFTTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XFTTAG);

		// XSL
		putValue(XFA.XSLTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.XSLTAG, XFA.DEBUGTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.XSLTAG);

		// output XSL
		putValue(XFA.OUTPUTXSLTAG, XFA.URITAG, ChildReln.getZeroOrOne(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.OUTPUTXSLTAG);

		// text node
		//Don't add TextNode since it is not an Element
		//addSample(XFA.TEXTNODETAG, "com.adobe.xfa.TextNode", true);

		// area - does not have a text child
		putAttribute(XFA.AREATAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.AREATAG, XFA.NAMETAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.AREATAG, XFA.LEVELTAG, new Int(XFA.LEVEL, 0), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(XFA.AREATAG);

		// trace node
		putValue(XFA.TRACETAG, XFA.AREATAG, ChildReln.getZeroOrMore(), null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		addKey(XFA.TRACETAG);

		// config model
		putAttribute(XFA.CONFIGTAG, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(XFA.CONFIGTAG, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0); 
		putElement(XFA.CONFIGTAG, XFA.PRESENTTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGTAG, XFA.TRACETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGTAG, XFA.AGENTTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGTAG, XFA.CONFIGURATIONKEYTAG, ChildReln.getZeroOrMore(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGTAG, XFA.PSMAPTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(XFA.CONFIGTAG, XFA.ACROBATTAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
	}
	
	protected Element newElement(int eTag, Element parent, Node prevSibling) {
		
		Element newNode;
		switch (eTag) {
		
		// TextNode is not an Element!
		//case XFA.TEXTNODETAG: 
		
		case XFA.CONFIGTAG: 
			newNode = new com.adobe.xfa.configuration.ConfigurationModel(parent, prevSibling);
			break;
		
		case XFA.CONFIGURATIONKEYTAG:
		case XFA.ADDFONTTAG:
		case XFA.MACROCACHETAG:
		case XFA.RENDERCACHETAG:
		case XFA.TEMPLATECACHETAG:
		case XFA.CACHETAG:
		case XFA.CALLOUTTAG:
		case XFA.METADATATAG:
		case XFA.COMMONTAG:
		case XFA.COMPRESSIONTAG:
		case XFA.ENCRYPTIONTAG:
		case XFA.VIEWERPREFERENCESTAG:
		case XFA.SILENTPRINTTAG:
		case XFA.FONTINFOTAG:
		case XFA.MEDIUMINFOTAG:
		case XFA.PSMAPTAG:
		case XFA.PDFATAG:
		case XFA.PAGEOFFSETTAG:
		case XFA.COMPRESSTAG:
		case XFA.XDPTAG:
		case XFA.PERMISSIONSTAG:
		case XFA.DATATAG:
		case XFA.DEBUGTAG:
		case XFA.MAPTAG:
		case XFA.IMAGETAG:
		case XFA.LISTFONTTAG:
		case XFA.LOGTAG:
		case XFA.MESSAGETAG:
		case XFA.MESSAGINGTAG:
		case XFA.OPENACTIONTAG:
		case XFA.BATCHOUTPUTTAG:
		case XFA.OUTPUTTAG:
		case XFA.PRESENTTAG:
		case XFA.PATHTAG:
		case XFA.PCLTAG:
		case XFA.ZPLTAG:
		case XFA.PSTAG:
		case XFA.WEBCLIENTTAG:
		case XFA.DRIVERTAG:
		case XFA.MACROTAG:
		case XFA.PDFTAG:
		case XFA.SCRIPTTAG:
		case XFA.REMOVEFONTTAG:
		case XFA.RESOLVEFONTTAG:
		case XFA.SEARCHFONTTAG:
		case XFA.TEMPTAG:
		case XFA.TEMPLATETAG:
		case XFA.TRANSFORMTAG:
		case XFA.UPDATEXDCTAG:
		case XFA.XCITAG:
		case XFA.XDCTAG:
		case XFA.XDCFONTAGENTTAG:
		case XFA.AGENTTAG:
		case XFA.ACROBATTAG:
		case XFA.ACROBAT7TAG:
		case XFA.XFTTAG:
		case XFA.XSLTAG:
		case XFA.OUTPUTXSLTAG:
		case XFA.TRACETAG:
			newNode = new com.adobe.xfa.configuration.ConfigurationKey(parent, prevSibling);
			break;
		
		case XFA.EQUATETAG:
		case XFA.EQUATERANGETAG:
		case XFA.VERSIONCONTROLTAG:
		case XFA.FONTTAG:
		case XFA.AREATAG:		
		case XFA.ADJUSTDATATAG:
		case XFA.ATTRIBUTESTAG:
		case XFA.BASETAG:
		case XFA.BEHAVIOROVERRIDETAG:
		case XFA.BINARYTAG:
		case XFA.CSPACETAG:
		case XFA.ENCRYPTTAG:
		case XFA.USERPASSWORDTAG:
		case XFA.MASTERPASSWORDTAG:
		case XFA.ENCRYPTIONLEVELTAG:
		case XFA.PLAINTEXTMETADATATAG:
		case XFA.PRINTTAG:
		case XFA.CHANGETAG:
		case XFA.CONTENTCOPYTAG:
		case XFA.FORMFIELDFILLINGTAG:
		case XFA.ACCESSIBLECONTENTTAG:
		case XFA.DOCUMENTASSEMBLYTAG:
		case XFA.TAGGEDTAG:
		case XFA.TAGGEDMODETAG:
		case XFA.SUBMITFORMATTAG:
		case XFA.CREATORTAG:
		case XFA.PRODUCERTAG:
		case XFA.SCRIPTMODELTAG:
		case XFA.COPIESTAG:
		case XFA.DESTINATIONTAG:
		case XFA.ENCODINGTAG:
		case XFA.EVENTTAG:
		case XFA.EXCLUDENSTAG:
		case XFA.FILTERTAG:
		case XFA.FONTINDEXTAG:
		case XFA.FONTRESOURCETAG:
		case XFA.FONTSELECTTAG:
		case XFA.GROUPPARENTTAG:
		case XFA.IFEMPTYTAG:
		case XFA.INCREMENTALLOADTAG:
		case XFA.INCREMENTALMERGETAG:
		case XFA.INTERACTIVETAG:
		case XFA.LEVELTAG:
		case XFA.LINEARIZEDTAG:
		case XFA.MODETAG:
		case XFA.MODIFYANNOTSTAG:
		case XFA.MSGIDTAG:
		case XFA.NAMEATTRTAG:
		case XFA.OPERATIONTAG:
		case XFA.PICTURETAG:
		case XFA.PRESENCETAG:
		case XFA.PRINTHIGHQUALITYTAG:
		case XFA.RANGETAG:
		case XFA.RECORDTAG:
		case XFA.RECURSIVETAG:
		case XFA.RENAMETAG:
		case XFA.SEVERITYTAG:
		case XFA.STARTNODETAG:
		case XFA.SUBMITURLTAG:
		case XFA.SUPPRESSBANNERTAG:
		case XFA.THRESHOLDTAG:
		case XFA.TOTAG:
		case XFA.TYPETAG:
		case XFA.VERSIONTAG:
		case XFA.WHITESPACETAG:
		case XFA.WINDOWTAG:
		case XFA.RELEVANTTAG:
		case XFA.EMBEDTAG:
		case XFA.SUBSETBELOWTAG:
		case XFA.DEFAULTTYPEFACETAG:
		case XFA.SYSTEMTAG:
		case XFA.ADOBETAG:
		case XFA.ADOBEEXTENSIONLEVELTAG:
		case XFA.NETWORKTAG:
		case XFA.LOCALTAG:
		case XFA.EMBEDRENDEREDOUTPUTTAG:
		case XFA.PACKETSTAG:
		case XFA.STARTPAGETAG:
		case XFA.LOCALETAG:
		case XFA.LOCALESETTAG:
		case XFA.DYNAMICRENDERTAG:
		case XFA.RENDERPOLICYTAG:
		case XFA.CURRENTPAGETAG:
		case XFA.LAYOUTTAG:
		case XFA.RUNSCRIPTSTAG:
		case XFA.EXCLUDETAG:
		case XFA.COMPRESSLOGICALSTRUCTURETAG:
		case XFA.COMPRESSOBJECTSTREAMTAG:
		case XFA.OUTPUTBINTAG:
		case XFA.VALIDATETAG:
		case XFA.VALIDATEAPPROVALSIGNATURESTAG:
		case XFA.VALIDATIONMESSAGINGTAG:
		case XFA.ADDVIEWERPREFERENCESTAG:
		case XFA.NUMBEROFCOPIESTAG:
		case XFA.DUPLEXOPTIONTAG:
		case XFA.PAGERANGETAG:
		case XFA.PRINTSCALINGTAG:
		case XFA.ENFORCETAG:
		case XFA.PICKTRAYBYPDFSIZETAG:
		case XFA.ADDSILENTPRINTTAG:
		case XFA.PRINTERNAMETAG:
		case XFA.PAGINATIONTAG:
		case XFA.PAGINATIONOVERRIDETAG:
		case XFA.FLIPLABELTAG:
		case XFA.JOGTAG:
		case XFA.STAPLETAG:
		case XFA.ALWAYSEMBEDTAG:
		case XFA.NEVEREMBEDTAG:
		case XFA.PARTTAG:
		case XFA.AMDTAG:
		case XFA.CONFORMANCETAG:
		case XFA.INCLUDEXDPCONTENTTAG:
		case XFA.OVERPRINTTAG:			
		case XFA.CONFIGURATIONVALUETAG:
			newNode = new com.adobe.xfa.configuration.ConfigurationValue(parent, prevSibling);
			break;
		
		case XFA.URITAG:
			newNode = new com.adobe.xfa.configuration.ConfigurationUri(parent, prevSibling);
			break;
			
		default:
			return null;
		}
		
		newNode.setClass(XFA.getAtom(eTag), eTag);
		return newNode;
	}

	private void addKey(int eTag) {
		// config value and config key are valid children of all keys
		putAttribute(eTag, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(eTag, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
	}

	private void addValue(int eTag) {
		putAttribute(eTag, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(eTag, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(eTag, XFA.TEXTNODETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(eTag);
	}

	private void addUri(int eTag) {
		putAttribute(eTag, EnumValue.getEnum(XFA.LOCKTAG, EnumAttr.getEnum(EnumAttr.BOOL_FALSE)), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putAttribute(eTag, XFA.DESCTAG, null, Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		putElement(eTag, XFA.TEXTNODETAG, ChildReln.getZeroOrOne(), Schema.XFAVERSION_10, Schema.XFAAVAILABILITY_CORE, /* 0, */ 0);
		setLeaf(eTag);
	}

	private void putValue(int eParent,
						  int eTag,
						  ChildReln oRelation,
						  Attribute oDefaultValue, /* = null */
						  int nVersionIntro, /* = 0 */
						  int nAvailability, /* = Schema.XFAAVAILABILITY_ALL */
						  int nVersionDep /* = 0 */) {
		ConfigValueChildReln oNewRelation = null;
		if (oDefaultValue == null)
			oNewRelation = new ConfigValueChildReln(oRelation, new StringAttr(XFA.getString(eTag),""));
		else
			oNewRelation = new ConfigValueChildReln(oRelation, oDefaultValue);
		putElement(eParent, eTag, oNewRelation, nVersionIntro, nAvailability, /* 0, */ nVersionDep);
	}

	/**
	 * Print out the default configuration schema.
	 *
	 * @param sSchemaName the schema to print. 
	 * @param sUri the file name.
	 * @param bWriteToStdout do we write to stdout when sUri is null.
	 * @return returns the configuration schema.
	 */
	public String printConfigSchema(String sSchemaName,
							   String sUri /* = null */,
							   boolean bWriteToStdout /* = true */) {
		StringBuilder sConfigurationSchema = new StringBuilder();
		boolean bPrint = true;

		int nPos = XFA.getTag(sSchemaName.intern());
		//
		// don't print out configvalues nor configkeys.
		//
		if (nPos >= 0) {
			if (nPos == XFA.CONFIGURATIONVALUETAG)
				bPrint = false;
			else if (nPos == XFA.CONFIGURATIONKEYTAG)
				bPrint = false;
		}
		else {
			nPos = XFA.AGENTTAG;
		}

		if (bPrint) {
			if (! mbInitPrintSchema)
				initPrintSchema();
			sConfigurationSchema.ensureCapacity(5 * 1024);
			//
			// <?xml ...>
			//
			sConfigurationSchema.append(gsXMLPrefix);
			sConfigurationSchema.append(gsNewLine);
			//
			// <xdp:xdp xmlns:xdp="http://ns.adobe.com/xdp/">
			//
			sConfigurationSchema.append('<');
			sConfigurationSchema.append(STRS.XDPNODE);
			sConfigurationSchema.append(' ');
			sConfigurationSchema.append(XFA.XMLNS);
			sConfigurationSchema.append(':');
			sConfigurationSchema.append(XFA.XDP);
			sConfigurationSchema.append('=');
			sConfigurationSchema.append('"');
			sConfigurationSchema.append("http://ns.adobe.com/xdp/");
			sConfigurationSchema.append('"');
			sConfigurationSchema.append('>');
			sConfigurationSchema.append(gsNewLine);
			//
			// [indent]<config xmlns="http://www.xfa.org/schema/xci/1.0/" xmlns:xfa="http://www.xfa.org/schema/xci/1.0">
			//
			sConfigurationSchema.append(STRS.INDENT);
			sConfigurationSchema.append('<');
			sConfigurationSchema.append(XFA.CONFIG);
			sConfigurationSchema.append(' ');
			sConfigurationSchema.append(XFA.XMLNS);
			sConfigurationSchema.append('=');
			sConfigurationSchema.append('"');
			sConfigurationSchema.append(ConfigurationModel.configurationNS());
			sConfigurationSchema.append('"');
			sConfigurationSchema.append(' ');
			sConfigurationSchema.append(XFA.XMLNS);
			sConfigurationSchema.append(':');
			sConfigurationSchema.append(XFA.XFA);
			sConfigurationSchema.append('=');
			sConfigurationSchema.append('"');
			sConfigurationSchema.append(ConfigurationModel.configurationNS());
			sConfigurationSchema.append('"');
			sConfigurationSchema.append('>');
			sConfigurationSchema.append(gsNewLine);
			//
			// Insert <"sSchema"> hierarchy.
			//
			if (nPos <= XFA.XFA_ELEMENTS) {
				String sName = null;
				if (nPos == XFA.AGENTTAG)
					sName = sSchemaName;
				printSchema(sConfigurationSchema, XFA.CONFIGTAG, nPos,
									ChildReln.zeroOrMore, sName, STRS.INDENT);
			}
			//
			// [indent]</config>
			//
			sConfigurationSchema.append(STRS.INDENT);
			sConfigurationSchema.append('<');
			sConfigurationSchema.append('/');
			sConfigurationSchema.append(XFA.CONFIG);
			sConfigurationSchema.append('>');
			sConfigurationSchema.append(gsNewLine);
			//
			// </xfa>
			//
			sConfigurationSchema.append('<');
			sConfigurationSchema.append('/');
			sConfigurationSchema.append(STRS.XDPNODE);
			sConfigurationSchema.append('>');
			sConfigurationSchema.append(gsNewLine);
		}
		String sConfiguration = sConfigurationSchema.toString();
		//
		// Send output to xci file.
		//
		if (!StringUtils.isEmpty(sUri)) {
			try {
				String outputStreamFile = sUri;
				// This indexOf(...) call used to be "endsWith(...)
				// Changed for the Java port so that in our multi-threaded
				// testing we could create files like "x.xci1" without having
				// them changed to "x.xci1.xci"
				if (outputStreamFile.indexOf("." + XFA.XCI) == -1)
					outputStreamFile = sUri + "." + XFA.XCI;
				Writer oStream = null;
				try {
					oStream = new OutputStreamWriter(new FileOutputStream(outputStreamFile), "UTF-8");
					oStream.write(sConfiguration);
					oStream.write(gsNewLine);
				}
				finally {
					if (oStream != null) {
						try {
							oStream.close();
						}
						catch (IOException ex) {
						}
					}
				}
				
			}
			catch (Exception e) {
				throw new ExFull(e);
			}
		}
		//
		// Print to stdout.
		//
		else if (bWriteToStdout) {
			System.out.println(sConfiguration);
		}
		return sConfiguration;
	}


	/*
	 * Add default attributes and comments for use with printSchema.
	 */
	private void initPrintSchema() {
		moPrintAttributes = new HashMap<String, String>(4);
		moPrintComments = new HashMap<String, String>(16);
		//
		// Attributes.
		//
		MsgFormat oPrintAttrPa = new MsgFormat(ResId.XFAConfigPrintAttrPa);
		moPrintAttributes.put("pa", oPrintAttrPa.toString());
		MsgFormat oPrintAttrTp = new MsgFormat(ResId.XFAConfigPrintAttrTp);
		moPrintAttributes.put("tp", oPrintAttrTp.toString());
		//
		// Comments.
		//
		MsgFormat oPrintCmntCommonUri
				= new MsgFormat(ResId.XFAConfigPrintCmmntCommonUri);
		moPrintComments.put("common.uri", oPrintCmntCommonUri.toString());
		MsgFormat oPrintCmntPaDestination
				= new MsgFormat(ResId.XFAConfigPrintCmmntPaDestination);
		moPrintComments.put("present.destination",
										oPrintCmntPaDestination.toString());
		MsgFormat oConfigPrintCmmntcSpace
				= new MsgFormat(ResId.XFAConfigPrintCmmntcSpace);
		moPrintComments.put("image.cSpace",
										oConfigPrintCmmntcSpace.toString());
		MsgFormat oPrintCmntCompressionTypeImage
				= new MsgFormat(ResId.XFAConfigPrintCmmntCompressionTypeTiff);
		moPrintComments.put("image.compression.type",
									oPrintCmntCompressionTypeImage.toString());
		MsgFormat oPrintCmntCompressionLevel
				= new MsgFormat(ResId.XFAConfigPrintCmmntCompressionLevel);  
		moPrintComments.put("pdf.compression.level",
									oPrintCmntCompressionLevel.toString());
		MsgFormat oPrintCmntPdfVersion
				= new MsgFormat(ResId.XFAConfigPrintCmmntPdfVersion);
		moPrintComments.put("pdf.version", oPrintCmntPdfVersion.toString());
		//
		// Comment for font.subsetBelow needs to be done for all devices.
		//
		MsgFormat oPrintCmntSubsetBelow
				= new MsgFormat(ResId.XFAConfigPrintCmmntSubsetBelow);  
		moPrintComments.put("pdf.fontInfo.subsetBelow",
										oPrintCmntSubsetBelow.toString());
		moPrintComments.put("pcl.fontInfo.subsetBelow",
										oPrintCmntSubsetBelow.toString());
		moPrintComments.put("image.fontInfo.subsetBelow",
										oPrintCmntSubsetBelow.toString());
		moPrintComments.put("ps.fontInfo.subsetBelow",
										oPrintCmntSubsetBelow.toString());
		moPrintComments.put("zpl.fontInfo.subsetBelow",
										oPrintCmntSubsetBelow.toString());
		MsgFormat oPrintCmntFontMapEquate
				= new MsgFormat(ResId.XFAConfigPrintCmmntFontMapEquate);
		moPrintComments.put("fontInfo.map.equate",
										oPrintCmntFontMapEquate.toString());
		MsgFormat oPrintCmntTraceArea
				= new MsgFormat(ResId.XFAConfigPrintCmmntTraceArea);
		moPrintComments.put("trace.area", oPrintCmntTraceArea.toString());
		mbInitPrintSchema = true;
	}


	private String getComment(String sSomeExpr) {
		String sAttr = moPrintComments.get(sSomeExpr);
		int nFoundAt = 0;
		//
		// recusivly remove the begining of the som expr until
		// we find a match in the table.
		//
		while (StringUtils.isEmpty(sAttr)) {
			nFoundAt = sSomeExpr.indexOf('.', nFoundAt);
			if (nFoundAt == -1)
				break;
			nFoundAt++;
			String sNewString = sSomeExpr.substring(nFoundAt);
			sAttr = moPrintComments.get(sNewString);
		}
		return sAttr;
	}


	private void printSchema(StringBuilder sBuffer, int eParentTag, int eTag,
								int nOccurs, String sContext, String sIndent) {
		String sNextIndent = sIndent + STRS.INDENT;
		String sTag = XFA.getString(eTag);
		//
		// common for key and value.
		//
		sBuffer.append(sNextIndent);
		sBuffer.append('<');
		sBuffer.append(sTag);
		//
		// add name attribute on agent key.
		//
		if (eTag == XFA.AGENTTAG) {
			sBuffer.append(' ');
			sBuffer.append(XFA.NAME);
			sBuffer.append('=');
			sBuffer.append('"');
			sBuffer.append(sContext);
			sBuffer.append('"');
		}
		//
		// add attrs.
		//
		SchemaPairs oAttrs = super.getNodeSchema(eTag).getValidAttributes();
		if (oAttrs != null) {
			for (int i = 0; i < oAttrs.size(); i++) {
				int eAttrTag = oAttrs.key(i);
				if (eAttrTag == XFA.LOCKTAG || eAttrTag == XFA.DESCTAG)
					continue;
				if (eTag == XFA.AGENTTAG && eAttrTag == XFA.NAMETAG)
					continue;
				Attribute oAttr = (Attribute) oAttrs.value(i);
				sBuffer.append(' ');
				sBuffer.append(XFA.getString(eAttrTag));
				sBuffer.append('=');
				sBuffer.append('"');
				sBuffer.append(oAttr.toString());
				sBuffer.append('"');
			}
		}
		//
		// make the key for retriving messages from resources.
		//
		StringBuilder sNewContext = new StringBuilder();
		if (sContext != null) {
			sNewContext.append(sContext);
		}
		if (sNewContext.length() != 0) {
			sNewContext.append('.');
		}
		sNewContext.append(sTag);
		//	
		// get the attribute strings from resources.
		//	
		String sAttr = moPrintAttributes.get(sNewContext.toString());
		if (!StringUtils.isEmpty(sAttr)) {
			sBuffer.append(' ');
			sBuffer.append(sAttr);
		}
		//
		// get any comments from resources.
		//
		StringBuilder sCommentText = new StringBuilder();
		String sComments = getComment(sNewContext.toString());
		if (sComments != null)
			sCommentText.append(sComments);
		if (nOccurs != ChildReln.zeroOrOne) {	
			sCommentText.append(' ');
			if (nOccurs == ChildReln.zeroOrMore)
				sCommentText.append("[0..n]");
			else if (nOccurs == ChildReln.oneOrMore)
				sCommentText.append("[1..n]");
			else if (nOccurs == ChildReln.oneOfChild)
				sCommentText.append("oneOfChild");
			sCommentText.append(' ');
		}

		//
		// at the leaves of the schema do ...
		//
		if (isLeaf(eTag)) {
			Attribute oDefaultValue = defaultAttribute(eTag, eParentTag);
			String sDefaultValue = "";
			if (oDefaultValue != null)
				sDefaultValue = oDefaultValue.toString();
			if (StringUtils.isEmpty(sDefaultValue)) {
				sBuffer.append(STRS.EMPTYELEMENT);
			}
			else {
				sBuffer.append(STRS.ENDTAG);
				sBuffer.append(sDefaultValue);
				sBuffer.append(STRS.CLOSING);
				sBuffer.append(sTag);
				sBuffer.append(STRS.ENDTAG);
			}
			//
			// print out enumerated values.
			//
			if (oDefaultValue instanceof EnumValue) {
				EnumAttr oEnum = ((EnumValue) oDefaultValue).getAttr();
				String oValues[] = oEnum.getStrings();
				if (oValues.length > 0) {
					sBuffer.append(' ');
					sBuffer.append(STRS.COMMENTSTART);
					sBuffer.append(' ');
					for (int i = 0; i < oValues.length; i++) {
						if (i != 0)
							sBuffer.append('|');
						sBuffer.append(oValues[i]);
					}
					sBuffer.append(' ');
					sBuffer.append(STRS.COMMENTEND);
				}
			}
			//
			// print out comment for config value.
			//
			if (sCommentText.length() != 0) {
				sBuffer.append(' ');
				sBuffer.append(STRS.COMMENTSTART);
				sBuffer.append(' ');
				sBuffer.append(sCommentText.toString());
				sBuffer.append(' ');
				sBuffer.append(STRS.COMMENTEND);
			}
			sBuffer.append(gsNewLine);
			return;
		}
		//
		// process config key.
		//
		sBuffer.append('>');
		//
		// print out comment for config key.
		//
		if (sCommentText.length() != 0) {
			sBuffer.append(' ');
			sBuffer.append(STRS.COMMENTSTART);
			sBuffer.append(' ');
			sBuffer.append(sCommentText);
			sBuffer.append(' ');
			sBuffer.append(STRS.COMMENTEND);
		}
		sBuffer.append(gsNewLine);
		SchemaPairs oChildren = super.getNodeSchema(eTag).getValidChildren();
		if (oChildren != null) {
			//
			// process only some of the children.
			//
			for (int i = 0; i < oChildren.size(); i++) {
				int eChild = oChildren.key(i);
				if (eChild != XFA.TEXTNODETAG
				&& eChild != XFA.CONFIGURATIONVALUETAG
				&& eChild != XFA.CONFIGURATIONKEYTAG) {
					ChildReln oReln = (ChildReln) oChildren.value(i);
					printSchema(sBuffer, eTag, eChild, oReln.getOccurrence(),
									sNewContext.toString(), sNextIndent);
				}
			}
		}
		//
		// close config key.
		//
		sBuffer.append(sNextIndent);
		sBuffer.append(STRS.CLOSING);
		sBuffer.append(sTag);
		sBuffer.append('>');
		sBuffer.append(gsNewLine);
	}
	
	private void setLeaf(int eTag) {
		mbLeafTags[eTag - mnElementMin] = true;
	}
	
	private boolean isLeaf(int eTag) {
		return mbLeafTags[eTag - mnElementMin];
	}
}
