package com.adobe.xfa.font;

import java.util.HashMap;
import java.util.Map;

import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.LineMetrics;
import com.adobe.fontengine.font.PDFFontDescription;
import com.adobe.xfa.ut.UnitSpan;

/**
 * @exclude from published api.
 */

public class FontItem extends FontInfo {
	
	static class InstanceKey {
		
		final UnitSpan mSize;
		final float mHorizontalScale;
		final float mVerticalScale;

		InstanceKey (UnitSpan size) {
			mSize = size;
			mHorizontalScale = 1;
			mVerticalScale = 1;
		}

		InstanceKey (UnitSpan size, double horizontalScale, double verticalScale) {
			mSize = size;
			mHorizontalScale = (float) horizontalScale;
			mVerticalScale = (float) verticalScale;
		}

		public boolean equals (Object object) {
			
			if (this == object)
				return true;
			
			// This overrides Object.equals(boolean) directly, so...
			if (object == null)
				return false;
			
			if (object.getClass() != getClass())
				return false;
			
			InstanceKey compare = (InstanceKey) object;
			
			if (! UnitSpan.match (mSize, compare.mSize))
				return false;
			
			return (mHorizontalScale == compare.mHorizontalScale) && (mVerticalScale == compare.mVerticalScale);
		}

		public int hashCode () {
			int result = mSize.hashCode();
			int f = Float.floatToIntBits (mHorizontalScale);
			result = (result * 31) ^ f;
			f = Float.floatToIntBits (mVerticalScale);
			result = (result * 31) ^ f;
			return result;
		}
	}

	private final Font mAFEFont;
	private PDFFontDescription mAFEPDFDesc;
	private double mAscent;
	private double mLegacyAscent;
	private double mDescent;
	private double mLineGap;
	private double mLegacyLineGap;
	private double mSpacing;
	private final Map<Integer, Integer> mGlyphMap = new HashMap<Integer, Integer>();
	private final Map<InstanceKey, FontInstance> mInstances = new HashMap<InstanceKey, FontInstance>();

	FontItem (FontInfo info, Font afeFont) {
		super (info);
		mAFEFont = afeFont;
	}

	public static double ctLegacyScale (double emScale, double advance) {
		assert (emScale != 0);
		double onePtRel = advance / emScale;
		long truncated = (long) (onePtRel * 65536);
		double result = truncated / 65536.0;
		return result;
	}

	public Font getAFEFont () {
		return mAFEFont;
	}

	public double getAscent () {
		return mAscent;
	}

	public double getLegacyAscent () {
		return mLegacyAscent;
	}

	public double getDescent () {
		return mDescent;
	}

	public double getLineGap () {
		return mLineGap;
	}

	public double getLegacyLineGap () {
		return mLegacyLineGap;
	}

	public double getSpacing () {
		return mSpacing;
	}

	public double getCharWidth (int c, boolean useHorizontalGlyphs) {
		int glyphID = getGlyphID (c);
		if (glyphID == 0) {
			return -1;
		}
		return getGlyphWidth (glyphID, useHorizontalGlyphs);
	}

	public int getNotDefGlyphID () {
		return 0;																// TODO:
	}

	public int getGlyphID (int c) {
		Integer value = mGlyphMap.get(c);
		
		// returns 0 if not found
		if (value == null)
			return 0;
		
		return value.intValue(); 
	}

	public double getGlyphWidth (int glyphID, boolean useHorizontalGlyphs) {	// TODO: do something with useHorizontalGlyphs
		double width = -1;
		PDFFontDescription pdfDesc = forcePDFDesc();
		if (pdfDesc != null) {
			try {
				width = mAFEPDFDesc.getAdvance (glyphID);
			} catch (Exception e) {												// TODO: what can we do if this throws?
			}
		}
		return ctLegacyScale (1000, width);										// TODO: why 1000, and not mAFEFont.getUnitsPerEmX()?
	}

	public boolean validateChar (int c, boolean useHorizontalGlyphs) {
		return true;				// TODO:
	}

	public FontInstance reconcile (UnitSpan size) {
		return reconcile (size, 1, 1);
	}

	public FontInstance reconcile (UnitSpan size, double horizontalScale, double verticalScale) {
		InstanceKey key = new InstanceKey (size, horizontalScale, verticalScale);
		FontInstance instance = lookup (key);
		if (instance == null) {
			instance = new FontInstance (this, size);
			mInstances.put (key, instance);
		}
		return instance;
	}

	public boolean equals (Object object) { // NOPMD - UselessOverridingMethod - here for documentation/FindBugs suppression
		
		// The fields added in this derived class do not participate in equality
		return super.equals(object);
	}

	public int hashCode() {
		int hash = 29;
		hash = hash * 31 ^ super.hashCode();
		return hash;
	}

	public static boolean match (FontItem o1, FontItem o2) {
		if (o1 == o2) {
			return true;
		}
		if ((o1 == null) || (o2 == null)) {
			return false;
		}
		return o1.equals (o2);
	}

	public void addSubsettedGlyphs (int glyph, int chr) {
//		TODO:
	}

	public FontItem getUnicodeFont () {
		return this;		// TODO:
	}

	public boolean isPWIDApplied () {
		return false;
	}

	public void mapGlyph (int c, int glyphID) {
		mGlyphMap.put(c, glyphID);
	}

	boolean loadMetrics () {
		try {		// TODO: apply all the machinations of C++ font service
			LineMetrics lineMetrics = mAFEFont.getCoolTypeLineMetrics();
			double scale = mAFEFont.getUnitsPerEmY();
			mAscent = lineMetrics.ascender / scale;
			mLegacyAscent = mAscent;
			mDescent = -lineMetrics.descender / scale;
			if ((mAscent + mDescent) < 1.0) {
				mAscent = 1.0 - mDescent;
			}
			mLineGap = 0.2;
			mLegacyLineGap = lineMetrics.linegap / scale;
			mSpacing = 1.2;
		} catch (Exception e) {
			return false;
		}
		return true;
	}

	private FontInstance lookup (InstanceKey key) {
		return mInstances.get (key);
	}

	private PDFFontDescription forcePDFDesc () {
		if (mAFEPDFDesc == null) {
			try {
				mAFEPDFDesc = mAFEFont.getPDFFontDescription();
			} catch (Exception e) {						// TODO: what can we do if this throws?
			}
		}
		return mAFEPDFDesc;
	}
}
