/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.protocol;

import java.net.Authenticator;
import java.net.PasswordAuthentication;

import com.adobe.xfa.ut.CharacterHolder;
import com.adobe.xfa.ut.IntegerHolder;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ObjectHolder;
import com.adobe.xfa.ut.StringHolder;


/**
 * An abstract class from which to derive protocol authenticators.
 */
public abstract class AuthenticationHandler extends Authenticator {

	/**
	 * @exclude from published api.
	 */
	public enum AuthenticationType {
		NONE,
		BASIC,
		DIGEST,
		CERTIFICATE
	}
	
	/**
	 * @exclude from published api.
	 */
	public enum ConnectStatus {
		NOTSET(0), 
		CONNECTED(200),
		CONNECTIONERROR(300), 
		UNAUTHORIZED(401),
		FORBIDDEN(403),
		FILENOTFOUND(404),	// Represent HTTP 404 - not found 
			 				// and HTTP 410 - server know the file is gone
		CLIENTAUTHCERTNEEDED(12044),
		INVALIDCA(12045);

		private ConnectStatus(int c) {
			code = c;
		}

//		public int getCode() {
//			return code;
//		}

		@SuppressWarnings("unused")
        private int code;
	}

	/**
	 * @exclude from published api.
	 */
	public enum HandlerType {
		CLIENT, 
		SERVER 
	}

	/**
	 * @exclude from published api.
	 */
	public enum SOAPRequestStatus {
		SOAPNOTSET,
		SOAPSUCCESS,
		SOAPFAULT
	}
	
	/** 
	 * am I a server handler or a client handler
	 *
	 * @exclude from published api.
	 */
	protected HandlerType meHandlerType = HandlerType.CLIENT; 

	/**
	 *  the user id, used for both transport and SOAP
	 *
	 * @exclude from published api.
	 */
	private String msUserID = null;
	
	/**
	 *  the password, used for both transport and SOAP
	 */
	private CharacterHolder maPassword = null;
	
	/**
	 *  a hint used by some applications to track their certificate
	 */
	private String msHint = null;

	/**
	 *  the transport layer authentication type
	 */
	private AuthenticationType meTransportAuthType = AuthenticationType.NONE;

	/**
	 *  the URL we are dealing with
	 */
	private String msURL = null;

	/**
	 *  the Host we are dealing with (server name)
	 */
	private String msHost = null;

	/**
	 *  type of transport authentication, to return to client
	 */
	private String msType = null;

	/**
	 *  the realm information, to return to client
	 */
	private String msRealm = null;

	/**
	 *  the status of the current connection
	 */
	private ConnectStatus	meTransportStatus = ConnectStatus.NOTSET;

	/**
	 *  the SOAP layer authentication type
	 */
	private AuthenticationType	meSOAPAuthType = AuthenticationType.NONE;

	/**
	 *  the http:// SOAP address
	 */
	private String msSOAPAddress = null; 

	/**
	 *  the name of the operation we are invoking in the SOAP message
	 */
	private String msSOAPOperation = null;

	/**
	 *  the name of our wsdl connection (wsdlConnection name="DataConnection")
	 */
	private String msConnectionName = null;

	/**
	 *  the status of the current soap request
	 */
	private SOAPRequestStatus meSOAPStatus = SOAPRequestStatus.SOAPNOTSET;

	/**
	 *  the <faultstring> returned in the SOAP message
	 */
	private String msSOAPFault = null;

	/**
	 *  the <faultcode> string returned in the SOAP message
	 */
	private String msSOAPFaultCode = null;

	/**
	 * Constructor
	 * @param eType The handler type; CLIENT or SERVER
	 *
	 * @exclude from published api.
	 */
	public AuthenticationHandler(HandlerType eType) {	
		meHandlerType = eType;
	}

	/**
	 * Called by Protocol to return the status of the authentication call.
	 * NOTE: You MUST return false in order to stop the authentication loop. 
	 * If true is returned, Protocol assumes you want to loop again (ask the 
	 * handler for credentials and try the http request again).
	 * @return True to indicate that we should try to connect with the 
	 *         credentials, or False to cancel
	 *
	 * @exclude from published api.
	 */
	public abstract boolean checkAuthenticationStatus();
		
	/**
	 * Called by xfawsdlconnectionsetproxy to return the status of the 
	 * authentication call, and if successful
	 * @return True to indicate that we should try to connect with the 
	 *         credentials, or False to cancel
	 *
	 * @exclude from published api.
	 */
	public abstract boolean checkSOAPAuthenticationStatus();

	/**
	 * Clone this AuthenticationHandler object.
	 * @return a copy of this object.
	 *
	 * @exclude from published api.
	 */
	public abstract AuthenticationHandler clone();

	/**
	 * Called by Protocol to return the user id and password for the URL
	 * @param sURL 	    passed in by the caller(Protocol) - the url we are 
	 *                  connecting to
	 * @param sRealm    passed in by the caller(Protocol) - the realm 
	 *                  information from the protocol layer
	 * @param sUserID   to be filled by the method - the user id to use for the 
	 *                  connection
	 * @param aPassword to be filled by the method - the password to use for  
	 *                  the connection
	 * @return True to indicate that we should try to connect with the credentials, or
	 *		   False to cancel
	 *
	 * @exclude from published api.
	 */
	public abstract boolean getBasicCredentials(String sURL, String sRealm, 
												StringHolder sUserID, 
												CharacterHolder aPassword);

	/**
	 * Called by xfawsdlconnectionsetproxy to return the user id and password 
	 * for the SOAP header
	 * @param sSOAPAddress    passed in by the caller - the SOAP address we are
	 *                        connecting to
	 * @param sConnectionName passed in by the caller - the wsdlConnection name
	 * @param sOperationName  passed in by the caller - the name of the 
	 *                        operation being called
	 * @param sUserID         to be filled by the method - the user id to use 
	 *                        for the connection
	 * @param aPassword       to be filled by the method - the password to use 
	 *                        for the connection
	 * @return True to indicate that we should try to connect with the 
	 *         credentials, or False to cancel
	 *
	 * @exclude from published api.
	 */
	public abstract boolean getBasicSOAPCredentials(String sSOAPAddress, 
													String sConnectionName, 
													String sOperationName, 
													StringHolder sUserID, 
													CharacterHolder aPassword);

	/**
	 * Called Protocol to return the certificate for the URL
	 * @param sUrl	        passed in by the caller(Protocol), the url we are 
	 *                      connecting to
	 * @param aCertPassword the password to encode the certificate with. This 
	 *                      parameter will only be populated if the pCertBuffer
	 *                      contains PKCS#12 data. 
	 * @param nCertSize     the size of the certificate buffer. This is only 
	 *                      populate if the pCertBuffer contains PKCS#12 data.
	 * @param oCertBuffer   depending on the application, this parameter may be
	 *                      populated with a buffer containing encrypted (using
	 *                      the aCertPassword specified parameter 2 above) 
	 *                      PKCS#12 data, or PCCERT_CONTEXT (Windows specific)
	 *						data, as some Windows applications may only have 
	 *                      access to the Windows PCCERT_CONTEXT and not to a 
	 *                      PKCS#12 stream. It this parameter is populated with
	 *						a PCCERT_CONTEXT then the nCertSize parameter will 
	 *                      be set to zero and the aCertPassword will be empty.
	 * @param sHint		    this parameter is populated by the caller and used
	 *                      as a hint so that the calling application knows 
	 *                      certificate they are working with when callback's
	 *                      are made.
	 *                         
	 * @return True to indicate that we should try to connect with the credentials, or
	 *		   False to cancel
	 *
	 * @exclude from published api.
	 */
	public abstract boolean getCertificateCredentials(String sUrl, 
													  CharacterHolder aCertPassword, 
													  IntegerHolder nCertSize,
											 		  ObjectHolder<byte[]> oCertBuffer, 
											 		  StringHolder sHint);
	
	/**
	 * This method is called by the underlying protocol code to give the 
	 * application a chance to deal with a server that states that Certificate
	 * Authority is unknown. The application has the opportunity to say whether
	 * it is ok to proceed or not. 
	 * @param sUrl passed in by the caller(Protocol), the url we are connecting 
	 * 		  to
	 * @return True to indicate that we should relax the Certificate Authority 
	 *         checking and use this certificate, otherwise, False to fail the
	 *          connection.
	 *
	 * @exclude from published api.
	 */
	public abstract boolean isCertificateAuthorityValid(String sUrl);

    /**
	 * Gets the name of our wsdl connection (wsdlConnection 
	 * name="DataConnection")
	 * @return the current wsdl connection name
	 *
	 * @exclude from published api.
	 */
	public String getConnectionName() {
		return msConnectionName;
	}

	/**
	 * Gets the status of the current connection
	 * @return the current status as an eConnectStatus enumerated value
	 *
	 * @exclude from published api.
	 */
	public ConnectStatus getConnectStatus() {
		return meTransportStatus;
	}
		
	/**
	 * Gets the current handler type; client or server
	 * @return the current handler type as an eHandlerType enumerated value
	 *
	 * @exclude from published api.
	 */
	public HandlerType getHandlerType() {
		return meHandlerType;
	}

	/**
	 * Gets the hint used by some applications to track their certificate
	 * @return the current hint
	 *
	 * @exclude from published api.
	 */
	public String getHint() {
		return msHint;
	}

	/**
	 * Gets the Host we are dealing with (server name)
	 * @return the current host
	 *
	 * @exclude from published api.
	 */
	public String getHost() {
		return msHost;
	}

	/**
	 * Gets the password, used for both transport and SOAP
	 * @return the current password
	 *
	 * @exclude from published api.
	 */
	public CharacterHolder getPassword() {
		return maPassword;
	}
	
	/**
	 * Gets the realm information
	 * @return the current realm
	 *
	 * @exclude from published api.
	 */
	public String getRealm() {
		return msRealm;
	}

	/**
	 * Gets the the http:// SOAP address.
	 * @return the SOAP address
	 *
	 * @exclude from published api.
	 */
	public String getSOAPAddress() {
		return msSOAPAddress;
	}
	
	/**
	 * Gets the SOAP layer authentication type.
	 * @return the SOAP authentication type as an eAuthenticationType 
	 *         enumerated value
	 *
	 * @exclude from published api.
	 */
	public AuthenticationType getSOAPAuthType() {
		return meSOAPAuthType;
	}

	/**
	 * Gets the <faultcode> string returned in the SOAP message
	 * @return the SOAP fault code
	 *
	 * @exclude from published api.
	 */
	public String getSOAPFaultCode() {
		return msSOAPFaultCode;
	}

	/**
	 * Gets the <faultstring> returned in the SOAP message
	 * @return the SOAP fault string
	 *
	 * @exclude from published api.
	 */
	public String getSOAPFaultString() {
		return msSOAPFault;
	}
	
	/**
	 * Gets the name of the operation we are invoking in the SOAP message
	 * @return the SOAP operation
	 *
	 * @exclude from published api.
	 */
	public String getSOAPOperation() {
		return msSOAPOperation;
	}

	/**
	 * Gets the status of the current soap request
	 * @return the SOAP request status as a eSOAPRequestStatus enumerated value
	 *
	 * @exclude from published api.
	 */
	public SOAPRequestStatus getSOAPRequestStatus() {
		return meSOAPStatus;
	}

	/**
	 * Gets the URL we are dealing with
	 * @return the current URL
	 *
	 * @exclude from published api.
	 */
	public String getURL() {
		return msURL;
	}
	
	/**
	 * Gets the transport layer authentication type
	 * @return an eAuthenticationType enumerated value indicating the type
	 *
	 * @exclude from published api.
	 */
	public AuthenticationType getTransportAuthType() {
		return meTransportAuthType;
	}

	/**
	 * Gets the type of transport authentication (basic, digest)
	 * @return the current type
	 *
	 * @exclude from published api.
	 */
	public String getType() {
		return msType;
	}
	
	/**
	 * Gets the user id, used for both transport and SOAP
	 * @return the current user id
	 *
	 * @exclude from published api.
	 */
	public String getUserID() {
		return msUserID;
	}
	
	/**
	 * This method is called by the underlying Protocol code to give the 
	 * application a chance to deal with a log message.
	 * @param oError Error message to log
	 *
	 * @exclude from published api.
	 */
	public void logMessage(MsgFormatPos oError) {
	}

	/**
	 *	reset the member variables
	 *
	 * @exclude from published api.
	 */
	public void reset() {
		// don't reset meHandlerType

		msUserID = "";
		maPassword = new CharacterHolder();
		msHint = "";

		meTransportAuthType = AuthenticationType.NONE;

		msURL = "";
		msHost = "";
		msType = "";
		msRealm = "";

		meTransportStatus = ConnectStatus.NOTSET;

		meSOAPAuthType = AuthenticationType.NONE;

		msSOAPAddress = "";
		msSOAPOperation = "";
		msConnectionName = "";

		meSOAPStatus = SOAPRequestStatus.SOAPNOTSET;
		msSOAPFault = "";
		msSOAPFaultCode = "";
	}

	/**
	 * Sets the name of our wsdl connection (wsdlConnection 
	 * name="DataConnection")
	 * @param sConnectionName The wsdl connection name
	 *
	 * @exclude from published api.
	 */
	public void	setConnectionName(String sConnectionName) {
		msConnectionName = sConnectionName;
	}

	/**
	 * Sets the status of the current connection.
	 * @param eStatus the status as an eConnectStatus enumeration value
	 *
	 * @exclude from published api.
	 */
	public void	setConnectStatus(ConnectStatus eStatus) {
		meTransportStatus = eStatus;
	}
		
	/**
	 * Sets the current Handler type (client or server)
	 * @param eHandlerType an eHandlerType enumeration value
	 *
	 * @exclude from published api.
	 */
	public void	setHandlerType(HandlerType eHandlerType) {
		meHandlerType = eHandlerType;
	}

	/**
	 * Sets the a hint used by some applications to track their certificate. 
	 * Client should set this.
	 * @param sHint the hint value
	 *
	 * @exclude from published api.
	 */
	public void	setHint(String sHint) {
		msHint = sHint;
	}
	
	/**
	 * Sets the Host we are dealing with (server name). 
	 * Protocol code should set this.
	 * @param sHost the host
	 *
	 * @exclude from published api.
	 */
	public void	setHost(String sHost) {
		msHost = sHost;
	}
	
	/**
	 * Sets the password, used for both transport and SOAP. 
	 * Client should set this.
	 * @param aPassword the password
	 *
	 * @exclude from published api.
	 */
	public void	setPassword(CharacterHolder aPassword) {
		maPassword = aPassword;
	}

	/**
	 * Sets the the realm information, to return to client.
	 * Protocol/SOAP code should set this.
	 * @param sRealm the realm
	 *
	 * @exclude from published api.
	 */
	public void	setRealm(String sRealm){
		msRealm = sRealm;
	}

	/**
	 * Sets the the http:// SOAP address.
	 * @param sSOAPAddress the SOAP address
	 *
	 * @exclude from published api.
	 */
	public void setSOAPAddress(String sSOAPAddress) {
		msSOAPAddress = sSOAPAddress;
	}

	/**
	 * Sets the SOAP layer authentication type.
	 * @param eAuthType the authentication type as an eAuthenticationType 
	 *        enumeration value
	 *
	 * @exclude from published api.
	 */
	public void	setSOAPAuthType(AuthenticationType eAuthType) {
		meSOAPAuthType = eAuthType;
	}

	/**
	 * Sets the <faultcode> string returned in the SOAP message
	 * @param sSOAPFaultCode the fault code
	 *
	 * @exclude from published api.
	 */
	public void setSOAPFaultCode(String sSOAPFaultCode) {
		msSOAPFaultCode = sSOAPFaultCode;
	}

	/**
	 * Sets the <faultstring> returned in the SOAP message
	 * @param sSOAPFault the fault string
	 *
	 * @exclude from published api.
	 */
	public void	setSOAPFaultString(String sSOAPFault) {
		msSOAPFault = sSOAPFault;
	}

	/**
	 * Sets the name of the operation we are invoking in the SOAP message
	 * @param sSOAPOperation the SOAP operation name
	 *
	 * @exclude from published api.
	 */
	public void setSOAPOperation(String sSOAPOperation) {
		msSOAPOperation = sSOAPOperation;
	}

	/**
	 * Sets the the status of the current SOAP request
	 * @param eSOAPStatus the status as an eSOAPRequestStatus enumeration
	 *                    value
	 *
	 * @exclude from published api.
	 */
	public void	setSOAPRequestStatus(SOAPRequestStatus eSOAPStatus) {
		meSOAPStatus = eSOAPStatus;
	}

	/**
	 * Sets the current transport authentication type
	 * @param eAuthType an eAuthenticationType enumerated value
	 *
	 * @exclude from published api.
	 */
	public void setTransportAuthType(AuthenticationType eAuthType) {
		meTransportAuthType = eAuthType;
	}

	/**
	 * Sets the type of transport authentication, to return to client.
	 * Protocol/soap code should set this.
	 * @param sType the type
	 *
	 * @exclude from published api.
	 */
	public void	setType(String sType) {
		msType = sType;
	}
	
	/**
	 * Sets the current URL. Protocol/soap code should set this.
	 * @param sURL the URL
	 *
	 * @exclude from published api.
	 */
	public void	setURL(String sURL) {
		msURL = sURL;
	}
	
	/**
	 * Sets the current user id. Client should set this.
	 * @param sUserID the user id
	 *
	 * @exclude from published api.
	 */
	public void setUserID(String sUserID) {
		msUserID = sUserID;
	}	

	/**
     * Called when password authorization is needed.  Subclasses should
     * override the default implementation, which returns null.
     * @return The PasswordAuthentication collected from the
     *		   user, or null if none is provided.
	 *
	 * @exclude from published api.
     */
    protected PasswordAuthentication getPasswordAuthentication() {
    	return null;
    }
}
