/*
 * ADOBE CONFIDENTIAL
 * 
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 * 
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or
 * copyright law. Dissemination of this information or reproduction of this
 * material is strictly forbidden unless prior written permission is
 * obtained from Adobe Systems Incorporated.
 */

package com.adobe.xfa.ut;

/**
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public interface Peer {

	/**
	 * This interface must be implemented by objects that use the
	 * <code>PeerList.add()</code> and <code>PeerList.remove()</code>
	 * methods.
	 * 
	 * <p>
	 * A <code>Peer</code> can add another <code>Peer</code> to its peer
	 * list via <code>addPeer()</code>. All <code>Peers</code> in the peer
	 * list will be notified of peer changes when <code>updateFromPeer()</code>
	 * is called.
	 * 
	 */
	static public final int UPDATED = 0;
	static public final int ATTR_CHANGED = 1;
	static public final int VALUE_CHANGED = 2;
	static public final int PARENT_CHANGED = 3;
	static public final int CHILD_ADDED = 4;
	static public final int CHILD_REMOVED = 5;
	static public final int DESCENDENT_ATTR_CHANGED = 6;
	static public final int DESCENDENT_VALUE_CHANGED = 7;
	static public final int DESCENDENT_ADDED = 8;
	static public final int DESCENDENT_REMOVED = 9;
	static public final int PERMS_LOCK_SET = 10;
	static public final int PERMS_LOCK_CLEARED = 11;
	static public final int PROTO_ATTR_CHANGED = 12;
	static public final int PROTO_VALUE_CHANGED = 13;
	static public final int PROTO_CHILD_ADDED = 14;
	static public final int PROTO_CHILD_REMOVED = 15;
	static public final int PROTO_DESCENDENT_ATTR_CHANGED = 16;
	static public final int PROTO_DESCENDENT_VALUE_CHANGED = 17;
	static public final int PROTO_DESCENDENT_ADDED = 18;
	static public final int PROTO_DESCENDENT_REMOVED = 19;
    
	/**
	 * Add a peer node to be notified of state changes.
	 * 
	 * @param peerNode -
	 *            The reference to the peer object to be added.
	 */
	public void addPeer(Peer peerNode);

	public void addPeeredNode(Peer peer);

	/**
	 * Removes this node from all peer relationships.
	 */
	public void clearPeers();
	
	/**
     * Makes this peer deaf to any updateFromPeer calls made by notifyPeers
     */
	public void deafen();

	/**
	 * return the requested peer
	 * 
	 * @param nPeer -
	 *            the 0-based position of the peer to retrieve.
	 * @return the peer at the requested position. When there are not more peers
	 *         to return, this will return a null object.
	 */
	public Peer getPeer(int nPeer /* =0 */);

	/**
     * Checks if this peer is deaf.
     * @return <code>true</code> if this peer does not listen to notifyPeer calls, else <code>false</code>.
     */
	public boolean isDeaf();

	/**
     * Checks if this peer is mute.
     * @return <code>true</code> if this peer does not notify its peers when notifyPeers is called, else <code>false</code>.
     */
	public boolean isMute();

	/**
     * Makes this peer no longer notify its peers when notifyPeers is called
     */
	public void mute();

	/**
     * Sends notification to the peer nodes that this object has changed.
     * @param eventType the event type as enumerated in this class
	 * @param arg1 additional event information
     * @param arg2 additional event information
	 */
	public void notifyPeers(int eventType, String arg1, Object arg2);

	/**
	 * Remove a peer node from the notification list.
	 * 
	 * @param peerNode the peer object to be removed
	 */
	public void removePeer(Peer peerNode);

	public void removePeeredNode(Peer peer);

	/**
     * Allows this peer to receive updateFromPeer calls
     */
	public void unDeafen();

	/**
     * Allows this peer to notify its peers when notifyPeers is called.
     */
	public void unMute();

	/** 
	 * This method will be called by the peer whenever its
     * state changes.
     * <pre>
     * eventType                        (String)arg1  (Object)arg2
     * 
     * UPDATED                          update type   (optional)
     * ATTR_CHANGED                     attr name     (optional)
     * CHILD_ADDED                      (optional)    added node
     * CHILD_REMOVED                    (optional)    removed node
     * VALUE_CHANGED                    new data      (optional)
     * PARENT_CHANGED                   (optional)    new parent
     * DESCENDENT_ATTR_CHANGED          attr name     changed node
     * DESCENDENT_VALUE_CHANGED         (optional)    changed node
     * DESCENDENT_ADDED                 (optional)    added node
     * DESCENDENT_REMOVED               (optional)    removed node
     * PERMS_LOCK_SET                   (optional)    (optional)
     * PERMS_LOCK_CLEARED               (optional)    (optional)
     * PROTO_ATTR_CHANGED               attr name     changed node
     * PROTO_CHILD_ADDED                (optional)    added node of the protoype
     * PROTO_CHILD_REMOVED              (optional)    removed node from the prototype
     * PROTO_VALUE_CHANGED              new data      prototype node
     * PROTO_DESCENDENT_ATTR_CHANGED    attr name     changed prototype node
     * PROTO_DESCENDENT_VALUE_CHANGED   (optional)    changed prototype node
     * PROTO_DESCENDENT_ADDED           (optional)    added node to the prototype
     * PROTO_DESCENDENT_REMOVED         (optional)    removed node from the prototype
     * </pre>
	 * @param peerNode the peer to this object.
	 * @param eventType the event type as enumerated in this class
	 * @param arg1 additional event information
     * @param arg2 additional event information

	 */
	public void updateFromPeer(Object peerNode, int eventType, String arg1, Object arg2);
	
	/**
	 * Called when this Peer is removed from <code>peer</code>.
	 * Derived classes can override this method to receive direct notifications
	 * when a peer is removed.
	 * @param peer the Peer from which this Peer has been removed from notification. 
	 */
	public void peerRemoved(Peer peer);
}
